<?php
/**
 * Dynamic Taxonomy Terms CSS File
 *
 * Dynamic taxonomy terms css file for handling terms related css
 *
 * @since    4.3.7
 * @package realhomes/common
 */

if ( ! function_exists( 'realhomes_generate_terms_dynamic_css' ) ) {
	/**
	 * Generate dynamic CSS rules for property taxonomies.
	 *
	 * This function loops through predefined taxonomies, retrieves terms and their meta values using the
	 * `realhomes_get_terms_with_meta` function, and generates CSS rules for each term. The CSS rules are based
	 * on term slugs and the meta value (e.g., colors).
	 *
	 * @return string A string of dynamically generated CSS rules.
	 */
	function realhomes_generate_terms_dynamic_css( $common_custom_css ) {
		// Define taxonomies and optional selectors
		$taxonomies = array(
			'property-city'    => array(),
			'property-status'  => array(
				'.rhea-ultra-status-box .rhea-ultra-status',
				'.rhea_title_box_4 .rhea_prop_status_sty',
				'.rhea_prop_status_sty',
				'.rh-ultra-status-box .rh-ultra-status'
			),
			'property-type'    => array(
				'.rhea_bottom_tags_box .rhea_property_types'
			),
			'property-feature' => array(),
		);

		$css_rules = '';
		$terms_with_colors = array();

		foreach ( $taxonomies as $taxonomy => $optional_selectors ) {

			$terms = get_terms( array(
				'taxonomy'   => $taxonomy,
				'hide_empty' => false, // Include all terms, even if no posts are assigned
			) );

			if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
				foreach ( $terms as $term ) {
					// Get the meta value for the term
					$meta_value = get_term_meta( $term->term_id, 'ere_property_taxonomy_term_color', true );

					// Only add terms that have the meta value
					if ( ! empty( $meta_value ) ) {
						$terms_with_colors[ $term->slug ] = $meta_value; // Use term slug as the key
					}
				}
			}

			foreach ( $terms_with_colors as $term_slug => $color ) {
				// Base CSS target (default class structure)
				$base_selector = sprintf( '.%s-%s', esc_attr( $taxonomy ), esc_attr( $term_slug ) );

				// Add base CSS rule
				$css_rules .= sprintf(
					'%s { background-color: %s; } ' . PHP_EOL,
					$base_selector,
					esc_attr( $color )
				);

				// Add optional selectors if defined
				if ( ! empty( $optional_selectors ) ) {
					foreach ( $optional_selectors as $optional_selector ) {
						$css_rules .= sprintf(
							'%s%s { background-color: %s; } ' . PHP_EOL,
							$optional_selector,
							$base_selector,
							esc_attr( $color )
						);
					}
				}
			}
		}

		$common_custom_css .= strip_tags( $css_rules );

		return $common_custom_css;
	}
}

add_filter( 'realhomes_common_custom_css', 'realhomes_generate_terms_dynamic_css' );
