/**
 * Javascript to handle google map for property single page.
 */
( function ( $ ) {
    "use strict";

    var mapContainer = document.getElementById( "property_map" );

    if ( typeof propertyMapData === "undefined" && mapContainer === null ) {
        return;
    }

    if ( ! propertyMapData.lat && ! propertyMapData.lng ) {
        return;
    }

    /**
     * Generates Google Maps link if enabled and coordinates exist
     * @param {Object} propertyData - Should contain { lat, lng, mapLinkDisplay, mapLinkText }
     * @returns {string} HTML link or empty string
     * @since 4.3.9
     */
    function generateMapLink() {
        // Check if map link is enabled and coordinates exist
        if ( propertyMapData.mapLinkDisplay !== 'true' || ! propertyMapData.lat || ! propertyMapData.lng ) {
            return '';
        }

        // Create Google Maps URL
        const url      = `https://www.google.com/maps?q=${encodeURIComponent( propertyMapData.lat )},${encodeURIComponent( propertyMapData.lng )}`,
              linkText = propertyMapData.mapLinkText;

        return `
        <p class="view-on-google-maps-button-wrapper">
            <a id="view-on-google-maps-button" class="view-on-google-maps-button" href="${url}" target="_blank" rel="noopener noreferrer">
                ${linkText.replace( /</g, '&lt;' ).replace( />/g, '&gt;' )}
                <svg xmlns="http://www.w3.org/2000/svg" width="18px" height="18px" viewBox="0 0 24 24" fill="none">
                    <path d="M10.0002 5H8.2002C7.08009 5 6.51962 5 6.0918 5.21799C5.71547 5.40973 5.40973 5.71547 5.21799 6.0918C5 6.51962 5 7.08009 5 8.2002V15.8002C5 16.9203 5 17.4801 5.21799 17.9079C5.40973 18.2842 5.71547 18.5905 6.0918 18.7822C6.5192 19 7.07899 19 8.19691 19H15.8031C16.921 19 17.48 19 17.9074 18.7822C18.2837 18.5905 18.5905 18.2839 18.7822 17.9076C19 17.4802 19 16.921 19 15.8031V14M20 9V4M20 4H15M20 4L13 11" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>
            </a>
        </p>
    `;
    }

    var iconURL  = propertyMapData.icon,
        iconSize = new google.maps.Size( 42, 57 ),
        mapZoom  = 15;

    /**
     * This code adds the zoom level for the Elementor widget "rh-single-property-map-v2".
     * The zoom level is retrieved from the "data-settings" attribute, which stores configuration value.
     * If no zoom level is specified, a default value (e.g., 15) is used.
     */
    const elementorMapContainer = document.querySelector( '.elementor-widget-rh-single-property-map-v2' );
    if ( elementorMapContainer ) {
        // Get the data-settings attribute and parse it
        const dataSettings = JSON.parse( elementorMapContainer.getAttribute( 'data-settings' ).replace( /&quot;/g, '"' ) );

        // Get the map zoom value (defaulting to 15 if missing)
        mapZoom = dataSettings.map_zoom?.size || 15;
    } else {
        // Zoom
        if ( propertyMapData.zoom > 0 ) {
            mapZoom = parseInt( propertyMapData.zoom );
        }
    }

    // Retina
    if ( window.devicePixelRatio > 1.5 ) {
        if ( propertyMapData.retinaIcon ) {
            iconURL  = propertyMapData.retinaIcon;
            iconSize = new google.maps.Size( 83, 113 );
        }
    }

    if ( propertyMapData.marker_type === 'circle' ) {
        var markerIcon = {
            path         : google.maps.SymbolPath.CIRCLE,
            scale        : 30,
            fillColor    : propertyMapData.marker_color,
            strokeColor  : propertyMapData.marker_color,
            fillOpacity  : 0.5,
            strokeWeight : 0.6
        }
    } else {
        var markerIcon = {
            url        : iconURL,
            size       : iconSize,
            scaledSize : new google.maps.Size( 42, 57 ),
            origin     : new google.maps.Point( 0, 0 ),
            anchor     : new google.maps.Point( 21, 56 )
        };
    }

    var propertyLocation   = new google.maps.LatLng( propertyMapData.lat, propertyMapData.lng ),
        propertyMapOptions = {
            center      : propertyLocation,
            zoom        : mapZoom,
            scrollwheel : false
        };

    // Map Styles
    if ( undefined !== propertyMapData.styles ) {
        propertyMapOptions.styles = JSON.parse( propertyMapData.styles );
    }

    // Setting Google Map Type
    switch ( propertyMapData.type ) {
        case 'satellite':
            propertyMapOptions.mapTypeId = google.maps.MapTypeId.SATELLITE;
            break;
        case 'hybrid':
            propertyMapOptions.mapTypeId = google.maps.MapTypeId.HYBRID;
            break;
        case 'terrain':
            propertyMapOptions.mapTypeId = google.maps.MapTypeId.TERRAIN;
            break;
        default:
            propertyMapOptions.mapTypeId = google.maps.MapTypeId.ROADMAP;
    }

    var propertyMap    = new google.maps.Map( mapContainer, propertyMapOptions ),
        propertyMarker = new google.maps.Marker( {
            position : propertyLocation,
            map      : propertyMap,
            icon     : markerIcon
        } );

    var innerHTML = "",
        boxText   = document.createElement( "div" );

    boxText.className = 'map-info-window';

    if ( rhMapsData.isUltra ) {
        if ( propertyMapData.thumb ) {
            innerHTML += '<span class="thumb-link"><img class="prop-thumb" src="' + propertyMapData.thumb + '" alt="' + propertyMapData.title + '"/></span>';
        }

        innerHTML += '<div class="rh-gm-thumb-detail">';
        innerHTML += '<h5 class="prop-title">' + propertyMapData.title + '</h5>';
        innerHTML += '<div>';

        if ( propertyMapData.propertyType ) {
            innerHTML += '<span class="type">' + propertyMapData.propertyType + '</span>';
        }

        if ( propertyMapData.price ) {
            innerHTML += '<p><span class="price">' + propertyMapData.price + '</span></p>';
        }

        innerHTML += generateMapLink();
        innerHTML += '</div>';
        innerHTML += '<div class="arrow-down"></div>';
        boxText.innerHTML = '<div class="rh-ultra-info-window">' + innerHTML + '</div>';

    } else {
        if ( propertyMapData.thumb ) {
            innerHTML += '<img class="prop-thumb" src="' + propertyMapData.thumb + '" alt="' + propertyMapData.title + '"/>';
        }

        innerHTML += '<h5 class="prop-title">' + propertyMapData.title + '</h5>';

        if ( propertyMapData.price ) {
            innerHTML += '<p><span class="price">' + propertyMapData.price + '</span></p>';
        }

        innerHTML += generateMapLink();
        innerHTML += '<div class="arrow-down"></div>';
        boxText.innerHTML = innerHTML;
    }

    // Info window close icon URL
    var closeIconURL = "";
    if ( ( typeof mapStuff !== "undefined" ) && mapStuff.closeIcon ) {
        closeIconURL = mapStuff.closeIcon;
    }

    let iwPixelOffsetY = -48;

    if ( rhMapsData.isUltra ) {
        iwPixelOffsetY = -70;
    } else if ( rhMapsData.isClassic ) {
        iwPixelOffsetY = -52;
    }

    if ( propertyMapData.marker_type === 'circle' ) {
        iwPixelOffsetY = -26;

        if ( rhMapsData.isUltra ) {
            iwPixelOffsetY = -22;
        } else if ( rhMapsData.isClassic ) {
            iwPixelOffsetY = -30;
        }
    }

    let iwMaxWidth       = 0,
        iwPixelOffsetX   = -122,
        iwCloseBoxMargin = '0 0 -16px -16px';

    if ( rhMapsData.isUltra ) {
        iwMaxWidth       = 450;
        iwPixelOffsetX   = -225;
        iwCloseBoxMargin = '8px 8px -24px -16px';
    }

    // Finalize info window
    var infoWindowOptions = {
        content                : boxText,
        disableAutoPan         : true,
        maxWidth               : iwMaxWidth,
        alignBottom            : true,
        pixelOffset            : new google.maps.Size( iwPixelOffsetX, iwPixelOffsetY ),
        zIndex                 : null,
        closeBoxMargin         : iwCloseBoxMargin,
        closeBoxURL            : closeIconURL,
        infoBoxClearance       : new google.maps.Size( 1, 1 ),
        isHidden               : false,
        pane                   : "floatPane",
        enableEventPropagation : false
    };

    var infoBox = new InfoBox( infoWindowOptions );

    google.maps.event.addListener( propertyMarker, 'click', function () {
        var scale                = Math.pow( 2, propertyMap.getZoom() );
        var offsety              = ( ( 150 / scale ) || 0 );
        var projection           = propertyMap.getProjection();
        var markerPosition       = propertyMarker.getPosition();
        var markerScreenPosition = projection.fromLatLngToPoint( markerPosition );
        var pointHalfScreenAbove = new google.maps.Point( markerScreenPosition.x, markerScreenPosition.y - offsety );
        var aboveMarkerLatLng    = projection.fromPointToLatLng( pointHalfScreenAbove );
        propertyMap.setCenter( aboveMarkerLatLng );
        infoBox.open( propertyMap, propertyMarker );
    } );

} )( jQuery );
