<?php
/**
 * Guest Property Views Manager
 *
 * This class handles the complete guest property view limitation functionality for the RealHomes theme.
 * It supports tracking property views using both the database, applies view limits based on
 * customizable timeframes (daily, weekly, bi-weekly, monthly, or forever), and redirects guests when
 * the view limit is reached. It also ensures cleanup of old records and initializes required database structures.
 *
 * Key Features:
 * - Tracks guest views using IP address
 * - Applies customizable limits via the Customizer settings.
 * - Redirects guests to a custom page or login when view limits are exceeded.
 * - Automatically creates and maintains the `realhomes_guest_property_view_stats` table.
 * - Includes scheduled cleanup of outdated view records.
 *
 * @since      4.4.1
 * @package    RealHomes
 */

if ( ! class_exists( 'RealHomes_Guest_Views_Manager' ) ) {

	class RealHomes_Guest_Views_Manager {

		/**
		 * Stores the current user's IP address.
		 *
		 * Used to track guest property views across requests without recalculating.
		 *
		 * @var string
		 */
		protected static $ip_address;

		/**
		 * Table name for storing guest property views.
		 *
		 * @var string
		 */
		protected string $table_name;

		/**
		 * Constructor for initializing guest property view tracking.
		 */
		public function __construct() {
			// Setting table name
			global $wpdb;
			$this->table_name = $wpdb->prefix . 'realhomes_guest_property_view_stats';

			// Setting current visitor IP
			self::$ip_address = realhomes_get_current_visitor_ip();

			add_action( 'wp_loaded', [ $this, 'maybe_create_guest_view_table' ] );
			add_action( 'template_redirect', [ $this, 'maybe_track_guest_view' ] );
			add_action( 'template_redirect', [ $this, 'maybe_limit_guest_access' ] );

			// Setting cleanup schedule cron job if required
			add_action( 'realhomes_cleanup_guest_views_cron', [ $this, 'cleanup_old_views' ] );
			$this->maybe_schedule_guest_views_cleanup();
		}

		/**
		 * Create the guest property view tracking table if it doesn't exist.
		 *
		 * Runs only when the guest property view limit feature is enabled.
		 * Uses dbDelta to create the table with IP, property ID, and view timestamp.
		 *
		 * @since 4.4.1
		 */
		public function maybe_create_guest_view_table() {

			global $wpdb;

			if ( $wpdb->get_var( "SHOW TABLES LIKE '{$this->table_name}'" ) !== $this->table_name ) {
				require_once ABSPATH . 'wp-admin/includes/upgrade.php';

				$sql = "CREATE TABLE {$this->table_name} (
					id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
					ip_address VARCHAR(45) NOT NULL,
					property_id BIGINT UNSIGNED NOT NULL,
					viewed_at DATETIME NOT NULL
				) {$wpdb->get_charset_collate()};";

				dbDelta( $sql );
			}
		}

		/**
		 * Track guest view for a single property.
		 *
		 * Records a view if the user is not logged in and viewing a single property.
		 *
		 * @since 4.4.1
		 */
		public function maybe_track_guest_view() {
			if ( is_user_logged_in() || ! is_singular( 'property' ) ) {
				return;
			}

			$property_id = get_queried_object_id();

			if ( $property_id ) {
				$this->record_view( $property_id );
			}
		}

		/**
		 * Record a guest view in the database.
		 *
		 * Inserts a row into the guest property views table with IP, property ID, and timestamp.
		 *
		 * @since 4.4.1
		 *
		 * @param int $property_id Property ID being viewed.
		 */
		public function record_view( $property_id ) {
			global $wpdb;
			$current_time = current_time( 'mysql' );

			if ( $wpdb->get_var( "SHOW TABLES LIKE '{$this->table_name}'" ) === $this->table_name ) {
				$wpdb->insert( $this->table_name, array(
					'ip_address'  => self::$ip_address,
					'property_id' => $property_id,
					'viewed_at'   => $current_time,
				) );
			}
		}

		/**
		 * Restrict guest access to property pages after reaching the view limit.
		 *
		 * Redirects the guest user to a custom page or login if the allowed view limit is exceeded.
		 *
		 * @since 4.4.1
		 */
		public function maybe_limit_guest_access() {
			if ( is_user_logged_in() || ! is_singular( 'property' ) ) {
				return;
			}

			$allowed_views = absint( get_option( 'realhomes_properties_views_limit_number', 5 ) );
			$current_views = $this->get_guest_view_count();

			if ( $current_views > $allowed_views ) {
				$page = new WP_Query( array(
					'post_type'      => 'page',
					'posts_per_page' => 1,
					'meta_key'       => '_wp_page_template',
					'meta_value'     => 'templates/properties-views-limit.php',
				) );

				if ( $page->have_posts() ) {
					wp_redirect( get_permalink( $page->posts[0]->ID ) );
					exit;
				}

				wp_redirect( wp_login_url() );
				exit;
			}
		}

		/**
		 * Get the number of unique properties viewed by the current guest.
		 *
		 * Applies time-based restrictions based on the configured view limit period.
		 *
		 * @since 4.4.1
		 *
		 * @return int Number of unique property views.
		 */
		public function get_guest_view_count() {
			global $wpdb;

			$limit_type     = get_option( 'realhomes_properties_views_time_limit', 'daily' );
			$time_condition = '';

			switch ( $limit_type ) {
				case 'daily':
					$time_condition = "AND viewed_at >= DATE_SUB(NOW(), INTERVAL 1 DAY)";
					break;
				case 'weekly':
					$time_condition = "AND viewed_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
					break;
				case 'twoweeks':
					$time_condition = "AND viewed_at >= DATE_SUB(NOW(), INTERVAL 14 DAY)";
					break;
				case 'monthly':
					$time_condition = "AND viewed_at >= DATE_SUB(NOW(), INTERVAL 1 MONTH)";
					break;
				case 'forever':
				default:
					$time_condition = '';
					break;
			}

			return absint( $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(DISTINCT property_id) FROM {$this->table_name} WHERE ip_address = %s {$time_condition}",
					self::$ip_address
				)
			) );
		}

		/**
		 * Cleanup old guest property views based on the selected reset interval.
		 *
		 * @since 4.4.1
		 */
		public function cleanup_old_views() {
			global $wpdb;
			$interval = get_option( 'realhomes_guest_property_view_period', 'forever' );

			// Map option to SQL interval
			switch ( $interval ) {
				case 'daily':
					$sql_interval = '1 DAY';
					break;
				case 'weekly':
					$sql_interval = '7 DAY';
					break;
				case 'biweekly':
					$sql_interval = '14 DAY';
					break;
				case 'monthly':
					$sql_interval = '1 MONTH';
					break;
				case 'forever':
				default:
					$sql_interval = '';
					break;
			}

			if ( $sql_interval ) {
				// Check if table exists
				$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $this->table_name ) );

				if ( $table_exists === $this->table_name ) {
					$deleted = $wpdb->query(
						"DELETE FROM {$this->table_name} WHERE viewed_at < DATE_SUB(NOW(), INTERVAL {$sql_interval})"
					);

					// Return number of deleted rows
					return $deleted;
				}
			}

			return 0;
		}

		/**
		 * Schedule the guest views cleanup event if not already scheduled.
		 *
		 * @since 4.4.1
		 */
		protected function maybe_schedule_guest_views_cleanup() {
			if ( ! wp_next_scheduled( 'realhomes_cleanup_guest_views_cron' ) ) {
				wp_schedule_event( time(), 'daily', 'realhomes_cleanup_guest_views_cron' );
			}
		}

	}

	/**
	 * Conditionally initializes the Guest Property Views Manager if the feature is enabled.
	 *
	 * @since 4.4.1
	 */
	function realhomes_maybe_initialize_guest_views_manager() {
		if ( 'true' === get_option( 'realhomes_properties_views_count_limit', 'false' ) ) {
			new RealHomes_Guest_Views_Manager();
		}
	}

	add_action( 'init', 'realhomes_maybe_initialize_guest_views_manager' );
}