<?php

/**
 * Handles rendering of custom icons (SVG or image) for property action buttons and meta fields.
 *
 * This class manages icon rendering based on Customizer settings, including caching,
 * sanitization of SVG content, and fallback support for image icons.
 *
 * @since 4.4.0
 */
class RealHomes_Render_Icons {
	/**
	 * List of Customizer setting IDs used to define custom icons for property action buttons.
	 *
	 * These keys correspond to option names stored in the WordPress database and are used
	 * to fetch and render SVG or image icons for features like favorites, compare, video, etc.
	 *
	 * @since 4.4.0
	 *
	 * @var array
	 */
	private static $customizer_ids = [
		'realhomes_favourite_button_icon',
		'realhomes_compare_button_icon',
		'realhomes_video_button_icon',
		'realhomes_image_button_icon',
		'realhomes_print_button_icon',
		'realhomes_share_button_icon',
		'realhomes_report_button_icon',
	];

	/**
	 * Caches and returns sanitized SVG icons for property action buttons.
	 *
	 * This method retrieves icon URLs from the Customizer settings defined in `self::$customizer_ids`,
	 * downloads SVG content, injects CSS classes, sanitizes the SVG markup, and stores the result in cache
	 * using `RealHomes_Cache::remember()` for improved performance.
	 *
	 * Only valid and sanitized SVGs are included in the returned cache array.
	 *
	 * @since 4.4.0
	 *
	 * @return array<string, string> Associative array of setting keys and sanitized SVG markup.
	 */
	public static function realhomes_cache_custom_icons() {
		return RealHomes_Cache::remember( 'property_action_buttons_svg_icons', function () {
			$svg_icons = [];

			foreach ( self::$customizer_ids as $field ) {
				$icon_url = get_option( $field );
				if ( empty( $icon_url ) ) {
					continue;
				}

				// Check if the file is an SVG.
				if ( preg_match( '/\.svg$/', $icon_url ) === 1 ) {
					// Download SVG content and cache it.
					$svg_file = wp_remote_get( $icon_url, [
						'timeout'     => 5,
						'redirection' => 2
					] );

					if ( ! is_array( $svg_file ) && is_wp_error( $svg_file ) ) {
						continue;
					}

					$svg_content = wp_remote_retrieve_body( $svg_file );

					$svg_class = 'custom-meta-icon custom-meta-icon-svg';
					if ( preg_match( '/<svg[^>]*\bclass\s*=\s*["\'](.*?)["\'][^>]*>/', $svg_content ) ) {
						$svg_content = str_replace( '<svg class="', '<svg class="' . $svg_class . ' ', $svg_content );
					} else {
						$svg_content = str_replace( '<svg', '<svg class="' . $svg_class . '"', $svg_content );
					}

					$sanitized_svg = ( new RealHomes_Sanitize_Svg() )->sanitize( $svg_content );

					if ( false !== $sanitized_svg ) {
						$svg_icons[ $field ] = $sanitized_svg;
					}
				}
			}

			return $svg_icons;

		}, 'property_action_buttons_svg_icons' );
	}

	/**
	 * Outputs a custom icon based on a customizer setting.
	 *
	 * This method checks if custom property meta icons are enabled, validates the provided customizer ID,
	 * attempts to retrieve a cached SVG icon, and falls back to rendering an image if no SVG is found.
	 *
	 * @since 4.4.0
	 *
	 * @param string $customizer_id The option key used to retrieve the custom icon URL from the database.
	 *
	 * @return bool True if an icon was successfully rendered, false otherwise.
	 */
	public static function realhomes_custom_icons( $customizer_id ) {
		// Early exit if conditions aren't met
		if ( ! realhomes_is_custom_property_meta_icons() || ! in_array( $customizer_id, self::$customizer_ids, true ) ) {
			return false;
		}

		// Get cached SVGs
		$svg_cache = self::realhomes_cache_custom_icons();

		if ( ! empty( $svg_cache[ $customizer_id ] ) ) {
			echo $svg_cache[ $customizer_id ];

			return true;
		}

		// Fallback to image if SVG not available
		$icon_url = get_option( $customizer_id );
		if ( ! empty( $icon_url ) ) {
			echo '<img class="custom-meta-icon custom-meta-icon-image" src="' . esc_url( $icon_url ) . '" alt="' . esc_attr__( 'Action button icon.', RH_TEXT_DOMAIN ) . '">';

			return true;
		}

		return false;
	}

	/**
	 * Outputs an icon in Elementor based on the selected icon control.
	 *
	 * This method checks the icon structure provided by Elementor's icon control,
	 * renders either an SVG, a font icon, or includes a fallback icon file.
	 *
	 * @since 4.4.0
	 *
	 * @param array  $button_icon            Elementor icon control array (from `Icons_Manager`).
	 * @param string $default_icon_file_name Optional fallback icon file name to include if no icon is selected.
	 *
	 * @return void
	 */
	public static function realhomes_generate_elementor_icon( $button_icon, $default_icon_file_name = '' ) {
		if ( ! empty( $button_icon ) ) {
			if ( is_array( $button_icon['value'] ) && ! empty( $button_icon['value']['url'] ) ) {
				?>
                <span><?php \Elementor\Icons_Manager::render_icon( $button_icon, [ 'aria-hidden' => 'true' ] ); ?></span>
				<?php
			} else if ( ! empty( $button_icon['library'] && ! empty( $button_icon['value'] ) ) ) {
				?>
                <i class="<?php echo esc_attr( $button_icon['library'] . ' ' . $button_icon['value'] ) ?>"></i>
				<?php
			} else if ( ! empty( $default_icon_file_name ) ) {
				include RHEA_ASSETS_DIR . '/icons/' . $default_icon_file_name;
			}
		}
	}
}