<?php
if ( ! function_exists( 'inspiry_get_google_fonts' ) ) {
	/**
	 * Retrieves an array of all available Google Fonts.
	 *
	 * This function fetches Google Fonts from a JSON file included in the theme package and caches them in the database.
	 * A hardcoded update date ($current_update_date) ensures that when a new font file is introduced, it is processed only once
	 * per installation, minimizing unnecessary JSON file parsing and improving performance.
	 *
	 * The function also removes italic variants from the font list and ensures the fonts data is stored with autoload disabled
	 * to optimize database performance.
	 *
	 * Additionally, if the fonts option is found to be autoloaded, it is deleted before storing the updated data.
	 *
	 * Related Trello Card: https://trello.com/c/1oPCwRM6/2341-compress-google-fonts-file-to-avoid-excessive-bandwidth-usage
	 *
	 * @modified 3.21.0
	 * @modified 4.3.9  march 13 2025
	 *
	 * @return array An associative array of Google Fonts with font families as keys.
	 */
	function inspiry_get_google_fonts() {
		global $wpdb;

		$option_name        = 'realhomes_customizer_google_fonts_array';
		$update_date_option = 'realhomes_customizer_google_fonts_update_date';

		// Retrieve the currently stored fonts and update date.
		$fonts           = get_option( $option_name );
		$old_update_date = get_option( $update_date_option );

		// Hardcoded last updated date.
		$current_update_date      = 'February 2 2025';
		$current_update_timestamp = strtotime( $current_update_date );

		// Always remove 'lastUpdated' key before proceeding
		if ( is_array( $fonts ) && isset( $fonts['lastUpdated'] ) ) {
			unset( $fonts['lastUpdated'] );
		}

		// If the stored fonts are valid and the file is not newer, return existing fonts.
		if (
			is_array( $fonts ) &&
			count( $fonts ) > 0 &&
			$current_update_timestamp < intval( $old_update_date )
		) {
			return $fonts;
		}

		// Check if the option is autoloaded and remove it if necessary.
		$autoload = $wpdb->get_var( $wpdb->prepare(
			"SELECT autoload FROM {$wpdb->options} WHERE option_name = %s",
			$option_name
		) );

		if ( 'yes' === $autoload ) {
			delete_option( $option_name );
		}

		// Process new fonts data.
		$fonts      = array();
		$fonts_file = wp_remote_fopen( get_template_directory_uri() . '/framework/customizer/google-fonts/webfonts.json' );

		if ( ! empty( $fonts_file ) ) {
			$fonts_array = json_decode( $fonts_file, true );

			if ( isset( $fonts_array['items'] ) && is_array( $fonts_array['items'] ) ) {
				$remove_variants = array(
					'italic',
					'100italic',
					'200italic',
					'300italic',
					'400italic',
					'500italic',
					'600italic',
					'700italic',
					'800italic',
					'900italic',
					'100i',
					'200i',
					'300i',
					'400i',
					'500i',
					'600i',
					'700i',
					'800i',
					'900i'
				);

				foreach ( $fonts_array['items'] as $font ) {
					$font['variants']  = array_diff( $font['variants'], $remove_variants );
					$font_id           = trim( str_replace( ' ', '+', $font['family'] ) );
					$fonts[ $font_id ] = $font;
				}
			}
		}

		// Update the fonts option with autoload disabled.
		update_option( $option_name, $fonts, false );

		// Store the current timestamp for future updates.
		update_option( $update_date_option, current_time( 'timestamp' ), false );

		return $fonts;
	}
}


/**
 * This class provides the Google Fonts related functionality.
 */
class Inspiry_Google_Fonts {

	/**
	 * All Google fonts.
	 *
	 * @var array
	 */
	public static $fonts = array();

	/**
	 * Google font families list.
	 *
	 * @var array
	 */
	public static $fonts_list = array();

	/**
	 * Instance
	 *
	 * @var Inspiry_Google_Fonts The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Provides singleton instance.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * Let's get started.
	 */
	public function __construct() {
		self::$fonts      = inspiry_get_google_fonts();
		self::$fonts_list = self::get_list();
		add_action( 'wp_ajax_inspiry_get_font_weights', array( $this, 'get_font_weights_markup' ) );
	}

	/**
	 * Get the fonts list.
	 */
	private static function get_list() {
		$fonts_list = array();

		$fonts = self::$fonts;
		if ( ! empty( $fonts ) && is_array( $fonts ) ) {
			foreach ( $fonts as $font_id => $font ) {
				$fonts_list[ $font_id ] = esc_html( $font['family'] );
			}

			ksort( $fonts_list, SORT_STRING );
		}

		return array( 'Default' => esc_html__( 'Default', RH_TEXT_DOMAIN ) ) + $fonts_list;
	}

	/**
	 * Get the font weights for given font ID.
	 *
	 * @param string $font_id The font ID.
	 * @param bool $show_default
	 * @param bool $join
	 *
	 * @return array|string
	 */
	public static function get_font_weights( $font_id, $show_default = true, $join = false ) {

		$font_weights = array();
		if ( $show_default ) {
			$font_weights['Default'] = esc_html__( 'Default', RH_TEXT_DOMAIN );
		}

		if ( empty( $font_id ) || 'Default' === $font_id ) {
			return $font_weights;
		}

		if ( isset( self::$fonts[ $font_id ] ) ) {
			$variants = self::$fonts[ $font_id ]['variants'];
			if ( ! empty( $variants ) && is_array( $variants ) ) {
				foreach ( $variants as $variant ) {
					if ( 'regular' === $variant ) {
						$variant = '400';
					}
					$font_weights[ $variant ] = $variant;
				}
			}
		}

		if ( $join ) {
			return join( ',', $font_weights );
		}

		return $font_weights;
	}

	/**
	 * Get the font weights from ID and display markup.
	 */
	public function get_font_weights_markup() {
		if ( isset( $_POST['family'] ) && ! empty( $_POST['family'] ) ) {
			$weights = self::get_font_weights( sanitize_text_field( $_POST['family'] ) );
			$output  = '';
			if ( ! empty( $weights ) && is_array( $weights ) ) {
				foreach ( $weights as $weight ) {
					$output .= '<option value="' . $weight . '">' . $weight . '</option>';
				}
			}

			echo $output;
		}
	}

	/**
	 * Get the font family from ID for fonts URL.
	 *
	 * @param string $font_id The font ID.
	 *
	 * @return string
	 */
	public static function get_font_family( $font_id ) {
		$font = self::$fonts[ $font_id ];

		return sprintf( "'%s', %s", $font['family'], $font['category'] );
	}
}

// Main instance of Inspiry_Google_Fonts.
Inspiry_Google_Fonts::instance();