<?php
/**
 * This file contain theme's real estate functions
 *
 * @package realhomes/functions
 */
if ( ! function_exists( 'inspiry_get_properties_sort_by_value' ) ) {
	/**
	 * Retrieves the properties sort by value.
	 *
	 * @return string Properties sort by value.
	 * 
	 * @since 3.10.1
	 */
	function inspiry_get_properties_sort_by_value() {

		$sort_by_value = '';

		if ( isset( $_GET['sortby'] ) ) {
			$sort_by_value = sanitize_text_field( $_GET['sortby'] );
		} else {
			$sort_by_value = get_option( 'theme_listing_default_sort' );

			if ( is_page_template( 'templates/properties.php' ) ) {
				$page_sort_by_value = get_post_meta( get_the_ID(), 'inspiry_properties_order', true );
				if ( 'default' !== $page_sort_by_value ) {
					$sort_by_value = $page_sort_by_value;
				}
			}
		}

		return apply_filters( 'inspiry_get_properties_sort_by_value', $sort_by_value );
	}
}

if ( ! function_exists( 'sort_properties' ) ) {
	/**
	 * This function adds sorting parameters to given query arguments
	 *
	 * @modified 4.1.0
     * @modified 4.3.8
	 *
	 * @param array $property_query_args Query arguments for properties.
	 * @param bool  $sort_by             Sorting criteria.
	 *
	 * @return mixed
	 */
	function sort_properties( $property_query_args, $sort_by = false ) {

		if ( ! $sort_by ) {
			$sort_by = inspiry_get_properties_sort_by_value();
		}

		// Handle sorting based on the selected option
		switch ( $sort_by ) {
			case 'title-asc':
				$property_query_args['orderby'] = 'title';
				$property_query_args['order']   = 'ASC';
				break;

			case 'title-desc':
				$property_query_args['orderby'] = 'title';
				$property_query_args['order']   = 'DESC';
				break;

			case 'price-asc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_price';
				$property_query_args['order']    = 'ASC';
				break;

			case 'price-desc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_price';
				$property_query_args['order']    = 'DESC';
				break;

			case 'date-asc':
				if ( isset( $property_query_args['meta_key'] ) && ( $property_query_args['meta_key'] == 'REAL_HOMES_featured' ) ) {
					$property_query_args['orderby'] = array(
						'meta_value_num' => 'DESC',
						'date'           => 'ASC',
					);
				} else {
					$property_query_args['orderby'] = 'date';
					$property_query_args['order']   = 'ASC';
				}
				break;

			case 'date-desc':
				if ( isset( $property_query_args['meta_key'] ) && ( $property_query_args['meta_key'] == 'REAL_HOMES_featured' ) ) {
					$property_query_args['orderby'] = array(
						'meta_value_num' => 'DESC',
						'date'           => 'DESC',
					);
				} else {
					$property_query_args['orderby'] = 'date';
					$property_query_args['order']   = 'DESC';
				}
				break;

			// New sorting options
			case 'bed-asc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_bedrooms';
				$property_query_args['order']    = 'ASC';
				break;

			case 'bed-desc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_bedrooms';
				$property_query_args['order']    = 'DESC';
				break;

			case 'bath-asc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_bathrooms';
				$property_query_args['order']    = 'ASC';
				break;

			case 'bath-desc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_bathrooms';
				$property_query_args['order']    = 'DESC';
				break;

			case 'area-asc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_size';
				$property_query_args['order']    = 'ASC';
				break;

			case 'area-desc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_size';
				$property_query_args['order']    = 'DESC';
				break;

			case 'lot-asc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_lot_size';
				$property_query_args['order']    = 'ASC';
				break;

			case 'lot-desc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_lot_size';
				$property_query_args['order']    = 'DESC';
				break;

			case 'garage-asc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_garage';
				$property_query_args['order']    = 'ASC';
				break;

			case 'garage-desc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_garage';
				$property_query_args['order']    = 'DESC';
				break;

			case 'year-asc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_year_built';
				$property_query_args['order']    = 'ASC';
				break;

			case 'year-desc':
				$property_query_args['orderby']  = 'meta_value_num';
				$property_query_args['meta_key'] = 'REAL_HOMES_property_year_built';
				$property_query_args['order']    = 'DESC';
				break;

			case 'views-desc':
			case 'views-asc':
				// Get properties sorted by views
				$viewed_properties = ere_get_properties_by_view_count( -1 );

				if ( ! empty( $viewed_properties ) && is_array( $viewed_properties ) ) {
					$ordered_ids = array_map( function( $item ) {
						return intval( $item->PID );
					}, $viewed_properties );

					if ( 'views-asc' === $sort_by ) {
						$ordered_ids = array_reverse( $ordered_ids );
					}

					// Filter existing post__in if defined
					if ( isset( $property_query_args['post__in'] ) && is_array( $property_query_args['post__in'] ) ) {
						$ordered_ids = array_values( array_intersect( $ordered_ids, $property_query_args['post__in'] ) );
					}

					$property_query_args['post__in'] = $ordered_ids;
					$property_query_args['orderby']  = 'post__in';
				}
				break;

			default:
				// Default sorting (if no match)
				$property_query_args['orderby'] = 'date';
				$property_query_args['order']   = 'DESC';
				break;
		}

		return apply_filters( 'inspiry_sort_properties', $property_query_args );
	}
}

if ( ! function_exists( 'realhomes_property_archive_sort' ) ) {
	/**
     * Modifies property archive query with respect to sorting
	 * @param $query
	 */
	function realhomes_property_archive_sort( $query ) {
		if ( $query->is_main_query() && ! is_admin() ) {
			if ( $query->is_post_type_archive( 'property' ) ) {
				$sort_query_args = sort_properties( array() );
				// order by
				if ( isset( $sort_query_args['orderby'] ) ) {
					$query->set( 'orderby', $sort_query_args['orderby'] );
				}
				// order by
				if ( isset( $sort_query_args['meta_key'] ) ) {
					$query->set( 'meta_key', $sort_query_args['meta_key'] );
				}
				// order by
				if ( isset( $sort_query_args['order'] ) ) {
					$query->set( 'order', $sort_query_args['order'] );
				}
			}
		}
	}

	add_action( 'pre_get_posts', 'realhomes_property_archive_sort' );
}

if ( ! function_exists( 'inspiry_get_sort_by_value' ) ) {
	/**
	 * Retrieves the sort value for the page.
     *
     * @since 3.10.2
	 *
	 * @param string $sort_by_value
	 *
	 * @return string sort by value.
	 */
	function inspiry_get_sort_by_value( $sort_by_value = 'default' ) {

		if ( isset( $_GET['sortby'] ) ) {
			$sort_by_value = sanitize_text_field( $_GET['sortby'] );
		}

		return $sort_by_value;
	}
}

if ( ! function_exists( 'inspiry_agents_sort_args' ) ) {
	/**
	 * Adds sorting parameters to agents query arguments.
     *
     * @since 3.10.2
	 *
	 * @param $agent_query_args
	 *
	 * @return mixed
	 */
	function inspiry_agents_sort_args( $agent_query_args ) {

		$sort_by = inspiry_get_sort_by_value();

		if ( 'title-asc' === $sort_by ) {
			$agent_query_args['orderby']  = 'title';
			$agent_query_args['order']    = 'ASC';

		} elseif ( 'title-desc' === $sort_by ) {
			$agent_query_args['orderby']  = 'title';
			$agent_query_args['order']    = 'DESC';

		} elseif ( 'total-asc' === $sort_by || 'total-desc' === $sort_by ) {

			$agents_to_sort = array();
			$agents = get_posts( array(
				'post_type'   => 'agent',
				'numberposts' => - 1,
			) );

			if ( ! empty( $agents ) && function_exists( 'ere_get_agent_properties_count' ) ) {
				foreach ( $agents as $agent ) {
					$agent_id = $agent->ID;
					$agents_to_sort[ $agent_id ] = ere_get_agent_properties_count( $agent_id );
				}
			}

			if ( ! empty( $agents_to_sort ) && is_array( $agents_to_sort ) ) {
				if ( 'total-asc' === $sort_by ) {
					asort( $agents_to_sort );
				} elseif ( 'total-desc' === $sort_by ) {
					arsort( $agents_to_sort );
				}

				$agent_query_args['post__in'] = array_keys( $agents_to_sort );
				$agent_query_args['orderby']  = 'post__in';
			}
		}

		return apply_filters( 'inspiry_agents_sort_args', $agent_query_args );
	}
}

if ( ! function_exists( 'inspiry_agent_sort_options' ) ) {
	/**
	 * Displays agent sort options for select.
	 *
     * @since 3.10.2
	 */
	function inspiry_agent_sort_options() {

		$sort_options = apply_filters( 'inspiry_agent_sort_options', array(
			'default'    => esc_html__( 'Default Order', RH_TEXT_DOMAIN ),
			'title-asc'  => esc_html__( 'Agent Name ( A to Z )', RH_TEXT_DOMAIN ),
			'title-desc' => esc_html__( 'Agent Name ( Z to A )', RH_TEXT_DOMAIN ),
			'total-asc'  => esc_html__( 'Listed Properties ( Low to High )', RH_TEXT_DOMAIN ),
			'total-desc' => esc_html__( 'Listed Properties ( High to Low )', RH_TEXT_DOMAIN )
		) );

		if ( ! empty( $sort_options ) && is_array( $sort_options ) ) {

			$html     = '';
			$selected = apply_filters( 'inspiry_agents_sort_by_value', inspiry_get_sort_by_value() );

			foreach ( $sort_options as $key => $value ) {
				$html .= sprintf( '<option value="%s"%s>%s</option>', esc_attr( $key ), selected( $selected, $key, false ), esc_html( $value ) );
			}

			echo wp_kses(
				$html,
				array(
					'option' => array(
						'value'    => array(),
						'selected' => array(),
					),
				)
			);
		}
	}
}

if ( ! function_exists( 'inspiry_agency_sort_options' ) ) {
	/**
	 * Displays agency sort options for select.
	 *
	 * @since 3.11.0
	 */
	function inspiry_agency_sort_options() {

		$sort_options = apply_filters( 'inspiry_agency_sort_options', array(
			'default'    => esc_html__( 'Default Order', RH_TEXT_DOMAIN ),
			'title-asc'  => esc_html__( 'Agency Name ( A to Z )', RH_TEXT_DOMAIN ),
			'title-desc' => esc_html__( 'Agency Name ( Z to A )', RH_TEXT_DOMAIN ),
			'total-asc'  => esc_html__( 'Listed Properties ( Low to High )', RH_TEXT_DOMAIN ),
			'total-desc' => esc_html__( 'Listed Properties ( High to Low )', RH_TEXT_DOMAIN ),
		) );

		if ( ! empty( $sort_options ) && is_array( $sort_options ) ) {

			$html = '';
			$selected = apply_filters( 'inspiry_agencies_sort_by_value', inspiry_get_sort_by_value() );

			foreach ( $sort_options as $key => $value ) {
				$html .= sprintf( '<option value="%s"%s>%s</option>', esc_attr( $key ), selected( $selected, $key, false ), esc_html( $value ) );
			}

			echo wp_kses(
				$html,
				array(
					'option' => array(
						'value'    => array(),
						'selected' => array(),
					),
				)
			);
		}
	}
}

if ( ! function_exists( 'inspiry_agencies_sort_args' ) ) {
	/**
	 * Adds sorting parameters to agencies query arguments.
	 *
	 * @since 3.11.0
	 *
	 * @param $agency_query_args
	 *
	 * @return mixed
	 */
	function inspiry_agencies_sort_args( $agency_query_args ) {

		$sort_by = inspiry_get_sort_by_value();

		if ( 'title-asc' === $sort_by ) {
			$agency_query_args['orderby']  = 'title';
			$agency_query_args['order']    = 'ASC';

		} elseif ( 'title-desc' === $sort_by ) {
			$agency_query_args['orderby']  = 'title';
			$agency_query_args['order']    = 'DESC';

		} elseif ( 'total-asc' === $sort_by || 'total-desc' === $sort_by ) {

			$agencies_to_sort = array();
			$agencies         = get_posts( array(
				'post_type'   => 'agency',
				'numberposts' => - 1,
			) );

			if ( ! empty( $agencies ) && function_exists( 'ere_get_agency_properties_count' ) ) {
				foreach ( $agencies as $agency ) {
					$agency_id = $agency->ID;
					$agencies_to_sort[ $agency_id ] = ere_get_agency_properties_count( $agency_id );
				}
			}

			if ( ! empty( $agencies_to_sort ) && is_array( $agencies_to_sort ) ) {
				if ( 'total-asc' === $sort_by ) {
					asort( $agencies_to_sort );
				} elseif ( 'total-desc' === $sort_by ) {
					arsort( $agencies_to_sort );
				}

				$agency_query_args['post__in'] = array_keys( $agencies_to_sort );
				$agency_query_args['orderby']  = 'post__in';
			}
		}

		return apply_filters( 'inspiry_agencies_sort_args', $agency_query_args );
	}
}

if ( ! function_exists( 'realhomes_admin_styles' ) ) {
	/**
	 * Register and load admin styles
	 *
	 * @param $hook
	 */
	function realhomes_admin_styles( $hook ) {
		wp_register_style( 'realhomes-admin-styles', get_theme_file_uri( 'common/css/admin-styles.min.css' ) );
		wp_enqueue_style( 'realhomes-admin-styles' );
	}

	add_action( 'admin_enqueue_scripts', 'realhomes_admin_styles', 100 );
}

if ( ! function_exists( 'inspiry_get_figure_caption' ) ) {
	/**
	 * Figure caption based on property statuses
	 *
	 * @param $post_id
	 *
	 * @return string
	 */
	function inspiry_get_figure_caption( $post_id ) {
		$status_terms = get_the_terms( $post_id, "property-status" );
		if ( ! empty( $status_terms ) && ! is_wp_error( $status_terms ) ) {
			$status_classes = '';
			$status_names   = '';
			$status_count   = 0;
			foreach ( $status_terms as $term ) {
				if ( $status_count > 0 ) {
					$status_names   .= ', ';  /* add comma before the term namee of 2nd and any later term */
					$status_classes .= ' ';
				}
				$status_names   .= $term->name;
				$status_classes .= $term->slug;
				$status_count ++;
			}

			if ( ! empty( $status_names ) ) {
				return '<figcaption class="' . $status_classes . '">' . $status_names . '</figcaption>';
			}

			return '';
		}
	}
}

if ( ! function_exists( 'display_figcaption' ) ) {
	/**
	 * Display figure caption for given property's post id
	 *
	 * @param $post_id
	 */
	function display_figcaption( $post_id ) {
		echo inspiry_get_figure_caption( $post_id );
	}
}

if ( ! function_exists( 'display_property_status_html' ) ) {
	/**
	 * Display property status.
	 *
	 * @param $post_id
	 *
	 */
	function display_property_status_html( $post_id ) {
		$status_terms = get_the_terms( $post_id, 'property-status' );

		if ( ! empty( $status_terms ) && ! is_wp_error( $status_terms ) ) {


			foreach ( $status_terms as $term ) {

			    $inspiry_property_status_bg = get_term_meta($term->term_id,'inspiry_property_status_bg',true);
				$status_bg = '';
			    if(!empty($inspiry_property_status_bg)){
			        $status_bg = " background:" . "$inspiry_property_status_bg; ";
                }
                $inspiry_property_status_text = get_term_meta($term->term_id,'inspiry_property_status_text',true);

			    $status_text = '';
			    if(!empty($inspiry_property_status_text)){
				    $status_text = " color:" . "$inspiry_property_status_text; ";
                }

			    ?>
                <span class="rh_prop_status_sty" style="<?php echo esc_attr($status_bg . $status_text)?>">
                <?php
				echo esc_html($term->name);
				?>
				</span>
                <?php
			}
		}

	}
}

if ( ! function_exists( 'display_property_status' ) ) {
	/**
	 * Display property status.
	 *
	 * @param $post_id
	 *
	 * @return string
	 */
	function display_property_status( $post_id ) {
		$status_terms = get_the_terms( $post_id, 'property-status' );

		if ( ! empty( $status_terms ) && ! is_wp_error( $status_terms ) ) {

			$status_names = '';
			$status_count = 0;

			foreach ( $status_terms as $term ) {
				if ( $status_count > 0 ) {
					$status_names .= ', ';  /* add comma before the term name of 2nd and any later term */
				}
				$status_names .= $term->name;
				$status_count ++;
			}

			if ( ! empty( $status_names ) ) {
				return $status_names;
			}
		}

		return '';
	}
}

if ( ! function_exists( 'inspiry_display_property_label' ) ) {
	/**
	 * Display property label
	 *
	 * @param $post_id
     * @param string $class
	 */
	function inspiry_display_property_label( $post_id, $class='' ) {

		$label_text = get_post_meta( $post_id, 'inspiry_property_label', true );
		$color      = get_post_meta( $post_id, 'inspiry_property_label_color', true );
		if ( ! empty ( $label_text ) ) {
			?>
            <span <?php if ( ! empty( $color ) ){ ?>style="background: <?php echo esc_attr( $color ); ?>"<?php } ?>
                  class='property-label <?php echo esc_attr($class)?>'><?php echo esc_html( $label_text ); ?></span>
			<?php

		}
	}
}

if ( ! function_exists( 'inspiry_get_property_types' ) ) {
	/**
	 * Get property types
	 *
	 * @param $property_post_id
	 *
	 * @return string
	 */
	function inspiry_get_property_types( $property_post_id ) {
		$type_terms = get_the_terms( $property_post_id, "property-type" );
		if ( ! empty( $type_terms ) && ! is_wp_error( $type_terms ) ) {
			$type_count         = count( $type_terms );
			$property_types_str = '<small> - ';
			$loop_count         = 1;
			foreach ( $type_terms as $typ_trm ) {
				$property_types_str .= $typ_trm->name;
				if ( $loop_count < $type_count && $type_count > 1 ) {
					$property_types_str .= ', ';
				}
				$loop_count ++;
			}
			$property_types_str .= '</small>';
		} else {
			$property_types_str = '&nbsp;';
		}

		return $property_types_str;
	}
}

if ( ! function_exists( 'inspiry_get_property_types_string' ) ) {
	/**
	 * Get property types
	 *
	 * @param $property_post_id
	 *
	 * @return string
	 */
	function inspiry_get_property_types_string( $property_post_id ) {
		$type_terms = get_the_terms( $property_post_id, "property-type" );
		if ( ! empty( $type_terms ) && ! is_wp_error( $type_terms ) ) {
			$type_count = count( $type_terms );
			$property_types_str = '';
			$loop_count         = 1;
			foreach ( $type_terms as $typ_trm ) {
				$property_types_str .= $typ_trm->name;
				if ( $loop_count < $type_count && $type_count > 1 ) {
					$property_types_str .= ', ';
				}
				$loop_count ++;
			}
		} else {
			$property_types_str = '&nbsp;';
		}

		return $property_types_str;
	}
}

if ( ! function_exists( 'inspiry_get_property_status' ) ) {
	/**
	 * Returns property status
	 *
	 * @param $post_id
	 *
	 * @return string
	 */
	function inspiry_get_property_status( $post_id ) {
		$status_terms = get_the_terms( $post_id, "property-status" );
		if ( ! empty( $status_terms ) && ! is_wp_error( $status_terms ) ) {
			$status_links = '';
			$status_count = 0;
			foreach ( $status_terms as $term ) {
				if ( $status_count > 0 ) {
					$status_links .= ' ';
				}
				$status_href  = get_term_link( $term );
				$status_href  = ( ! is_wp_error( $status_href ) ) ? $status_href : '';
				$status_links .= '<a href=" ' . $status_href . ' ">' . $term->name . '</a>';
				$status_count ++;
			}

			if ( ! empty( $status_links ) ) {
				return $status_links;
			}

			return '';
		}
	}
}

if ( ! function_exists( 'inspiry_get_number_of_photos' ) ) {
	/**
	 * Returns the number of photos in a gallery of property
	 *
	 * @param  $post_id
	 *
	 * @return int
	 * @since  2.6.3
	 */
	function inspiry_get_number_of_photos( $post_id ) {
		if ( function_exists( 'rwmb_meta' ) ) {
			$properties_images = rwmb_meta( 'REAL_HOMES_property_images', 'type=plupload_image', $post_id );

			return count( $properties_images );
		}

		return 0;
	}
}

if ( ! function_exists( 'inspiry_add_term_children' ) ) {
	/**
	 * A recursive function to add children terms to given array
	 *
	 * @param $parent_id
	 * @param $tax_terms
	 * @param $terms_array
	 * @param string $prefix
	 */
	function inspiry_add_term_children( $parent_id, $tax_terms, &$terms_array, $prefix = '' ) {
		if ( ! empty( $tax_terms ) && ! is_wp_error( $tax_terms ) ) {
			foreach ( $tax_terms as $term ) {
				if ( $term->parent == $parent_id ) {
					$terms_array[ $term->slug ] = $prefix . $term->name;
					inspiry_add_term_children( $term->term_id, $tax_terms, $terms_array, $prefix . '- ' );
				}
			}
		}
	}
}

if ( ! function_exists( 'inspiry_properties_filter' ) ) {
	/**
	 * Add properties filter parameters to given query arguments
	 *
	 * @param $properties_query_args  Array   query arguments
	 *
	 * @return mixed    Array   modified query arguments
	 */
	function inspiry_properties_filter( $properties_query_args ) {

		// Apply pagination
		$paged = 1;
		if ( get_query_var( 'paged' ) ) {
			$paged = get_query_var( 'paged' );
		} elseif ( get_query_var( 'page' ) ) { // if is static front page
			$paged = get_query_var( 'page' );
		}

		$properties_query_args['paged'] = $paged;

		$page_id    = get_the_ID();
		$tax_query  = array();
		$meta_query = array();

		if ( '1' === get_post_meta( $page_id, 'inspiry_featured_properties_only', true ) ) {
			$meta_query[] = array(
				'key'     => 'REAL_HOMES_featured',
				'value'   => 1,
				'compare' => '=',
				'type'    => 'NUMERIC',
			);
		}
		
		// Number of properties on each page
		$theme_number_of_properties = get_option( 'theme_number_of_properties', 6 );
		$number_of_properties       = get_post_meta( $page_id, 'inspiry_posts_per_page', true );

		if ( $number_of_properties ) {
			$properties_query_args['posts_per_page'] = $number_of_properties;
		} elseif ( ! empty( $theme_number_of_properties ) ) {
			$properties_query_args['posts_per_page'] = $theme_number_of_properties;
		} else {
			$properties_query_args['posts_per_page'] = 6;
		}

		// Locations
		$locations = get_post_meta( $page_id, 'inspiry_properties_locations', false );
		if ( ! empty( $locations ) && is_array( $locations ) ) {
			$tax_query[] = array(
				'taxonomy' => 'property-city',
				'field'    => 'slug',
				'terms'    => $locations
			);
		}

		// Statuses
		$statuses = get_post_meta( $page_id, 'inspiry_properties_statuses', false );
		if ( ! empty( $statuses ) && is_array( $statuses ) ) {
			$tax_query[] = array(
				'taxonomy' => 'property-status',
				'field'    => 'slug',
				'terms'    => $statuses
			);
		}

		// Types
		$types = get_post_meta( $page_id, 'inspiry_properties_types', false );
		if ( ! empty( $types ) && is_array( $types ) ) {
			$tax_query[] = array(
				'taxonomy' => 'property-type',
				'field'    => 'slug',
				'terms'    => $types
			);
		}

		// Features
		$features = get_post_meta( $page_id, 'inspiry_properties_features', false );
		if ( ! empty( $features ) && is_array( $features ) ) {
			$tax_query[] = array(
				'taxonomy' => 'property-feature',
				'field'    => 'slug',
				'terms'    => $features
			);
		}

		// If more than one taxonomies exist then specify the relation
		$tax_count = count( $tax_query );
		if ( $tax_count > 1 ) {
			$tax_query['relation'] = 'AND';
		}
		if ( $tax_count > 0 ) {
			$properties_query_args['tax_query'] = $tax_query;
		}

		// Minimum Bedrooms
		$min_beds = get_post_meta( $page_id, 'inspiry_properties_min_beds', true );
		if ( ! empty( $min_beds ) ) {
			$min_beds = intval( $min_beds );
			if ( $min_beds > 0 ) {
				$meta_query[] = array(
					'key'     => 'REAL_HOMES_property_bedrooms',
					'value'   => $min_beds,
					'compare' => '>=',
					'type'    => 'DECIMAL'
				);
			}
		}

		// Minimum Bathrooms
		$min_baths = get_post_meta( $page_id, 'inspiry_properties_min_baths', true );
		if ( ! empty( $min_baths ) ) {
			$min_baths = intval( $min_baths );
			if ( $min_baths > 0 ) {
				$meta_query[] = array(
					'key'     => 'REAL_HOMES_property_bathrooms',
					'value'   => $min_baths,
					'compare' => '>=',
					'type'    => 'DECIMAL'
				);
			}
		}

		// Min & Max Price
		$min_price = get_post_meta( $page_id, 'inspiry_properties_min_price', true );
		$max_price = get_post_meta( $page_id, 'inspiry_properties_max_price', true );
		if ( ! empty( $min_price ) && ! empty( $max_price ) ) {
			$min_price = doubleval( $min_price );
			$max_price = doubleval( $max_price );
			if ( $min_price >= 0 && $max_price > $min_price ) {
				$meta_query[] = array(
					'key'     => 'REAL_HOMES_property_price',
					'value'   => array( $min_price, $max_price ),
					'type'    => 'NUMERIC',
					'compare' => 'BETWEEN'
				);
			}
		} elseif ( ! empty( $min_price ) ) {
			$min_price = doubleval( $min_price );
			if ( $min_price > 0 ) {
				$meta_query[] = array(
					'key'     => 'REAL_HOMES_property_price',
					'value'   => $min_price,
					'type'    => 'NUMERIC',
					'compare' => '>='
				);
			}
		} elseif ( ! empty( $max_price ) ) {
			$max_price = doubleval( $max_price );
			if ( $max_price > 0 ) {
				$meta_query[] = array(
					'key'     => 'REAL_HOMES_property_price',
					'value'   => $max_price,
					'type'    => 'NUMERIC',
					'compare' => '<='
				);
			}
		}


		// Agents, use array_filter to remove empty values.
		$agents          = get_post_meta( $page_id, 'inspiry_properties_by_agents', false );
		$filtered_agents = ( is_array( $agents ) ? array_filter( $agents ) : '' );
		if ( is_array( $filtered_agents ) && count( $filtered_agents ) >= 1 ) {
			$meta_query[] = array(
				'key'     => 'REAL_HOMES_agents',
				'value'   => $filtered_agents,
				'compare' => 'IN',
			);
		}

		// If more than one meta query elements exist then specify the relation.
		$meta_count = count( $meta_query );
		if ( $meta_count > 1 ) {
			$meta_query['relation'] = 'AND';
		}
		if ( $meta_count > 0 ) {
			$properties_query_args['meta_query'] = $meta_query;
		}

		return $properties_query_args;
	}

	add_filter( 'inspiry_properties_filter', 'inspiry_properties_filter' );
}

if ( ! function_exists( 'inspiry_property_id_meta_update' ) ) {
	/**
	 * This function adds Property ID to properties
	 * in the backend if Auto Property ID generation
	 * is enabled.
	 *
	 * @since 3.2.0
	 */
	function inspiry_property_id_meta_update( $meta_id, $property_id, $meta_key, $meta_value ) {

		// Check if parameters are empty.
		if ( empty( $meta_id ) || empty( $property_id ) || empty( $meta_key ) ) {
			return;
		}

		// Check if meta key is not REAL_HOMES_property_id.
		if ( 'REAL_HOMES_property_id' !== $meta_key ) {
			return;
		}

		// Check if auto property id auto generation is enabled.
		$auto_property_id = get_option( 'inspiry_auto_property_id_check' );
		if ( ! empty( $auto_property_id ) && 'true' === $auto_property_id ) {

			$property_id_old     = get_post_meta( $property_id, 'REAL_HOMES_property_id', true );
			$property_id_pattern = get_option( 'inspiry_auto_property_id_pattern' );
			$property_id_value   = preg_replace( '/{ID}/', $property_id, $property_id_pattern );

			if ( $property_id_old !== $property_id_value ) {
				update_post_meta( $property_id, 'REAL_HOMES_property_id', $property_id_value );
			}
		}

	}

	add_action( 'added_post_meta', 'inspiry_property_id_meta_update', 10, 4 );
	add_action( 'updated_post_meta', 'inspiry_property_id_meta_update', 10, 4 );
}

if ( ! function_exists( 'inspiry_gallery_properties_filter' ) ) {
	/**
	 * Add gallery properties filter parameters to given query arguments
	 *
	 * @param array $properties_query_args - Query arguments.
	 *
	 * @return mixed|array - Modified query arguments.
	 */
	function inspiry_gallery_properties_filter( $properties_query_args ) {

		// Get the page ID.
		$page_id = get_the_ID();

		// Taxonomy query.
		$tax_query = array();

		// Meta query.
		$meta_query = array();

		// Number of properties on each page
		$properties_query_args['posts_per_page'] = 6;

		// Give priority to page meta value
		$number_of_properties = get_post_meta( $page_id, 'inspiry_gallery_posts_per_page', true );
		if ( ! empty( $number_of_properties ) ) {
			$properties_query_args['posts_per_page'] = intval( $number_of_properties );
		} else if ( 'ultra' === INSPIRY_DESIGN_VARIATION ) {
			global $gallery_properties;
			$properties_query_args['posts_per_page'] = intval( $gallery_properties );
		}

		// Locations
		$locations = get_post_meta( $page_id, 'inspiry_gallery_properties_locations', false );
		if ( ! empty( $locations ) && is_array( $locations ) ) {
			$tax_query[] = array(
				'taxonomy' => 'property-city',
				'field'    => 'slug',
				'terms'    => $locations,
			);
		}

		// Statuses
		$statuses = get_post_meta( $page_id, 'inspiry_gallery_properties_statuses', false );
		if ( ! empty( $statuses ) && is_array( $statuses ) ) {
			$tax_query[] = array(
				'taxonomy' => 'property-status',
				'field'    => 'slug',
				'terms'    => $statuses,
			);
		}

		// Types
		$types = get_post_meta( $page_id, 'inspiry_gallery_properties_types', false );
		if ( ! empty( $types ) && is_array( $types ) ) {
			$tax_query[] = array(
				'taxonomy' => 'property-type',
				'field'    => 'slug',
				'terms'    => $types,
			);
		}

		// Features
		$features = get_post_meta( $page_id, 'inspiry_gallery_properties_features', false );
		if ( ! empty( $features ) && is_array( $features ) ) {
			$tax_query[] = array(
				'taxonomy' => 'property-feature',
				'field'    => 'slug',
				'terms'    => $features,
			);
		}

		// If more than one taxonomies exist then specify the relation.
		$tax_count = count( $tax_query );
		if ( $tax_count > 1 ) {
			$tax_query['relation'] = 'AND';
		}
		if ( $tax_count > 0 ) {
			$properties_query_args['tax_query'] = $tax_query;
		}

		if ( 'ultra' !== INSPIRY_DESIGN_VARIATION ) {

			// Meta query for gallery.
			$meta_query[] = array(
				'key' => '_thumbnail_id',
			);
			$properties_query_args['meta_query'] = $meta_query;
		}

		return $properties_query_args;
	}

	add_filter( 'inspiry_gallery_properties_filter', 'inspiry_gallery_properties_filter' );
}

if ( ! function_exists( 'inspiry_get_property_agents' ) ) {
	/**
	 * Returns array containing agent/agents IDs related to given or current property ID.
	 *
	 * @param int $property_id
	 *
	 * @return array|bool|mixed
	 */
	function inspiry_get_property_agents( $property_id = 0 ) {

		if ( ! $property_id ) {
			$property_id = get_the_ID();
		}

		if ( $property_id ) {

			$property_agents = get_post_meta( $property_id, 'REAL_HOMES_agents' );

			// Remove invalid ids.
			if ( ! empty( $property_agents ) ) {
				$property_agents = array_filter( $property_agents, function ( $v ) {
					return ( $v > 0 );
				} );
			}

			if ( ! empty( $property_agents ) ) {
				// Remove duplicated ids.
				$property_agents = array_unique( $property_agents );

				// Return valid agents array.
				return $property_agents;
			}

		}

		return false;
	}
}

if ( ! function_exists( 'inspiry_new_function' ) ) {
	/**
	 * Returns comma separated list of property agents names.
	 *
	 * @param int $property_id
	 *
	 * @return bool|string
	 */
	function inspiry_get_property_agents_names( $property_id = 0 ) {
		if ( ! $property_id ) {
			$property_id = get_the_ID();
		}

		$property_agents = inspiry_get_property_agents( $property_id );
		if ( ! empty( $property_agents ) ) {
			$agents_names = array();
			foreach ( $property_agents as $single_agent_id ) {
				if ( 0 < $single_agent_id ) {
					$agents_names[] = get_the_title( $single_agent_id );
				}
			}

			if ( ! empty( $agents_names ) ) {
				return implode( ', ', $agents_names );
			}
		}

		return false;
	}
}

if ( ! function_exists( 'rh_any_text' ) ) {
	/**
	 * Return text string for word 'Any'
	 *
	 * @return string
	 */
	function rh_any_text() {
		if ( function_exists( 'ere_any_text' ) ) {
			return ere_any_text();
		}

		return esc_html__( 'Any', RH_TEXT_DOMAIN );
	}
}

if ( ! function_exists( 'inspiry_prop_detail_login' ) ) {
	/**
	 * Check if login required to vew property detail
	 *
	 * @return bool
	 */
	function inspiry_prop_detail_login() {
		$require_login = get_option( 'inspiry_prop_detail_login', 'no' );

		return ( 'yes' == $require_login && ! is_user_logged_in() );
	}
}

if ( ! function_exists( 'realhomes_filter_assigned_agents_args' ) ) {
	/**
	 * Filter Assigned Agents Query Arguments for single property page as per the WPML Language Selected
	 *
	 * @param array $agent_properties_args - Query arguments.
	 *
	 * @return array
	 * 
	 * @since 3.21.0
	 */
	function realhomes_filter_assigned_agents_args( $agent_properties_args ) {

		// Check if WPML is active
		if ( class_exists( 'SitePress' ) ) {

			$languages = apply_filters( 'wpml_active_languages', NULL, 'orderby=id&order=desc' );

			foreach ( $languages as $language ) {
				$agent[] = apply_filters( 'wpml_object_id',  $agent_properties_args["meta_query"][0]["value"], 'agent', false,  $language["code"] );
			}

			$agent_properties_args = array(
				'suppress_filters' => false,
				'post_type'        => 'property',
				'posts_per_page'   => $agent_properties_args["posts_per_page"],
				'meta_query'       => array(
					array(
						'key'     => 'REAL_HOMES_agents',
						'value'   => implode( ',', $agent ),
						'compare' => 'IN',
					),
				),
				'paged'            => $agent_properties_args["paged"],
			);
		}

		return $agent_properties_args;

	}
	add_filter('inspiry_agent_single_properties', 'realhomes_filter_assigned_agents_args');
}


if ( ! function_exists( 'realhomes_get_property_agents' ) ) {
	/**
	 * The function generates html for password protected property
	 *
	 * @since 4.3.6
	 *
	 * @return bool|null
	 */
	function realhomes_password_protected_post_content() {

		if ( 'classic' === INSPIRY_DESIGN_VARIATION || ! is_singular( 'property' ) ) {
			return false;
		}

		$property_id = get_the_ID();

		if ( 'ultra' === INSPIRY_DESIGN_VARIATION ) {
			?>
            <div class="protected-property-wrap">
				<?php
				if ( ! empty( get_the_post_thumbnail( $property_id ) ) ) {
					$image_url = get_the_post_thumbnail_url( $property_id, 'large' );
				} else {
					$image_url = get_inspiry_image_placeholder_url( 'large' );
				}
				?>
                <div class="rh-property-featured-image" style="background-image: url('<?php echo esc_url( $image_url ); ?>')">
                    <div class="rh-image-overlay"></div>
                    <div class="rh-ultra-content-container">
						<?php echo get_the_password_form(); ?>
                    </div>
                </div>
            </div>
			<?php
		} else {
			?>
            <div class="rh_page rh_page--fullWidth">
				<?php echo get_the_password_form(); ?>
            </div><!-- /.rh_page -->
			<?php
		}
	}
}


if ( ! function_exists( 'realhomes_validate_location_coordinates' ) ) {
	/**
	 * Validate a property's location coordinates.
	 *
	 * This function checks if the provided property location contains valid
	 * latitude and longitude values. Latitude must be between -90 and 90, and
	 * longitude must be between -180 and 180.
	 *
	 * @param array|null $property_location  An associative array containing 'latitude' and 'longitude'.
	 *                                       Example: ['latitude' => 25.276987, 'longitude' => 55.296249]
	 *                                       If the array is null or malformed, the function returns false.
	 *
	 * @return array True if both latitude and longitude are valid; false otherwise.
	 */
	function realhomes_validate_location_coordinates( $property_location ) {
		$validated_coordinates = false;

		if ( ! empty( $property_location ) ) {
			// Handle raw property location string and explode it
			if ( is_string( $property_location ) ) {
				$coordinates = explode( ',', $property_location );
				$latitude    = isset( $coordinates[0] ) ? floatval( $coordinates[0] ) : null;
				$longitude   = isset( $coordinates[1] ) ? floatval( $coordinates[1] ) : null;
			}
            else if ( is_array( $property_location ) ) {
				$latitude  = isset( $property_location['latitude'] ) ? floatval( $property_location['latitude'] ) : null;
				$longitude = isset( $property_location['longitude'] ) ? floatval( $property_location['longitude'] ) : null;
			} else {
				$latitude = $longitude = null;
			}

			// Validate the latitude and longitude
			if ( $latitude !== null && $longitude !== null &&
				$latitude >= -90 && $latitude <= 90 &&
				$longitude >= -180 && $longitude <= 180 ) {
				$validated_coordinates = [
					'latitude'  => $latitude,
					'longitude' => $longitude,
				];
			}
		}

		return $validated_coordinates;
	}
}

if ( ! function_exists( 'realhomes_alter_similar_properties_args' ) ) {
	/**
	 * Alters the arguments for similar properties based on the Customizer setting.
	 *
	 * @since 4.3.9
	 *
	 * @param array $args The original query arguments.
	 *
	 * @return array Modified query arguments.
	 */
	function realhomes_alter_similar_properties_args( $args ) {
		// Get the customizer option for tax relation
		$tax_relation = get_option( 'realhomes_similar_properties_tax_relation', 'or' ); // Default to 'OR'

		// Ensure the tax_query exists and contains more than one taxonomy
		if ( isset( $args['tax_query'] ) && is_array( $args['tax_query'] ) && count( $args['tax_query'] ) > 1 ) {
			$args['tax_query']['relation'] = ( 'and' === $tax_relation ) ? 'AND' : 'OR';
		}

		return $args;
	}

	add_filter( 'inspiry_similar_properties_filter', 'realhomes_alter_similar_properties_args', 99 );
}


if ( ! function_exists( 'realhomes_toggle_property_state' ) ) {
	/**
	 * Processing the request to activate or deactivate the property status
	 * from front-end dashboard
	 *
	 * @since 4.4.0
	 *
	 * @return void
	 */
	function realhomes_toggle_property_state() {
		// Verify AJAX nonce for security
		//check_ajax_referer( 'toggle_property_nonce', 'security' );

		// Get the property ID and state from the request
		$property_id = intval( $_POST['property_id'] );
		$state       = sanitize_text_field( $_POST['state'] );

		if ( $property_id && in_array( $state, [ 'publish', 'draft' ] ) ) {
			// Update the property post status
			$update_result = wp_update_post( [
				'ID'          => $property_id,
				'post_status' => $state
			], true );

			if ( is_wp_error( $update_result ) ) {
				wp_send_json_error( [ 'message' => $update_result->get_error_message() ] );
			} else {
				wp_send_json_success();
			}
		} else {
			wp_send_json_error( [ 'message' => 'Invalid property ID or state.' ] );
		}
	}

	add_action( 'wp_ajax_realhomes_toggle_property_state', 'realhomes_toggle_property_state' );
}


if ( ! function_exists( 'realhomes_is_dual_price_enabled' ) ) {
	/**
	 * Checks if dual price functionality is enabled in RVR mode.
	 *
	 * @since 4.4.4
	 *
	 * @param int $property_id Optional. Property ID to check for sale price meta.
	 *
	 * @return bool True if dual price is enabled; false otherwise.
	 */
	function realhomes_is_dual_price_enabled( $property_id = 0 ) {
		// Dual price option must be enabled globally and RVR mode must be active.
		$is_dual_price_globally_enabled = (
			inspiry_is_rvr_enabled() &&
			'enable' === get_option( 'ere_property_dual_price_status', 'disable' )
		);

		if ( ! $is_dual_price_globally_enabled ) {
			return false;
		}

		// If a property ID is passed, ensure it has a sale price.
		if ( $property_id > 0 ) {
			$dual_price_value = get_post_meta( $property_id, 'REAL_HOMES_property_price_for_sale', true );
			return ! empty( $dual_price_value );
		}

		return true;
	}
}