<?php
/**
 * Template for Single Property Print
 *
 * Provides the print layout for single property pages.
 *
 * @since      4.4.2
 * @package    realhomes
 */

// Return early if the post is password protected.
if ( post_password_required() ) {
	return;
}

// Get current property ID.
$property_id = get_the_ID();

// Determine assets path based on design variation.
$assets_path = ( 'ultra' === INSPIRY_DESIGN_VARIATION ) ? 'assets/ultra' : 'assets/modern';
?>
<!doctype html>
<html <?php language_attributes(); ?>>
<head>
    <meta charset="<?php bloginfo( 'charset' ); ?>">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">

	<?php wp_head(); ?>

    <!-- Include print stylesheet -->
    <link rel="stylesheet" href="<?php echo esc_url( get_theme_file_uri( '/common/css/print-property.min.css' ) ); ?>" type="text/css" media="all">

	<?php
	// If logo filter option is set, apply filter styles for print.
	$logo_filter_for_print = get_option( 'inspiry_logo_filter_for_print', 'none' );
	if ( 'none' !== $logo_filter_for_print ) {
		?>
        <style>
            .print-property-header-logo img {
                -webkit-filter: <?php echo esc_attr( $logo_filter_for_print ); ?>(100%);
                filter: <?php echo esc_attr( $logo_filter_for_print ); ?>(100%);
            }
        </style>
		<?php
	}
	?>
</head>
<body <?php body_class(); ?>>

<div class="print-property-wrap">

    <!-- Print Header: Logo & User Phone -->
    <header class="print-property-header">
        <div class="print-property-header-logo">
			<?php get_template_part( 'assets/modern/partials/header/site-logo' ); ?>
        </div>
        <div class="print-property-header-user-phone">
			<?php get_template_part( 'assets/modern/partials/header/user-phone' ); ?>
        </div>
    </header>

    <div class="property-content">
		<?php
		// Define property sections with display options.
		$sortable_property_sections = array(
			'content'                   => 'true',
			'additional-details'        => 'true',
			'common-note'               => get_option( 'theme_display_common_note', 'true' ),
			'features'                  => 'true',
			'floor-plans'               => 'true',
			'map'                       => get_option( 'theme_display_google_map', 'true' ),
			'walkscore'                 => get_option( 'inspiry_display_walkscore', 'false' ),
			'energy-performance'        => get_option( 'inspiry_display_energy_performance', 'true' ),
			'rvr/location-surroundings' => ( get_option( 'realhomes_rvr_surroundings', 'true' ) || get_option( 'realhomes_rvr_outdoor_features', 'true' ) ) ? 'true' : 'false',
			'rvr/optional-services'     => get_option( 'realhomes_rvr_services', 'true' ),
			'rvr/property-policies'     => get_option( 'realhomes_rvr_property_policies', 'true' ),
			'rvr/guests-accommodation'  => get_option( 'inspiry_guests_accommodation_display', 'true' ),
			'rvr/price-details'         => get_option( 'inspiry_price_details_display', 'true' ),
			'rvr/seasonal-prices'       => get_option( 'inspiry_seasonal_prices_display', 'true' ),
			'agent'                     => get_option( 'theme_display_agent_info', 'true' ) ? 'true' : 'false',
			'rvr/availability-calendar' => 'false',
			'property-views'            => 'false',
			'attachments'               => 'false',
			'schedule-a-tour'           => 'false',
			'children'                  => 'false',
			'video'                     => 'false',
			'virtual-tour'              => 'false',
			'mortgage-calculator'       => 'false',
			'yelp-nearby-places'        => 'false',
		);

		// Allow filtering sections order.
		$sortable_property_sections = apply_filters( 'realhomes_property_default_sections', $sortable_property_sections );
		$property_sections_order    = array_keys( $sortable_property_sections );

		if ( have_posts() ) {
			while ( have_posts() ) {
				the_post();
				?>
                <div class="print-property-head">
                    <div class="print-property-title-wrapper">
						<?php inspiry_property_qr_code(); ?>

                        <div class="print-property-title-wrapper-inner">

                            <h1 class="print-property-title"><?php the_title(); ?></h1>

							<?php
							// Display address if enabled.
							if ( 'true' === get_option( 'inspiry_display_property_address', 'true' ) ) {
								?>
                                <p class="print-property-address">
									<?php echo esc_html( get_post_meta( $property_id, 'REAL_HOMES_property_address', true ) ); ?>
                                </p>
								<?php
							}
							?>
                        </div>
                    </div>

                    <div class="print-property-price">
						<?php
						// Display property status terms.
						$status_terms = get_the_terms( $property_id, 'property-status' );
						if ( ! empty( $status_terms ) ) {
							?>
                            <p class="status">
								<?php
								$status_count = 0;
								foreach ( $status_terms as $term ) {
									if ( $status_count > 0 ) {
										echo ', ';
									}
									echo esc_html( $term->name );
									$status_count++;
								}
								?>
                            </p>
							<?php
						}

						// Display property price.
						if ( function_exists( 'ere_property_price' ) ) {
							?>
                            <p class="price">
								<?php ere_property_price( $property_id, true ); ?>
                            </p>
							<?php
						}
						?>
                    </div>
                </div>

				<?php
				// Display gallery images if enabled, else fallback to featured image.
				if ( 'gallery-images' === get_option( 'realhomes_property_media_in_print', 'gallery-images' ) ) {
					$properties_images = rwmb_meta( 'REAL_HOMES_property_images', 'type=plupload_image&size=post-featured-image', $property_id );
					if ( ! empty( $properties_images ) ) {
						?>
                        <div class="print-property-gallery">
							<?php
							foreach ( $properties_images as $property_image ) {
								echo '<img src="' . esc_url( $property_image['url'] ) . '" alt="' . esc_attr( $property_image['title'] ) . '" />';
							}
							?>
                        </div>
						<?php
					}
				} else if ( has_post_thumbnail() ) {
					$image_id  = get_post_thumbnail_id();
					$image_url = wp_get_attachment_url( $image_id );
					$image_alt = the_title_attribute( [ 'echo' => false ] );
					?>
                    <div id="property-featured-image">
                        <img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $image_alt ); ?>" />
                    </div>
					<?php
				}
				?>

                <main class="print-property-content">
                    <div id="property-content-section-overview" class="property-content-section-overview">
						<?php
						// Property ID and featured label.
						$realhomes_property_id = get_post_meta( $property_id, 'REAL_HOMES_property_id', true );
						$prop_id_field_label   = get_option( 'inspiry_prop_id_field_label' );
						?>
                        <div class="print-property-meta">
                            <div class="print-property-id">
                                <p class="title"><?php echo $prop_id_field_label ? esc_html( $prop_id_field_label ) : esc_html__( 'Property ID', RH_TEXT_DOMAIN ); ?>:</p>
                                <p class="id">
									<?php
									if ( ! empty( $realhomes_property_id ) ) {
										echo esc_html( $realhomes_property_id );
									} else {
										esc_html_e( 'None', RH_TEXT_DOMAIN );
									}
									?>
                                </p>
                            </div>

							<?php
							// Show featured label if property is marked featured.
							if ( '1' === get_post_meta( $property_id, 'REAL_HOMES_featured', true ) ) {
								?><span class="featured-label"><?php realhomes_featured_label(); ?></span><?php
							}
							?>
                        </div>

						<?php
						// Load meta info based on design variation.
						if ( 'ultra' === INSPIRY_DESIGN_VARIATION ) {
							get_template_part( 'assets/ultra/partials/property/single/meta-icons' );

						} else if ( 'classic' === INSPIRY_DESIGN_VARIATION ) {
							?>
                            <div class="classic-property-meta-wrapper">
								<?php get_template_part( 'assets/classic/partials/property/single/metas' ); ?>
                            </div>
							<?php
						} else {
							get_template_part( 'assets/modern/partials/property/single/meta' );
						}
						?>
                    </div>

					<?php
					// Loop through enabled sections and load them.
					if ( ! empty( $property_sections_order ) && is_array( $property_sections_order ) ) {
						foreach ( $property_sections_order as $section ) {
							if ( isset( $sortable_property_sections[ $section ] ) && 'true' === $sortable_property_sections[ $section ] ) {
								get_template_part( $assets_path . '/partials/property/single/' . $section );
							}
						}
					}
					?>
                </main>
				<?php
			}
		}
		?>
    </div>
</div>

<?php wp_footer(); ?>
</body>
</html>