<?php
/**
 * Price related functions.
 *
 * @link       https://realhomes.io/
 * @since      1.0.0
 *
 * @package    easy-real-estate
 * @subpackage easy-real-estate/includes
 */
if ( ! function_exists( 'ere_format_amount' ) ) {
	/**
	 * Return given amount in a configured format.
	 *
	 * @since  1.0.0
	 *
	 * @param string $amount An amount that has to be formatted.
	 *
	 * @return string
	 */
	function ere_format_amount( $amount ) {

		// Return if amount is empty or not a number.
		if ( empty( $amount ) || is_nan( $amount ) ) {
			return '';
		}

		// If RealHomes Currency Switcher plugin is installed and current currency cookie is set.
		if ( function_exists( 'realhomes_currency_switcher_enabled' ) && realhomes_currency_switcher_enabled() ) {
			$formatted_converted_price = realhomes_switch_currency( $amount );

			return apply_filters( 'inspiry_property_converted_price', $formatted_converted_price, $amount );
		} else {
			$currency_sign       = ere_get_currency_sign();
			$decimals            = intval( get_option( 'theme_decimals', '0' ) );
			$decimal_point       = get_option( 'theme_dec_point', '.' );
			$thousands_separator = get_option( 'theme_thousands_sep', ',' );
			$currency_position   = get_option( 'theme_currency_position', 'before' );
			$formatted_price     = number_format( $amount, $decimals, $decimal_point, $thousands_separator );
			$formatted_price     = apply_filters( 'inspiry_property_price', $formatted_price, $amount );

			return ( 'after' == $currency_position ) ? $formatted_price . $currency_sign : $currency_sign . $formatted_price;
		}
	}
}

if ( ! function_exists( 'ere_get_currency_sign' ) ) {
	/**
	 * Get Currency
	 *
	 * @return string
	 */
	function ere_get_currency_sign() {
		return apply_filters( 'inspiry_currency_sign', get_option( 'theme_currency_sign', esc_html__( '$', ERE_TEXT_DOMAIN ) ) );
	}
}

if ( ! function_exists( 'ere_set_currency_sign_space' ) ) {
	/**
	 * Set space before or after currency sign if required
	 *
	 * @since 2.2.5
	 *
	 * @return string
	 */
	function ere_set_currency_sign_space( $sign ) {

		$currency_sign_space = get_option( 'ere_currency_sign_space', 'none' );

		if ( 'before' === $currency_sign_space ) {
			$sign = ' ' . $sign;
		}

		if ( 'after' === $currency_sign_space ) {
			$sign = $sign . ' ';
		}

		return $sign;
	}

	add_filter( 'inspiry_currency_sign', 'ere_set_currency_sign_space' );
}

if ( ! function_exists( 'ere_no_price_text' ) ) {
	/**
	 * Returns text to display in case of empty price
	 *
	 * @return string
	 */
	function ere_no_price_text() {
		return apply_filters( 'inspiry_no_price_text', get_option( 'theme_no_price_text', esc_html__( 'Price On Call', ERE_TEXT_DOMAIN ) ) );
	}
}

if ( ! function_exists( 'ere_property_sale_price' ) ) {
	/**
	 * Returns property current and old price if not empty otherwise current price.
	 *
	 * @since  0.6.0
	 *
	 * @param string $current_price current price.
	 * @param string $old_price     Old price.
	 *
	 * @return string
	 */
	function ere_property_sale_price( $current_price, $old_price ) {

		if ( ! empty( $current_price ) && ! empty( $old_price ) ) {
			return sprintf( '<span class="property-price-wrapper"><ins class="property-current-price">%s</ins> <del class="property-old-price">%s</del></span>', $current_price, $old_price );
		}

		return $current_price;
	}
}


if ( ! function_exists( 'ere_get_property_old_price' ) ) {
	/**
	 * Get the property's old price by meta key.
	 *
	 * @since 0.6.0 Updated in 2.3.4 to support custom meta key for dual price.
	 *
	 * @param int    $property_id Property ID. Defaults to current post.
	 * @param string $meta_key    Optional meta key. Defaults to standard old price key.
	 *
	 * @return string Formatted price or empty string if none.
	 */
	function ere_get_property_old_price( $property_id = 0, $meta_key = 'REAL_HOMES_property_old_price' ) {

		if ( empty( $property_id ) ) {
			$property_id = get_the_ID();
		}

		$amount = floatval( get_post_meta( $property_id, $meta_key, true ) );

		return ere_format_amount( $amount );
	}
}


if ( ! function_exists( 'ere_get_property_price' ) ) {
	/**
	 * Returns property price in configured format.
	 *
	 * @param int  $property_id Property ID to get price for.
	 * @param bool $show_old_price
	 * @param bool $markup      Property price with html
	 * @param bool $show_dual   Show dual price or not
	 *
	 * @return string
	 */
	function ere_get_property_price( $property_id = 0, $show_old_price = false, $markup = false, $show_dual = true ) {

		// Set property ID if it's not given.
		if ( empty( $property_id ) ) {
			$property_id = get_the_ID();
		}

		// Dual price override.
		if ( ere_is_dual_price_enabled() && $show_dual ) {
			return ere_get_dual_property_price_output( $property_id, $markup, $show_old_price );
		}

		// Get property price.
		$amount = floatval( get_post_meta( $property_id, 'REAL_HOMES_property_price', true ) );

		// Return no price text if price is empty.
		if ( empty( $amount ) || is_nan( $amount ) ) {
			return ere_no_price_text();
		}

		$price = ere_format_amount( $amount );

		/**
		 * Filter condition to show property current and old price for all or specific properties.
		 *
		 * @since 0.6.0
		 *
		 * @param bool $show_old_price Current boolean. False by default.
		 * @param int  $property_id    Current property id.
		 */
		if ( true === apply_filters( 'ere_show_properties_old_price', $show_old_price, $property_id ) ) {
			$price = ere_property_sale_price( $price, ere_get_property_old_price( $property_id ) );
		}

		// Get price prefix & postfix.
		$price_prefix  = ere_get_price_prefix( $property_id );
		$price_postfix = ere_get_price_postfix( $property_id );

		if ( true == $markup ) {
			$price_markup = '';

			if ( ! empty( $price_prefix ) ) {
				$price_markup .= '<span class="ere-price-prefix">' . $price_prefix . '</span> ';
			}

			$price_markup .= '<span class="ere-price-display">' . $price . '</span>';

			if ( ! empty( $price_postfix ) ) {
				$price_markup .= ' <span class="ere-price-slash">/</span> <span class="ere-price-postfix">' . $price_postfix . '</span>';
			}

			return $price_markup;
		}

		return $price_prefix . ' ' . $price . ' ' . $price_postfix;
	}
}

if ( ! function_exists( 'ere_get_property_price_plain' ) ) {
	/**
	 * Returns property price in configured format without prefix and postfix.
	 *
	 * @param int  $property_id Property ID to get price for.
	 * @param bool $show_old_price
	 *
	 * @return string
	 */
	function ere_get_property_price_plain( $property_id = 0, $show_old_price = false ) {

		// Set property ID if it's not given.
		if ( empty( $property_id ) ) {
			$property_id = get_the_ID();
		}

		// Get property price.
		$amount = floatval( get_post_meta( $property_id, 'REAL_HOMES_property_price', true ) );

		// Return no price text if price is empty.
		if ( empty( $amount ) || is_nan( $amount ) ) {
			return ere_no_price_text();
		}

		$price = ere_format_amount( $amount );

		/**
		 * Filter condition to show property current and old price for all or specific properties.
		 *
		 * @since 0.6.0
		 *
		 * @param bool $show_old_price Current boolean. False by default.
		 * @param int  $property_id    Current property id.
		 */
		if ( true === apply_filters( 'ere_show_properties_old_price', $show_old_price, $property_id ) ) {
			$price = ere_property_sale_price( $price, ere_get_property_old_price() );
		}

		return $price;
	}
}

if ( ! function_exists( 'ere_property_price' ) ) {
	/**
	 * Output property price.
	 *
	 * @param int  $property_id
	 * @param bool $show_old_price
	 */
	function ere_property_price( $property_id = 0, $show_old_price = false, $markup = false, $show_dual = true ) {
		echo ere_get_property_price( $property_id, $show_old_price, $markup, $show_dual );
	}
}

if ( ! function_exists( 'ere_get_property_floor_price' ) ) {
	/**
	 * Returns floor price in configured format
	 *
	 * @param array $floor_plan An array of floor plan information.
	 *
	 * @return string
	 */
	function ere_get_property_floor_price( $floor_plan ) {

		$amount = $floor_plan['inspiry_floor_plan_price'] ?? '';

		// If the result is a non-numeric string (e.g., "Call for price"), return as is
		if ( ! is_numeric( $amount ) ) {
			return esc_html( $amount );
		}

		// If it's numeric but zero or invalid, show "no price" text
		if ( empty( $amount ) || is_nan( $amount ) ) {
			return ere_no_price_text();
		}

		$price = ere_format_amount( $amount );

		// Retrieve price postfix (optional)
		$price_postfix = isset( $floor_plan['inspiry_floor_plan_price_postfix'] ) ? esc_html( $floor_plan['inspiry_floor_plan_price_postfix'] ) : '';

		// Generate markup
		if ( 'ultra' === INSPIRY_DESIGN_VARIATION ) {
			$price_markup = '<span class="floor-price-value">' . $price . '</span> ' . '<span class="floor-price-post-fix">' . $price_postfix . '</span>';
		} else {
			$price_markup = '<span class="floor-price-value">' . $price . '</span> ' . $price_postfix;
		}

		return $price_markup;
	}
}

if ( ! function_exists( 'ere_property_floor_price' ) ) {
	/**
	 * Display floor plan price.
	 *
	 * @param array $floor_plan An array of floor plan information.
	 *
	 * @return void
	 */
	function ere_property_floor_price( $floor_plan ) {
		echo ere_get_property_floor_price( $floor_plan );
	}
}


if ( ! function_exists( 'ere_get_dual_property_price_output' ) ) {
	/**
	 * Returns dual price output (rent and sale) with or without markup.
	 *
	 * @since 2.3.4
	 *
	 * @param int  $property_id
	 * @param bool $markup
	 * @param bool $show_old_price
	 *
	 * @return string
	 */
	function ere_get_dual_property_price_output( $property_id, $markup = false, $show_old_price = false ) {

		$rent_price = floatval( get_post_meta( $property_id, 'REAL_HOMES_property_price', true ) );
		$sale_price = floatval( get_post_meta( $property_id, 'REAL_HOMES_property_price_for_sale', true ) );

		if ( empty( $sale_price ) && empty( $rent_price ) ) {
			return ere_no_price_text();
		}

		$rent_price_formatted = $rent_price ? ere_format_amount( $rent_price ) : '';
		$sale_price_formatted = $sale_price ? ere_format_amount( $sale_price ) : '';
		$show_property_old_price = true === apply_filters( 'ere_show_properties_old_price', $show_old_price, $property_id );

		// Apply old price if enabled
		if ( $rent_price_formatted && $show_property_old_price ) {
			$rent_price_formatted = ere_property_sale_price(
				$rent_price_formatted,
				ere_get_property_old_price( $property_id, 'REAL_HOMES_property_old_price' )
			);
		}

		if ( $sale_price_formatted && $show_property_old_price ) {
			$sale_price_formatted = ere_property_sale_price(
				$sale_price_formatted,
				ere_get_property_old_price( $property_id, 'REAL_HOMES_property_old_price_for_sale' )
			);
		}

		$rent_prefix  = ere_get_price_prefix( $property_id, 'rent' );
		$rent_postfix = ere_get_price_postfix( $property_id, 'rent' );

		$sale_prefix  = ere_get_price_prefix( $property_id, 'sale' );
		$sale_postfix = ere_get_price_postfix( $property_id, 'sale' );

		$output = '';

		if ( $markup ) {
			// Markup output
			if ( $rent_price_formatted ) {
				$output .= '<span class="ere-price-wrapper rent">';
				$output .= '<span class="ere-price-block ere-rent-price">';
				if ( $rent_prefix ) {
					$output .= '<span class="ere-price-prefix">' . esc_html( $rent_prefix ) . '</span> ';
				}
				$output .= '<span class="ere-price-display">' . $rent_price_formatted . '</span>';
				if ( $rent_postfix ) {
					$output .= ' <span class="ere-price-slash">/</span> <span class="ere-price-postfix">' . esc_html( $rent_postfix ) . '</span>';
				}
				$output .= '</span></span>';
			}

			if ( $sale_price_formatted ) {
				$output .= '<span class="ere-price-wrapper sale">';
				$output .= '<span class="ere-price-block ere-sale-price">';
				if ( $sale_prefix ) {
					$output .= '<span class="ere-price-prefix">' . esc_html( $sale_prefix ) . '</span> ';
				}
				$output .= '<span class="ere-price-display">' . $sale_price_formatted . '</span>';
				if ( $sale_postfix ) {
					$output .= ' <span class="ere-price-slash">/</span> <span class="ere-price-postfix">' . esc_html( $sale_postfix ) . '</span>';
				}
				$output .= '</span></span>';
			}
		} else {
			// Raw but wrapped output
			if ( $rent_price_formatted ) {
				$output .= '<span class="ere-price-wrapper rent">';
				$output .= wp_kses_post( trim( $rent_prefix . ' ' . $rent_price_formatted . ' ' . $rent_postfix ) );
				$output .= '</span>';
			}

			if ( $sale_price_formatted ) {
				$output .= '<span class="ere-price-wrapper sale">';
				$output .= wp_kses_post( trim( $sale_prefix . ' ' . $sale_price_formatted . ' ' . $sale_postfix ) );
				$output .= '</span>';
			}
		}

		return $output;
	}
}



if ( ! function_exists( 'ere_get_price_prefix' ) ) {
	/**
	 * Get property price prefix.
	 *
	 * @since 2.3.4
	 *
	 * @param int    $property_id
	 * @param string $context 'rent' or 'sale'
	 *
	 * @return string
	 */
	function ere_get_price_prefix( $property_id, $context = 'rent' ) {

		if ( ere_is_dual_price_enabled( $property_id ) && 'sale' === $context ) {
			return get_post_meta( $property_id, 'REAL_HOMES_property_price_for_sale_prefix', true );
		}

		return get_post_meta( $property_id, 'REAL_HOMES_property_price_prefix', true );
	}
}


if ( ! function_exists( 'ere_get_price_postfix' ) ) {
	/**
	 * Get property price postfix.
	 *
	 * @since 2.3.4
	 *
	 * @param int    $property_id
	 * @param string $context 'rent' or 'sale'
	 *
	 * @return string
	 */
	function ere_get_price_postfix( $property_id, $context = 'rent' ) {

		if ( ere_is_dual_price_enabled( $property_id ) && 'sale' === $context ) {
			return get_post_meta( $property_id, 'REAL_HOMES_property_price_for_sale_postfix', true );
		}

		return get_post_meta( $property_id, 'REAL_HOMES_property_price_postfix', true );
	}
}
