( function ( $ ) {
    "use strict";

    $( document ).ready( function () {

        // Get Canvas tag to use it for Property Views Graph
        const graphDiv = document.getElementById( 'property-views-graph' );

        // Check if Canvas tag and Ajax request data is available before making a request.
        if ( graphDiv && typeof property_analytics !== 'undefined' ) {
            const bgColor          = graphDiv.getAttribute( 'data-bg-color' );
            const borderColor      = graphDiv.getAttribute( 'data-border-color' );
            const propertyID       = graphDiv.getAttribute( 'data-property-id' );
            const isUserLoggedIn   = Boolean( property_analytics.is_user_logged_in || false );
            const propertyViewsKey = 'propertyViews';

            let setBorderColor = property_analytics.border_color;
            let setBgColor     = property_analytics.background_color;
            let setPropertyID  = property_analytics.property_id;

            if ( borderColor && '' !== borderColor ) {
                setBorderColor = borderColor;
            }
            if ( bgColor && '' !== bgColor ) {
                setBgColor = bgColor;
            }
            if ( propertyID && '' !== propertyID ) {
                setPropertyID = propertyID;
            }

            $.ajax( {
                type     : 'post',
                dataType : 'json',
                url      : property_analytics.ajax_url,
                data     : {
                    action      : 'inspiry_property_views',
                    property_id : setPropertyID,
                    nonce       : property_analytics.ajax_nonce
                },
                success  : function ( response ) {
                    try {
                        if ( ! response || ! response.success || ! response.data ) {
                            throw 'Invalid or missing response data.';
                        }

                        const ctx  = graphDiv.getContext( '2d' );
                        const data = response.data;

                        // Create the chart
                        new Chart( ctx, {
                            type    : property_analytics.chart_type,
                            data    : {
                                labels   : data.dates,
                                datasets : [
                                    {
                                        label           : property_analytics.data_label,
                                        borderColor     : setBorderColor,
                                        backgroundColor : setBgColor,
                                        data            : data.views
                                    }
                                ]
                            },
                            options : {
                                scales : {
                                    y : {
                                        ticks : {
                                            stepSize    : 1,
                                            beginAtZero : true
                                        }
                                    }
                                }
                            }
                        } );

                        // Save RID to localStorage if the user is not logged in
                        if ( ! isUserLoggedIn ) {
                            let storedRIDs = JSON.parse( localStorage.getItem( propertyViewsKey ) ) || [];
                            if ( data.rid ) {
                                storedRIDs.push( data.rid );
                                localStorage.setItem( propertyViewsKey, JSON.stringify( storedRIDs ) );
                            } else {
                                console.warn( 'RID not found in the response data.' );
                            }
                        }

                    } catch ( error ) {
                        console.error( 'Property Analytics chart couldn\'t be drawn:', error );
                    }
                },
                error    : function ( xhr, status, error ) {
                    console.error( 'AJAX Error:', status, error );
                }
            } );

            // Check if user is logged in after login and send property views to server
            if ( isUserLoggedIn ) {
                const storedPropertyViews = JSON.parse( localStorage.getItem( 'propertyViews' ) ) || [];

                if ( storedPropertyViews.length > 0 ) {
                    sendPropertyViewsToServer( storedPropertyViews );
                }
            }

            // Function to send RIDs to the server
            function sendPropertyViewsToServer( rids ) {
                if ( Array.isArray( rids ) && rids.length > 0 ) {
                    $.ajax( {
                        type    : "POST",
                        url     : property_analytics.ajax_url,
                        data    : {
                            action  : "ere_update_user_id_in_property_views",
                            nonce   : property_analytics.ajax_nonce,
                            rids    : rids,
                            user_id : property_analytics.current_user_id
                        },
                        success : function ( response ) {
                            // Log the response for debugging
                            if ( response.success ) {
                                // Clear local storage after success
                                localStorage.removeItem( propertyViewsKey );
                            } else {
                                console.error( "Failed to update property views:", response.message || 'No error message provided.' );
                            }
                        },
                        error   : function ( xhr, status, error ) {
                            console.error( "AJAX error while updating property views:", status, error );
                        }
                    } );
                } else {
                    console.warn( "No stored property views to send to server." );
                }
            }
        }
    } );
} )( jQuery );