<?php
/**
 * Bookings and Appointment Plugin for WooCommerce.
 *
 * Multiple Days & Time
 *
 * @author      Tyche Softwares
 * @package     BKAP/Frontend/Multiple-Days-Time
 * @category    Classes
 * @since       7.9.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Class BKAP_Multiple_Days_Time
 */
class BKAP_Multiple_Days_Time {

	/**
	 * Initialize hooks
	 */
	public function __construct() {

		add_action( 'wp_ajax_bkap_get_available_start_times', array( $this, 'bkap_get_available_start_times_callback' ) );
		add_action( 'wp_ajax_nopriv_bkap_get_available_start_times', array( $this, 'bkap_get_available_start_times_callback' ) );

		add_action( 'wp_ajax_bkap_get_available_end_times', array( $this, 'bkap_get_available_end_times_callback' ) );
		add_action( 'wp_ajax_nopriv_bkap_get_available_end_times', array( $this, 'bkap_get_available_end_times_callback' ) );

		add_action( 'wp_ajax_bkap_calculate_availability_and_price', array( $this, 'bkap_calculate_availability_and_price' ) );
		add_action( 'wp_ajax_nopriv_bkap_calculate_availability_and_price', array( $this, 'bkap_calculate_availability_and_price' ) );
	}

	/**
	 * AJAX callback to calculate booking price based on selected date/time range
	 */
	public function bkap_calculate_availability_and_price() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'bkap_booking_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed. Please refresh the page and try again.', 'bkap' ) ) );
		}

		$product_id   = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
		$variation_id = isset( $_POST['variation_id'] ) ? absint( $_POST['variation_id'] ) : 0;

		$product      = wc_get_product( $product_id );
		$product_type = $product->get_type();

		if ( 'variable' === $product_type ) {
			if ( isset( $variation_id ) && ( $variation_id == '' || $variation_id == 0 ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Please choose product options&hellip;', 'woocommerce' ),
					)
				);
			}
		}
		$start_date = isset( $_POST['start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['start_date'] ) ) : '';
		$end_date   = isset( $_POST['end_date'] ) ? sanitize_text_field( wp_unslash( $_POST['end_date'] ) ) : '';
		$start_time = isset( $_POST['start_time'] ) ? sanitize_text_field( wp_unslash( $_POST['start_time'] ) ) : '';
		$end_time   = isset( $_POST['end_time'] ) ? sanitize_text_field( wp_unslash( $_POST['end_time'] ) ) : '';
		$quantity   = isset( $_POST['quantity'] ) ? (int) sanitize_text_field( wp_unslash( $_POST['quantity'] ) ) : 1;

		$resource_id              = isset( $_POST['resource_id'] ) && 0 != $_POST['resource_id'] ? sanitize_text_field( wp_unslash( $_POST['resource_id'] ) ) : '';
		$resource_bookings_placed = isset( $_POST['resource_bookings_placed'] ) ? bkap_common::frontend_json_decode( $_POST['resource_bookings_placed'] ) : ''; // phpcs:ignore
		$person_data              = isset( $_POST['person_ids'] ) ? $_POST['person_ids'] : array(); // phpcs:ignore
		$total_person             = isset( $_POST['total_person'] ) ? sanitize_text_field( wp_unslash( $_POST['total_person'] ) ) : 0;

		if ( ! $product_id || ! $start_date || ! $end_date ) {
			wp_send_json_error(
				array(
					'message' => __( 'Invalid booking data provided.', 'woocommerce-booking' ),
				)
			);
		}

		$_product     = wc_get_product( $product_id );
		$product_type = $_product->get_type();
		if ( 'variation' == $product_type ) {
			$product_type = 'variable';
		}

		$booking_settings = bkap_setting( $product_id );

		$availability_status = '';
		$availability_count  = 0;

		$is_full_range = isset( $booking_settings['booking_multiple_days_time_block_type'] ) && 'full_range' === $booking_settings['booking_multiple_days_time_block_type'] ? true : false;

		// CRITICAL: Check complete range availability.
		if ( ! empty( $start_date ) && ! empty( $start_time ) && ! empty( $end_date ) && ! empty( $end_time ) ) {

			$global_settings = bkap_global_setting();
			// Calculate rental duration.
			$start_timestamp = strtotime( $start_date );
			$end_timestamp   = strtotime( $end_date );
			$days            = ( $end_timestamp - $start_timestamp ) / 86400;

			if ( ! $is_full_range ) {

				$start_time_explode = explode( ' - ', $start_time );
				$end_time_explode   = explode( ' - ', $end_time );
				$start_time_from    = $start_time_explode[0];
				$start_time_to      = $start_time_explode[1];
				$end_time_from      = $end_time_explode[0];
				$end_time_to        = $end_time_explode[1];
				$availability       = bkap_get_selected_slots_timeslot_availability(
					$product_id,
					$start_date,
					$start_time_from,
					$start_time_to,
					$end_date,
					$end_time_from,
					$end_time_to,
					$quantity
				);

				if ( isset( $availability['overall']['available'] ) ) {
					if ( $availability['overall']['qty_available'] == 'unlimited' ) {
						/* Translators: Available Booking Quantity */
						$message = apply_filters( 'bkap_availability_message_multiple_days_time', sprintf( __( 'Unlimited booking is available for the selected booking details.', 'woocommerce-booking' ), $availability['overall']['qty_available'] ), $product_id, $availability['overall']['qty_available'], $start_date, $start_time, $end_date, $end_time );
					} elseif ( $availability['overall']['qty_available'] == 1 ) {
						/* Translators: Available Booking Quantity */
						$message = apply_filters( 'bkap_availability_message_multiple_days_time', sprintf( __( '%s booking is available for the selected booking details.', 'woocommerce-booking' ), $availability['overall']['qty_available'] ), $product_id, $availability['overall']['qty_available'], $start_date, $start_time, $end_date, $end_time );
					} else {
						/* Translators: Available Booking Quantity */
						$message = apply_filters( 'bkap_availability_message_multiple_days_time', sprintf( __( '%s bookings are available for the selected booking details.', 'woocommerce-booking' ), $availability['overall']['qty_available'] ), $product_id, $availability['overall']['qty_available'], $start_date, $start_time, $end_date, $end_time );
					}
					$availability_count  = $availability['overall']['qty_available'];
					$availability_status = 'success';
				} else {
					$message             = $availability['message'];
					$availability_status = 'error';
				}
			} else {

				$start_timestamp = strtotime( $start_date );
				$end_timestamp   = strtotime( $end_date );

				$diff_seconds = $end_timestamp - $start_timestamp;
				$days         = $diff_seconds / 86400;
				$total_days   = ceil( $days ) + 1;

				$availability = bkap_get_multiple_days_time_availability(
					$product_id,
					gmdate( 'Y-m-d', strtotime( $start_date ) ),
					$total_days
				);

				// Check availability across the ENTIRE range.
				$range_check = bkap_check_range_availability(
					$availability,
					$start_date,
					$start_time,
					$end_date,
					$end_time,
					$quantity
				);

				if ( $range_check['available'] ) {
					if ( $range_check['min_available'] == 1 ) {
						/* Translators: Available Booking Quantity */
						$message = apply_filters( 'bkap_availability_message_multiple_days_time', sprintf( __( '%s booking is available for the selected booking details.', 'woocommerce-booking' ), $range_check['min_available'] ), $product_id, $range_check['min_available'], $start_date, $start_time, $end_date, $end_time );
					} elseif ( $range_check['min_available'] === 'unlimited' ) {
						/* Translators: Available Booking Quantity */
						$message = apply_filters( 'bkap_availability_message_multiple_days_time', sprintf( __( 'Unlimited booking is available for the selected booking details.', 'woocommerce-booking' ), $range_check['min_available'] ), $product_id, $range_check['min_available'], $start_date, $start_time, $end_date, $end_time );
					} else {
						/* Translators: Available Booking Quantity */
						$message = apply_filters( 'bkap_availability_message_multiple_days_time', sprintf( __( '%s bookings are available for the selected booking details.', 'woocommerce-booking' ), $range_check['min_available'] ), $product_id, $range_check['min_available'], $start_date, $start_time, $end_date, $end_time );
					}
					$availability_count  = $range_check['min_available'];
					$availability_status = 'success';
				} else {
					$message             = $range_check['message'];
					$availability_status = 'error';
				}
			}
		}

		$duration_message = '';

		/* Price Calculation */
		$total_price = 0;
		if ( $is_full_range ) {

			/**
			 * Convert date & time to timestamps for accurate calculations
			 */
			$start_datetime_str = trim( $start_date . ' ' . $start_time );
			$end_datetime_str   = trim( $end_date . ' ' . $end_time );
			$start_timestamp    = strtotime( $start_datetime_str );
			$end_timestamp      = strtotime( $end_datetime_str );

			if ( ! $start_timestamp || ! $end_timestamp || $end_timestamp <= $start_timestamp ) {
				wp_send_json_error( array( 'message' => __( 'Invalid date or time range.', 'woocommerce-booking' ) ) );
			}

			$range_price    = $booking_settings['booking_multiple_days_time_range_price']; // set price
			$range_price_by = $booking_settings['booking_multiple_days_time_range_price_by']; // unit

			$diff_seconds = $end_timestamp - $start_timestamp;
			$hours        = $diff_seconds / 3600;
			$days         = $diff_seconds / 86400;

			if ( '' === $range_price ) {
				$range_price = bkap_common::bkap_get_price( $product_id, $variation_id, $product_type );
			}

			$booking_duration_label = apply_filters( 'bkap_booking_duration_lable', __( 'Booking duration:', 'woocommerce-booking' ) );

			if ( 'hours' === $range_price_by ) {
				$hours = ceil( $hours );

				$hour_text  = __( 'Hour', 'woocommerce-booking' );
				$hours_text = __( 'Hours', 'woocommerce-booking' );

				if ( $hours == 1 ) {
					$duration_message = sprintf( '%s %s %s', $booking_duration_label, $hours, $hour_text );
				} else {
					$duration_message = sprintf( '%s %s %s', $booking_duration_label, $hours, $hours_text );
				}
				$duration_message = apply_filters( 'bkap_range_price_hour_duration_message', $duration_message, $hours, $range_price );
				$total_price      = round( $hours * (float) $range_price, 2 );
			} else {
				$total_days = ceil( $days );
				$day_text   = __( 'Day', 'woocommerce-booking' );
				$days_text  = __( 'Days', 'woocommerce-booking' );

				if ( $total_days == 1 ) {
					$duration_message = sprintf( '%s %s %s', $booking_duration_label, $total_days, $day_text );
				} else {
					$duration_message = sprintf( '%s %s %s', $booking_duration_label, $total_days, $days_text );
				}
				$duration_message = apply_filters( 'bkap_range_price_day_duration_message', $duration_message, $total_days, $range_price );
				$total_price      = round( $total_days * (float) $range_price, 2 );
			}
		} else {

			$timeslot_config = isset( $booking_settings['booking_multiple_days_time_data'] ) 
				? $booking_settings['booking_multiple_days_time_data'] 
				: array();

			$start_time_explode = explode( ' - ', $start_time );
			$start_price        = self::bkap_get_multiple_days_time_price_by_slot( $timeslot_config, $start_date, $start_time_explode[0], $start_time_explode[1], 'range' );
			if ( '' === $start_price ) {
				$start_price = bkap_common::bkap_get_price( $product_id, $variation_id, $product_type );
			}

			$end_time_explode = explode( ' - ', $end_time );
			$end_price        = self::bkap_get_multiple_days_time_price_by_slot( $timeslot_config, $end_date, $end_time_explode[0], $end_time_explode[1], 'range' );
			if ( '' === $end_price ) {
				$end_price = bkap_common::bkap_get_price( $product_id, $variation_id, $product_type );
			}

			$start_price_message = apply_filters( 'bkap_booking_start_time_price_label', __( 'Booking Start Time Price: ', 'woocommerce-booking' ) ) . wc_price( $start_price * $quantity );
			$end_price_message   = apply_filters( 'bkap_booking_end_time_price_label', __( 'Booking End Time Price: ', 'woocommerce-booking' ) ) . wc_price( $end_price * $quantity );
			$duration_message    = $start_price_message . '</br>' . $end_price_message;

			$hidden_start_time_price = '<input name="bkap_start_time_price" id="bkap_start_time_price" type="hidden" value="' . $start_price * $quantity . '">';
			$hidden_end_time_price   = '<input name="bkap_end_time_price" id="bkap_end_time_price" type="hidden" value="' . $end_price * $quantity . '">';

			$duration_message .= $hidden_start_time_price . $hidden_end_time_price;

			$total_price = $start_price + $end_price;
		}

		$resource_price = 0;
		// Calculate resource price.
		if ( '' !== $resource_id ) {
			$resource_id = explode( ',', $resource_id );
			if ( count( $resource_id ) > 0 ) {
				foreach ( $resource_id as $id ) {
					$resource = new BKAP_Product_Resource( $id, $product_id );
					$_price   = $resource->get_base_cost();

					if ( isset( $global_settings->resource_price_per_day ) && 'on' === $global_settings->resource_price_per_day ) {
						$unit = 2; // 2 because slot booking will have two timeslots.
						if ( $is_full_range ) {
							$unit = 'hours' === $range_price_by ? $hours : $total_days;
						}
						$_price = $_price * $unit;
					}
					$resource_price += $_price;
				}
			}
		}

		/* Person Price Calculations */
		$person_price        = 0;
		$person_base_price   = 0;
		$person_block_price  = 0;
		$person_total        = 0;
		$consider_person_qty = false;
		if ( count( $person_data ) > 0 ) {

			if ( isset( $booking_settings['bkap_price_per_person'] ) && 'on' === $booking_settings['bkap_price_per_person'] ) {
				$consider_person_qty = true;
			}

			$person_types = $booking_settings['bkap_person_data'];
			foreach ( $person_data as $p_id => $p_value ) {
				if ( 1 === absint( $p_value['person_id'] ) ) {
					$person_total = absint( $p_value['person_val'] );
				} elseif ( 'on' === $booking_settings['bkap_person_type'] ) {
					$person_total += absint( $p_value['person_val'] );
					if ( absint( $p_value['person_val'] ) > 0 ) {
						if ( $consider_person_qty ) {
							$person_base_price  = (float) $person_base_price + ( (float) $person_types[ $p_value['person_id'] ]['base_cost'] ); // 100, 50
							$person_block_price = (float) $person_block_price + ( (float) $person_types[ $p_value['person_id'] ]['block_cost'] * absint( $p_value['person_val'] ) ); // 50, 10, 150
						} else {
							$person_base_price  = (float) $person_base_price + (float) $person_types[ $p_value['person_id'] ]['base_cost'];
							$person_block_price = (float) $person_block_price + (float) $person_types[ $p_value['person_id'] ]['block_cost'];
						}
					}
				}
			}

			$person_price = $person_base_price + $person_block_price;
			$person_price = apply_filters( 'bkap_booking_person_price_calculations', $person_price, $person_base_price, $person_block_price );
		}

		if ( ! $is_full_range ) {
			if ( $resource_price > 0 && apply_filters( 'bkap_show_resource_price_detail', true ) ) {
				$duration_message = $duration_message . '</br>' . apply_filters( 'bkap_booking_resource_price_label', __( 'Resource Price: ', 'woocommerce-booking' ) ) . wc_price( $resource_price * $quantity );
			}

			$person_price = $person_price * 2;
			if ( $person_price > 0 && apply_filters( 'bkap_show_person_price_detail', true ) ) {
				$duration_message = $duration_message . '</br>' . apply_filters( 'bkap_booking_person_price_label', __( 'Person Price: ', 'woocommerce-booking' ) ) . wc_price( $person_price * $quantity );
			}
		}

		$total_price = $total_price + $resource_price + $person_price;
		$total_price = $total_price * $quantity;

		$wc_price_args   = bkap_common::get_currency_args();
		$formatted_price = wc_price( $total_price, $wc_price_args );

		$display_price = get_option( 'book_price-label' ) . ' ' . $formatted_price;

		$price_status        = true;
		$availability_status = 'success' === $availability_status ? true : false;
		$response            = array(
			'price_data'        => array(
				'success'                => $price_status,
				'total_price_calculated' => $total_price,
				'bkap_price_charged'     => $total_price,
				'bkap_price'             => $display_price,
			),
			'availability_data' => array(
				'success'      => $availability_status,
				'availability' => $availability_count,
				'duration'     => '' !== $duration_message ? $duration_message . '</br>' : '',
				'message'      => $message,
			),
		);

		if ( ! $price_status || ! $availability_status ) {
			wp_send_json_error( $response );
		} else {
			wp_send_json_success( $response );
		}
	}

	/**
	 * Get the price for a multiple-days time slot.
	 *
	 * @param array  $timeconfig  Full time configuration array (from booking settings).
	 * @param string $date        Date string (any format parseable by strtotime()).
	 * @param string $from_time   Start time (e.g. '09:00').
	 * @param string $to_time     End time (e.g. '10:00').
	 * @param string $type        Pricing type: 'range' (block full date/time range)
	 *                            or 'slot' (block only selected slots).
	 * @return float Price for the slot or range.
	 */
	public static function bkap_get_multiple_days_time_price_by_slot( $timeconfig, $date, $from_time, $to_time, $type = '' ) {

		$price = '';
		if ( empty( $date ) || empty( $timeconfig ) || empty( $from_time ) ) {
			return $price;
		}

		$weekday        = 'booking_weekday_' . gmdate( 'w', strtotime( $date ) );
		$weekday_config = isset( $timeconfig[ $weekday ] ) && is_array( $timeconfig[ $weekday ] ) && ! empty( $timeconfig[ $weekday ] );
		$date_config    = isset( $timeconfig[ $date ] ) && is_array( $timeconfig[ $date ] ) && ! empty( $timeconfig[ $date ] );

		if ( ! $weekday_config && ! $date_config ) {
			return $price;
		}

		$slots = $timeconfig[ $weekday ];

		foreach ( $slots as $slot ) {

			$slot_from  = isset( $slot['from'] ) ? trim( $slot['from'] ) : '';
			$slot_to    = isset( $slot['to'] ) ? trim( $slot['to'] ) : '';
			$apply_to   = isset( $slot['apply_to'] ) ? trim( $slot['apply_to'] ) : 'both';
			$slot_price = isset( $slot['price'] ) && $slot['price'] !== '' ? (float) $slot['price'] : '';

			if ( 'range' === $type ) {
				if ( $slot_from === $from_time && $slot_to === $to_time ) {
					$price = $slot_price;
					break;
				}
			} elseif ( $slot_from === $from_time ) {
				$price = $slot_price;
				break;
			}
		}

		return '' !== $price ? (float) $price : '';
	}

	/**
	 * AJAX callback to get available start times for a specific date
	 */
	public function bkap_get_available_start_times_callback() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'bkap_booking_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed. Please refresh the page and try again.', 'bkap' ) ) );
		}
		$calendar_date = isset( $_POST['date'] ) ? sanitize_text_field( wp_unslash( $_POST['date'] ) ) : '';
		$start_date    = isset( $_POST['start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['start_date'] ) ) : '';
		$product_id    = isset( $_POST['product_id'] ) ? intval( $_POST['product_id'] ) : 0;
		$current_time  = current_time( 'timestamp' );

		$times = bkap_get_available_times_ajax( $product_id, $start_date, 'start', gmdate( 'j-n-Y', $current_time ), gmdate( 'H:i', $current_time ) );

		$booking_settings = bkap_setting( $product_id );
		$block_type       = isset( $booking_settings['booking_multiple_days_time_block_type'] ) ? $booking_settings['booking_multiple_days_time_block_type'] : 'full_range';
		$is_full_range    = 'full_range' === $block_type ? true : false;

		if ( empty( $times ) ) {
			wp_send_json_error(
				array(
					'times'   => array(),
					/* Translators: Start Date */
					'message' => sprintf( __( 'No timeslots available for %s', 'woocommerce-booking' ), $calendar_date ),
				)
			);
			return;
		} else {
			// Filter slots for start time (apply_to = 'both' or 'start').
			$global_settings       = bkap_global_setting();
			$is_list_view          = isset( $global_settings->booking_timeslot_display_mode ) && 'list-view' === $global_settings->booking_timeslot_display_mode ? true : false;
			$start_time_label      = get_option( 'book_start_time-label', __( 'Start Time', 'woocommerce-booking' ) );
			$start_time_label      = apply_filters( 'bkap_change_start_time_label', $start_time_label, $booking_settings, $product_id );
			$start_time_label_html = '<label id="bkap_start_time_label">' . $start_time_label . '</label>';
			$start_time_label_html = apply_filters( 'bkap_change_start_time_label_section', $start_time_label_html );

			if ( $is_list_view ) {
				$block_html = $start_time_label_html . '<ul class="bkap-duration-block">';
			} else {
				$choose_a_time = get_option( 'book_time-select-option', __( 'Choose a time', 'woocommerce-booking' ) );
				$block_html = $start_time_label_html . '<select name="bkap_start_time" id="bkap_start_time" class="bkap_start_time_slot"><option value="">' . $choose_a_time . '</option>';
			}
			foreach ( $times as $key => $value ) {

				if ( $is_full_range ) {
					$timeslot_value   = gmdate( 'H:i', strtotime( $value['time'] ) );
					$timeslot_display = date_i18n( bkap_common::bkap_get_time_format(), strtotime( $value['time'] ) );
				} else {
					$timeslot_value   = gmdate( 'H:i', strtotime( $value['start_time'] ) ) . ' - ' . gmdate( 'H:i', strtotime( $value['end_time'] ) );
					$timeslot_display = date_i18n( bkap_common::bkap_get_time_format(), strtotime( $value['start_time'] ) ) . ' - ' . date_i18n( bkap_common::bkap_get_time_format(), strtotime( $value['end_time'] ) );
				}

				$bookings_left = 'unlimited' === $value['available'] ? __( 'Unlimited', 'woocommerce-booking' ) : $value['available'];
				if ( $is_list_view ) {

					if ( 'unlimited' !== $value['available'] ) {

						if ( 0 === $value['booked'] ) {
							if ( 1 === $value['available'] ) {
								/* Translators: %s Bookings. */
								$bookings_left = sprintf( _n( '(%s Booking)', '(%s Booking)', $bookings_left, 'woocommerce-booking' ), $bookings_left );
							} else {
								/* Translators: %s Bookings. */
								$bookings_left = sprintf( _n( '(%s Bookings)', '(%s Bookings)', $bookings_left, 'woocommerce-booking' ), $bookings_left );
							}
						} elseif ( 1 === $value['available'] ) {
							/* Translators: %s Bookings. */
							$bookings_left = sprintf( _n( '(%s Booking left)', '(%s Booking left)', $bookings_left, 'woocommerce-booking' ), $bookings_left );
						} else {
							/* Translators: %s Bookings Left. */
							$bookings_left = sprintf( _n( '(%s Bookings left)', '(%s Bookings left)', $bookings_left, 'woocommerce-booking' ), $bookings_left );
						}
						$bookings_left = '<small style="display:block;" class="booking-spaces-left">' . $bookings_left . '</small>';
					} else {
						$bookings_left = '';
					}
					$booking_left = apply_filters( 'bkap_booking_show_availability_count_in_start_list', $bookings_left, $value );
					$block_html .=
						'<li class="bkap_block" data-block="' . esc_attr( $timeslot_value ) . '">
						<a href="#" data-value="' . $timeslot_value . '">' . $timeslot_display . $bookings_left .
						'</a></li>';
				} else {
					$booking_left = apply_filters( 'bkap_booking_show_availability_count_in_start_dropdown', '', $bookings_left, $value );
					$block_html  .= '<option value="' . esc_attr( $timeslot_value ) . '">' . $timeslot_display . $booking_left . '</option>';
				}
			}

			$block_html .= $is_list_view ? '</ul>' : '</select>';

			wp_send_json_success(
				array(
					'times' => $block_html,
				)
			);
		}
	}

	/**
	 * AJAX callback to get available end times based on start date/time and end date
	 */
	public function bkap_get_available_end_times_callback() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'bkap_booking_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed. Please refresh the page and try again.', 'bkap' ) ) );
		}

		$calendar_date    = isset( $_POST['date'] ) ? sanitize_text_field( wp_unslash( $_POST['date'] ) ) : '';
		$calendar_enddate = isset( $_POST['enddate'] ) ? sanitize_text_field( wp_unslash( $_POST['enddate'] ) ) : '';
		$start_date       = isset( $_POST['start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['start_date'] ) ) : '';
		$end_date         = isset( $_POST['end_date'] ) ? sanitize_text_field( wp_unslash( $_POST['end_date'] ) ) : '';
		$start_time       = isset( $_POST['start_time'] ) ? sanitize_text_field( wp_unslash( $_POST['start_time'] ) ) : '';
		$product_id       = isset( $_POST['product_id'] ) ? intval( $_POST['product_id'] ) : 0;

		if ( empty( $start_date ) || empty( $start_time ) || empty( $end_date ) ) {
			wp_send_json_error(
				array(
					'message' => __( 'Invalid request - missing required fields', 'woocommerce-booking' )
				)
			);
			return;
		}

		if ( str_contains( $start_time, ' - ' ) ) {
			$start_time_explode = explode( ' - ', $start_time );
			$start_time         = $start_time_explode[0];
		}

		// Call the helper function with start date/time for filtering.
		$times = bkap_get_available_times_ajax(
			$product_id,
			$end_date,
			'end',
			$start_date,
			$start_time
		);

		$booking_settings = bkap_setting( $product_id );
		$block_type       = isset( $booking_settings['booking_multiple_days_time_block_type'] ) ? $booking_settings['booking_multiple_days_time_block_type'] : 'full_range';
		$is_full_range    = 'full_range' === $block_type ? true : false;

		if ( empty( $times ) ) {
			wp_send_json_success(
				array(
					'times'   => '',
					'message' => __( 'No timeslots available for this date', 'woocommerce-booking' ),
				)
			);
			return;
		} else {
			$global_settings = bkap_global_setting();
			$is_list_view    = isset( $global_settings->booking_timeslot_display_mode ) && 'list-view' === $global_settings->booking_timeslot_display_mode ? true : false;
			$end_time_label  = get_option( 'book_end_time-label', __( 'End Time', 'woocommerce-booking' ) );

			$end_time_label      = apply_filters( 'bkap_change_end_time_label', $end_time_label, $booking_settings, $product_id );
			$end_time_label_html = '<label id="bkap_end_time_label">' . $end_time_label . '</label>';
			$end_time_label_html = apply_filters( 'bkap_change_end_time_label_section', $end_time_label_html );

			if ( $is_list_view ) {
				$block_html = $end_time_label_html . '<ul class="bkap-duration-block">';
			} else {
				$choose_a_time = get_option( 'book_time-select-option', __( 'Choose a time', 'woocommerce-booking' ) );
				$block_html    = $end_time_label_html . '<select name="bkap_end_time" id="bkap_end_time" class="bkap_end_time_slot"><option value="">' . $choose_a_time . '</option>';
			}

			foreach ( $times as $key => $value ) {

				if ( $is_full_range ) {
					$timeslot_value   = gmdate( 'H:i', strtotime( $value['time'] ) );
					$timeslot_display = date_i18n( bkap_common::bkap_get_time_format(), strtotime( $value['time'] ) );
				} else {
					$timeslot_value   = gmdate( 'H:i', strtotime( $value['start_time'] ) ) . ' - ' . gmdate( 'H:i', strtotime( $value['end_time'] ) );
					$timeslot_display = date_i18n( bkap_common::bkap_get_time_format(), strtotime( $value['start_time'] ) ) . ' - ' . date_i18n( bkap_common::bkap_get_time_format(), strtotime( $value['end_time'] ) );
				}

				$bookings_left = 'unlimited' === $value['available'] ? __( 'Unlimited', 'woocommerce-booking' ) : $value['available'];
				if ( $is_list_view ) {
					if ( 'unlimited' !== $value['available'] ) {

						if ( 0 === $value['booked'] ) {
							if ( 1 === $value['available'] ) {
								/* Translators: %s Bookings. */
								$bookings_left = sprintf( _n( '(%s Booking)', '(%s Booking)', $bookings_left, 'woocommerce-booking' ), $bookings_left );
							} else {
								/* Translators: %s Bookings. */
								$bookings_left = sprintf( _n( '(%s Bookings)', '(%s Bookings)', $bookings_left, 'woocommerce-booking' ), $bookings_left );
							}
						} elseif ( 1 === $value['available'] ) {
							/* Translators: %s Bookings. */
							$bookings_left = sprintf( _n( '(%s Booking left)', '(%s Booking left)', $bookings_left, 'woocommerce-booking' ), $bookings_left );
						} else {
							/* Translators: %s Bookings Left. */
							$bookings_left = sprintf( _n( '(%s Bookings left)', '(%s Bookings left)', $bookings_left, 'woocommerce-booking' ), $bookings_left );
						}
						$bookings_left = '<small style="display:block;" class="booking-spaces-left">' . $bookings_left . '</small>';
					} else {
						$bookings_left = '';
					}

					$booking_left = apply_filters( 'bkap_booking_show_availability_count_in_end_list', $bookings_left, $value );

					$block_html .=
						'<li class="bkap_block" data-block="' . esc_attr( $timeslot_value ) . '">
						<a href="#" data-value="' . $timeslot_value . '">' . $timeslot_display . $bookings_left . 
						'</a></li>';
				} else {
					$booking_left = apply_filters( 'bkap_booking_show_availability_count_in_end_dropdown', '', $bookings_left, $value );
					$block_html  .= '<option value="' . esc_attr( $timeslot_value ) . '">' . $timeslot_display . $booking_left . '</option>';
				}
			}

			$block_html .= $is_list_view ? '</ul>' : '</select>';

			wp_send_json_success(
				array(
					'times' => $block_html,
				)
			);
		}
	}

	/**
	 * Get time slots for a specific date
	 * 
	 * @param string $date Date in dd-mm-yyyy format
	 * @param array $booking_settings Booking settings array
	 * @param string $slot_type 'start' or 'end'
	 * @return array Array of time slots
	 */
	private function get_time_slots_for_date( $date, $booking_settings, $slot_type = 'both' ) {
		$time_slots = array();
		
		// Check if multiple days time data exists
		if ( ! isset( $booking_settings['booking_multiple_days_time_data'] ) ) {
			return $time_slots;
		}
		
		$time_data = $booking_settings['booking_multiple_days_time_data'];
		
		// Check for specific date first
		if ( isset( $time_data[ $date ] ) && is_array( $time_data[ $date ] ) ) {
			$time_slots = $this->parse_time_slots( $time_data[ $date ], $slot_type );
		} else {
			// Check for weekday slots
			$date_obj = DateTime::createFromFormat( 'j-n-Y', $date );
			if ( $date_obj ) {
				$weekday_key = 'booking_weekday_' . $date_obj->format( 'w' );
				
				if ( isset( $time_data[ $weekday_key ] ) && is_array( $time_data[ $weekday_key ] ) ) {
					$time_slots = $this->parse_time_slots( $time_data[ $weekday_key ], $slot_type );
				}
			}
		}
		
		return $time_slots;
	}

	/**
	 * Parse time slots from settings
	 * 
	 * @param array $slots Raw slot data
	 * @param string $slot_type 'start', 'end', or 'both'
	 * @return array Formatted time slots
	 */
	private function parse_time_slots( $slots, $slot_type = 'both' ) {
		$parsed_slots = array();
		
		foreach ( $slots as $slot ) {
			if ( ! is_array( $slot ) ) {
				continue;
			}
			
			// Check if slot should be shown for this type
			$apply_to = isset( $slot['apply_to'] ) ? $slot['apply_to'] : 'both';
			
			$show_in_start = ( $apply_to === 'both' || $apply_to === 'start' );
			$show_in_end = ( $apply_to === 'both' || $apply_to === 'end' );
			
			if ( $slot_type === 'start' && ! $show_in_start ) {
				continue;
			}
			
			if ( $slot_type === 'end' && ! $show_in_end ) {
				continue;
			}
			
			// Add 'from' time (start of slot)
			if ( isset( $slot['from'] ) ) {
				$parsed_slots[] = array(
					'value' => $slot['from'],
					'label' => $slot['from'],
					'max_booking' => isset( $slot['max_booking'] ) ? absint( $slot['max_booking'] ) : 0,
					'slot_type' => 'from'
				);
			}
			
			// Optionally add 'to' time (end of slot) - useful for end time selection
			if ( $slot_type === 'end' && isset( $slot['to'] ) ) {
				$parsed_slots[] = array(
					'value' => $slot['to'],
					'label' => $slot['to'],
					'max_booking' => isset( $slot['max_booking'] ) ? absint( $slot['max_booking'] ) : 0,
					'slot_type' => 'to'
				);
			}
		}
		
		// Remove duplicates and sort
		$parsed_slots = $this->unique_time_slots( $parsed_slots );
		usort( $parsed_slots, array( $this, 'sort_time_slots' ) );
		
		return $parsed_slots;
	}

	/**
	 * Remove duplicate time slots
	 */
	private function unique_time_slots( $slots ) {
		$unique = array();
		$seen = array();
		
		foreach ( $slots as $slot ) {
			$key = $slot['value'];
			if ( ! in_array( $key, $seen ) ) {
				$unique[] = $slot;
				$seen[] = $key;
			}
		}
		
		return $unique;
	}

	/**
	 * Sort time slots by time value
	 */
	private function sort_time_slots( $a, $b ) {
		return strcmp( $a['value'], $b['value'] );
	}

	/**
	 * Filter end times for same day booking
	 */
	private function filter_end_times_same_day( $time_slots, $start_time ) {
		$filtered = array();
		
		$start_time_minutes = $this->time_to_minutes( $start_time );
		
		foreach ( $time_slots as $slot ) {
			$slot_time_minutes = $this->time_to_minutes( $slot['value'] );
			
			// Only include times after start time
			if ( $slot_time_minutes > $start_time_minutes ) {
				$filtered[] = $slot;
			}
		}
		
		return $filtered;
	}

	/**
	 * Convert time to minutes for comparison
	 */
	private function time_to_minutes( $time ) {
		$parts = explode( ':', $time );
		if ( count( $parts ) !== 2 ) {
			return 0;
		}
		return ( absint( $parts[0] ) * 60 ) + absint( $parts[1] );
	}

	/**
	 * Filter available slots based on bookings
	 */
	private function filter_available_slots( $date, $time_slots, $product_id, $slot_type, $start_date = '', $start_time = '' ) {
		global $wpdb;
		
		$available = array();
		
		foreach ( $time_slots as $slot ) {
			$time = $slot['value'];
			$max_booking = $slot['max_booking'];
			
			// Get existing bookings for this slot
			$booked_count = $this->get_booked_count( $date, $time, $product_id, $slot_type );

			// Check if slot is available
			if ( $max_booking == 0 || $booked_count < $max_booking ) {
				$available[] = array(
					'value' => $time,
					'label' => $time . ' (' . ( $max_booking - $booked_count ) . ' available)',
					'available' => $max_booking - $booked_count
				);
			}
		}
		
		return $available;
	}
}