<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use Automattic\WooCommerce\Admin\Features\Navigation\Menu;
use Automattic\WooCommerce\Admin\Features\Navigation\Screen;

/**
 * Store Admin Class.
 *
 * General admin class to handle all things admin side for store.
 *
 * @category Admin
 * @package  WooCommerce Product Vendors/Admin
 * @version  2.0.0
 */
class WC_Product_Vendors_Store_Admin {
	public static $self;

	/**
	 * Initialize.
	 *
	 * @since 2.0.0
	 * @version 2.0.35
	 */
	public static function init() {
		self::$self = new self();

		// enqueues scripts and styles.
		add_action( 'admin_enqueue_scripts', array( self::$self, 'enqueue_scripts_styles' ) );

		// displays count bubble on pending items such as products or vendors.
		add_filter( 'add_menu_classes', array( self::$self, 'pending_items_count' ) );

		// Perform some actions when a user is deleted.
		add_action( 'delete_user', array( self::$self, 'delete_user' ) );

		// adds the screen ids to WooCommerce so WooCommerce scripts and styles will load.
		add_filter( 'woocommerce_screen_ids', array( self::$self, 'add_screen_ids_to_wc' ) );

		// add fields to taxonomy edit page.
		add_action( WC_PRODUCT_VENDORS_TAXONOMY . '_edit_form_fields', array( self::$self, 'edit_vendor_fields' ) );

		// add fields to taxonomy on create page.
		add_action( WC_PRODUCT_VENDORS_TAXONOMY . '_add_form_fields', array( self::$self, 'add_vendor_fields' ) );

		// save custom fields from taxonomy.
		add_action( 'edited_' . WC_PRODUCT_VENDORS_TAXONOMY, array( self::$self, 'save_vendor_fields' ) );

		// save custom fields from taxonomy.
		add_action( 'created_' . WC_PRODUCT_VENDORS_TAXONOMY, array( self::$self, 'save_vendor_fields' ) );

		// validate the email field when a term is first created.
		add_filter( 'pre_insert_term', array( self::$self, 'validate_email_field' ), 10, 2 );

		// output a notice if a vendor is missing the email field
		add_action( 'admin_notices', array( self::$self, 'invalid_vendor_data_notice' ) );

		// modify taxonomy columns.
		add_filter( 'manage_edit-' . WC_PRODUCT_VENDORS_TAXONOMY . '_columns', array( self::$self, 'modify_vendor_columns' ) );

		// modify taxonomy columns.
		add_filter( 'manage_' . WC_PRODUCT_VENDORS_TAXONOMY . '_custom_column', array( self::$self, 'render_vendor_columns' ), 10, 3 );

		// add a new column to users.
		add_filter( 'manage_users_columns', array( self::$self, 'add_custom_user_column' ) );

		// modify user columns.
		add_action( 'manage_users_custom_column', array( self::$self, 'add_user_column_data' ), 10, 3 );

		// add vendor section to user profile.
		add_action( 'edit_user_profile', array( self::$self, 'add_product_vendor_user_profile_section' ) );
		add_action( 'show_user_profile', array( self::$self, 'add_product_vendor_user_profile_section' ) );

		// save user profile.
		add_action( 'edit_user_profile_update', array( self::$self, 'save_product_vendor_user_profile_section' ) );

		// add commission top level menu item.
		add_action( 'admin_menu', array( self::$self, 'register_commissions_menu_item' ) );

		// Register menu items in the WooCommerce navigation. This feature was removed in WC 9.3.
		add_action( 'admin_menu', array( self::$self, 'register_navigation_items' ) );

		// set the screen option.
		add_filter( 'set-screen-option', array( self::$self, 'set_screen_option' ), 99, 3 );

		// adds fields to attachments.
		add_filter( 'attachment_fields_to_edit', array( self::$self, 'add_attachments_field' ), 10, 2 );

		// save fields to attachments.
		add_filter( 'attachment_fields_to_save', array( self::$self, 'save_attachments_field' ), 10, 2 );

		// add vendor settings section to products tab.
		add_filter( 'woocommerce_get_sections_products', array( self::$self, 'add_vendor_settings_section' ) );

		// get vendor settings.
		add_filter( 'woocommerce_get_settings_products', array( self::$self, 'add_vendor_settings' ), 10, 2 );

		// save vendor settings.
		add_filter( 'woocommerce_admin_settings_sanitize_option', array( self::$self, 'save_vendor_settings' ), 10, 3 );

		// add sold by vendor on order item.
		add_action( 'woocommerce_after_order_itemmeta', array( self::$self, 'add_sold_by_order_item_detail' ), 10, 3 );

		// add a commission field to the product general tab.
		add_action( 'woocommerce_product_options_general_product_data', array( self::$self, 'add_product_commission_field_general' ) );

		// save commission field for the product general tab.
		add_action( 'woocommerce_process_product_meta_simple', array( self::$self, 'save_product_commission_field_general' ) );
		add_action( 'woocommerce_process_product_meta_booking', array( self::$self, 'save_product_commission_field_general' ) );
		add_action( 'woocommerce_process_product_meta_subscription', array( self::$self, 'save_product_commission_field_general' ) );
		add_action( 'woocommerce_process_product_meta_variable-subscription', array( self::$self, 'save_product_commission_field_general' ) );

		// add a commission field for the product variation.
		add_action( 'woocommerce_product_after_variable_attributes', array( self::$self, 'add_product_commission_field_variation' ), 10, 3 );

		// save commission field for product variation.
		add_action( 'woocommerce_process_product_meta_variable', array( self::$self, 'save_product_commission_field_variable' ) );
		add_action( 'woocommerce_save_product_variation', array( self::$self, 'save_product_commission_field_variation' ), 10, 2 );

		// add variation commission bulk edit.
		add_action( 'woocommerce_variable_product_bulk_edit_actions', array( self::$self, 'add_variation_vendor_bulk_edit' ) );

		// add a pass shipping/tax field to the product general tab.
		add_action( 'woocommerce_product_options_general_product_data', array( self::$self, 'add_product_pass_shipping_tax_field_general' ) );

		// save pass shipping/tax field for the product general tab.
		add_action( 'woocommerce_process_product_meta', array( self::$self, 'save_product_pass_shipping_tax_field_general' ) );

		// add pass shipping/tax to product bulk edit menu.
		add_action( 'woocommerce_product_bulk_edit_end', array( self::$self, 'add_product_bulk_edit_pass_shipping_tax' ) );

		// save pass shipping/tax to product bulk edit.
		add_action( 'woocommerce_product_bulk_edit_save', array( self::$self, 'save_product_bulk_edit_pass_shipping_tax' ) );

		// clear reports transients.
		add_action( 'woocommerce_new_order', array( self::$self, 'clear_reports_transients' ) );
		add_action( 'save_post', array( self::$self, 'clear_reports_transients' ) );
		add_action( 'delete_post', array( self::$self, 'clear_reports_transients' ) );
		add_action( 'woocommerce_order_status_changed', array( self::$self, 'clear_reports_transients' ) );
		add_action( 'wcpv_commissions_status_changed', array( self::$self, 'clear_reports_transients' ) );

		// reports ajax search for vendors.
		add_action( 'wp_ajax_wcpv_vendor_search_ajax', array( self::$self, 'vendor_search_ajax' ) );

		// exports commissions for the current view.
		add_action( 'wp_ajax_wcpv_export_commissions_ajax', array( self::$self, 'export_commissions_ajax' ) );

		// exports unpaid commissions.
		add_action( 'wp_ajax_wcpv_export_unpaid_commissions_ajax', array( self::$self, 'export_unpaid_commissions_ajax' ) );

		// process when vendor role is updated from pending to admin or manager.
		add_action( 'set_user_role', array( self::$self, 'role_update' ), 10, 3 );

		// add clear transients button in WC system tools.
		add_filter( 'woocommerce_debug_tools', array( self::$self, 'add_debug_tool' ) );

		// Filter order item meta label.
		add_filter( 'woocommerce_order_item_display_meta_key', array( self::$self, 'filter_order_attribute_label' ) );
		add_filter( 'woocommerce_order_item_display_meta_value', array( self::$self, 'filter_order_attribute_value' ), 10, 2 );

		// add quick edit items and process save.
		add_action( 'quick_edit_custom_box', array( self::$self, 'quick_edit' ), 10, 2 );
		add_action( 'bulk_edit_custom_box', array( self::$self, 'quick_edit' ), 10, 2 );
		add_action( 'save_post', array( self::$self, 'bulk_and_quick_edit_save_post' ), 10, 2 );

		// saves the vendor to the product.
		add_action( 'save_post', array( self::$self, 'save_product_vendor' ) );


		// output a notice to migrate to new admin storage.
		add_action( 'admin_notices', array( self::$self, 'migrate_admin_storage_notice' ) );

		return true;
	}

	/**
	 * Returns the current class object.
	 *
	 * @since 2.0.35
	 * @version 2.0.35
	 */
	public static function get_instance() {
		return self::$self;
	}

	/**
	 * Adds our screen ids to WC so scripts can load
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return array $screen_ids
	 */
	public function add_screen_ids_to_wc( $screen_ids ) {
		$screen_ids[] = 'edit-wcpv_product_vendors';

		return $screen_ids;
	}

	/**
	 * Gets the screen ids that needs styles or scripts
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return array
	 */
	public function get_screen_ids() {
		return apply_filters( 'wcpv_store_admin_screen_ids', array(
			'edit-wcpv_product_vendors',
			'toplevel_page_wcpv-commissions',
			'product',
			'edit-product',
			'woocommerce_page_wc-reports',
			'woocommerce_page_wc-settings',
		) );
	}

	/**
	 * Enqueue scripts and styles
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function enqueue_scripts_styles() {
		$current_screen = get_current_screen();

		$asset_file = WC_PRODUCT_VENDORS_PATH . '/build/admin/wcpv-admin-scripts.asset.php';

		if ( ! file_exists( $asset_file ) ) {
			return;
		}

		$asset_file = require_once $asset_file;

		wp_register_script( 'wcpv-admin-scripts', WC_PRODUCT_VENDORS_PLUGIN_URL . '/build/admin/wcpv-admin-scripts.js', $asset_file['dependencies'] ?? array(), $asset_file['version'] ?? WC_PRODUCT_VENDORS_VERSION, true );
		wp_register_style( 'wcpv-admin-styles', WC_PRODUCT_VENDORS_PLUGIN_URL . '/build/admin/wcpv-admin-scripts.css', array( 'woocommerce_admin_styles' ), $asset_file['version'] ?? WC_PRODUCT_VENDORS_VERSION );

		$localized_vars = array(
			'isPendingVendor'                            => current_user_can( 'wc_product_vendors_pending_vendor' ), // phpcs:ignore WordPress.WP.Capabilities.Unknown -- Custom capability registered by WC Product Vendors
			'pending_vendor_message'                     => __( 'Thanks for registering to become a vendor.  Your application is being reviewed at this time.', 'woocommerce-product-vendors' ),
			'modalLogoTitle'                             => __( 'Add Logo', 'woocommerce-product-vendors' ),
			'buttonLogoText'                             => __( 'Add Logo', 'woocommerce-product-vendors' ),
			'ajaxurl'                                    => admin_url( 'admin-ajax.php' ),
			'vendor_search_nonce'                        => wp_create_nonce( '_wcpv_vendor_search_nonce' ),
			'i18n_matches_1'                             => _x( 'One result is available, press enter to select it.', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_matches_n'                             => _x( '%qty% results are available, use up and down arrow keys to navigate.', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_no_matches'                            => _x( 'No matches found', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_ajax_error'                            => _x( 'Loading failed', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_input_too_short_1'                     => _x( 'Please enter 1 or more characters', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_input_too_short_n'                     => _x( 'Please enter %qty% or more characters', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_input_too_long_1'                      => _x( 'Please delete 1 character', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_input_too_long_n'                      => _x( 'Please delete %qty% characters', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_selection_too_long_1'                  => _x( 'You can only select 1 item', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_selection_too_long_n'                  => _x( 'You can only select %qty% items', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_load_more'                             => _x( 'Loading more results&hellip;', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_searching'                             => _x( 'Searching&hellip;', 'enhanced select', 'woocommerce-product-vendors' ),
			'i18n_import_csv_override_confirmation_text' => esc_html__( 'This will remove all existing shipping rules and replace them with the rule(s) in your imported CSV file. Are you sure this is what you intend to do?', 'woocommerce-product-vendors' ),
			'currency_format'                            => str_replace( array( '%1$s', '%2$s', '%s' ), '%s', get_woocommerce_price_format() ),
			'currency_symbol'                            => get_woocommerce_currency_symbol(),
			'currency'                                   => get_woocommerce_currency(),
			'currency_name'                              => get_woocommerce_currencies()[ get_woocommerce_currency() ],
			/* translators: 1: formatted amount with currency symbol, 2: currency code, 3: currency name */
			'i18n_fixed_commission_text'                 => __( 'Default commission is set to a <b>fixed amount</b> of <b>%1$s</b> (%2$s - %3$s) for each product sale.', 'woocommerce-product-vendors' ),
			/* translators: %1$s: commission percentage */
			'i18n_percentage_commission_text'            => __( 'Default commission is set to <b>%1$s%</b> of the product price.', 'woocommerce-product-vendors' ),
			'i18n_manual_payment_desc'                   => __( 'Payments must be processed manually through external payment methods like wire transfers, checks, or cash. After completing each payment, commissions need to be manually marked as paid in the system.', 'woocommerce-product-vendors' ),
			'i18n_weekly_payment_desc'                   => __( 'Automatically process payments every week through PayPal. Requires valid PayPal API credentials.', 'woocommerce-product-vendors' ),
			'i18n_biweekly_payment_desc'                 => __( 'Automatically process payments every two weeks through PayPal. Requires valid PayPal API credentials.', 'woocommerce-product-vendors' ),
			'i18n_monthly_payment_desc'                  => __( 'Automatically process payments once a month through PayPal. Requires valid PayPal API credentials.', 'woocommerce-product-vendors' ),
			'i18n_commission_percentage_message'         => __( 'For percentage: enter 1-100 (no % symbol needed).', 'woocommerce-product-vendors' ),
			'i18n_commission_fixed_message'              => __( 'For fixed amount: enter the numeric value (e.g. 20).', 'woocommerce-product-vendors' ),
		);

		wp_localize_script( 'wcpv-admin-scripts', 'wcpv_admin_local', $localized_vars );

		if ( in_array( $current_screen->id, $this->get_screen_ids() ) ) {

			if ( ! WC_Product_Vendors_Utils::is_vendor() ) {
				wp_enqueue_script( 'wcpv-admin-scripts' );
			}

			wp_enqueue_style( 'wcpv-admin-styles' );

			wp_enqueue_script( 'wc-users', WC()->plugin_url() . '/assets/js/admin/users.js', array( 'jquery', 'wc-enhanced-select' ), WC_VERSION, true );

			$countries = array_merge( WC()->countries->get_allowed_country_states(), WC()->countries->get_shipping_country_states() );
			$countries = function_exists( 'wc_esc_json' ) ? wc_esc_json( wp_json_encode( $countries ) ) : _wp_specialchars( wp_json_encode( $countries ), ENT_QUOTES, 'UTF-8', true );

			wp_localize_script(
				'wc-users',
				'wc_users_params',
				array(
					'countries'              => $countries,
					'i18n_select_state_text' => esc_attr__( 'Select an option&hellip;', 'woocommerce-product-vendors' ),
				)
			);
		}

		return true;
	}

	/**
	 * Role update / send email
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.31
	 * @param int $user_id
	 * @param string $new_role
	 * @param array $old_roles
	 * @return bool
	 */
	public function role_update( $user_id, $new_role, $old_roles ) {
		if ( ! current_user_can( 'manage_vendors' ) ) {
			return;
		}

		$vendor_roles = array( 'wc_product_vendors_admin_vendor', 'wc_product_vendors_manager_vendor' );

		$approved_already = get_user_meta( $user_id, '_wcpv_vendor_approval', true );

		// Remove vendor approval if vendor role is changed to non vendor role.
		foreach ( $old_roles as $old_role ) {
			if ( in_array( $old_role, $vendor_roles ) && ! in_array( $new_role, $vendor_roles ) ) {
				delete_user_meta( $user_id, '_wcpv_vendor_approval' );
			}
		}

		if (
			! in_array( $new_role, $old_roles ) &&
			in_array( $new_role, $vendor_roles ) &&
			'yes' !== $approved_already
		) {

			$emails = WC()->mailer()->get_emails();

			if ( ! empty( $emails ) ) {
				$emails['WC_Product_Vendors_Approval']->trigger( $user_id, $new_role, $old_roles );
			}
		}

		// Remove pending new vendor from saved list.
		WC_Product_Vendors_Utils::delete_new_pending_vendor( $user_id );

		return true;
	}

	/**
	 * Shows the pending count bubble on sidebar menu items for pending vendors
	 * and pending product approvals.
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.4.0
	 * @param object $menu
	 * @return array $menu modified menu
	 */
	public function pending_items_count( $menu ) {
		if ( ! current_user_can( 'manage_vendors' ) ) { // phpcs:ignore WordPress.WP.Capabilities.Unknown
			return $menu;
		}

		$pending_vendors_count = ( false !== get_transient( 'wcpv_new_pending_vendor_list' ) ) ? count( get_transient( 'wcpv_new_pending_vendor_list' ) ) : array();

		// Cache key should vary based on the current user's capability to read private posts.
		$cache_key_suffix = current_user_can( get_post_type_object( 'product' )->cap->read_private_posts ) ? 'private' : 'public_' . get_current_user_id();
		$cache_key        = "wcpv_pending_products_count_{$cache_key_suffix}";

		// Fetch product count from cache.
		$pending_products_count = wp_cache_get( $cache_key, 'wcpv' );

		if ( false === $pending_products_count ) {
			$products               = wp_count_posts( 'product', 'readable' );
			$pending_products_count = ! empty( $products->pending ) ? $products->pending : '';

			wp_cache_set( $cache_key, $pending_products_count, 'wcpv', HOUR_IN_SECONDS );
		}

		foreach ( $menu as $menu_key => $menu_data ) {
			if ( 'users.php' === $menu_data[2] && ! empty( $pending_vendors_count ) ) {
				$menu[ $menu_key ][0] .= ' <span class="update-plugins count-' . $pending_vendors_count . '" title="' . esc_attr__( 'Vendors awaiting review', 'woocommerce-product-vendors' ) . '"><span class="plugin-count">' . number_format_i18n( $pending_vendors_count ) . '</span></span>';
			}

			if ( 'edit.php?post_type=product' === $menu_data[2] && '' !== $pending_products_count ) {
				$menu[ $menu_key ][0] .= ' <span class="update-plugins count-' . $pending_products_count . '" title="' . esc_attr__( 'Products awaiting review', 'woocommerce-product-vendors' ) . '"><span class="plugin-count">' . number_format_i18n( $pending_products_count ) . '</span></span>';
			}
		}

		return $menu;
	}

	/**
	 * Perform action when user is deleted.
	 *
	 * @since 2.0.31
	 * @version 2.0.31
	 * @param int $user_id
	 * @return void
	 */
	public function delete_user( $user_id ) {
		WC_Product_Vendors_Utils::delete_new_pending_vendor( $user_id );
	}

	/**
	 * Adds vendor fields to vendor create page
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.16
	 * @return bool
	 */
	public function add_vendor_fields() {
		$tzstring = WC_Product_Vendors_Utils::get_default_timezone_string();

		/**
		 * Optionally override the views/html-create-vendor-fields-page.php view: filters must return a string to be passed into include_once.
		 *
		 * @since 2.1.77
		 *
		 * @param string $path Default path to the view.
		 */
		include_once( apply_filters( 'wcpv_create_vendor_fields_page_template', 'views/html-create-vendor-fields-page.php' ) );

		return true;
	}

	/**
	 * Adds additional fields for product vendor term
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.16
	 * @param object $taxonomy
	 * @return bool
	 */
	public function edit_vendor_fields( $term ) {
		wp_enqueue_script( 'wc-enhanced-select' );

		$jquery_tiptip_handle = 'jquery-tiptip';
		if ( version_compare( WC_VERSION, '10.3', '>=' ) ) {
			$jquery_tiptip_handle = 'wc-jquery-tiptip';
		}
		wp_enqueue_script( $jquery_tiptip_handle );

		$vendor_data = WC_Product_Vendors_Utils::get_vendor_data_by_id( $term->term_id );

		$description          = ! empty( $vendor_data['description'] ) ? $vendor_data['description'] : '';
		$notes                = ! empty( $vendor_data['notes'] ) ? $vendor_data['notes'] : '';
		$logo                 = ! empty( $vendor_data['logo'] ) ? $vendor_data['logo'] : '';
		$profile              = ! empty( $vendor_data['profile'] ) ? $vendor_data['profile'] : '';
		$email                = ! empty( $vendor_data['email'] ) ? $vendor_data['email'] : '';
		$commission           = ( isset( $vendor_data['commission'] ) && is_numeric( $vendor_data['commission'] ) ) ? $vendor_data['commission'] : '';
		$commission_type      = ! empty( $vendor_data['commission_type'] ) ? $vendor_data['commission_type'] : 'percentage';
		$instant_payout       = ! empty( $vendor_data['instant_payout'] ) ? $vendor_data['instant_payout'] : 'no';
		$paypal               = ! empty( $vendor_data['paypal'] ) ? $vendor_data['paypal'] : '';
		$per_product_shipping = ! empty( $vendor_data['per_product_shipping'] ) ? $vendor_data['per_product_shipping'] : 'no';
		$enable_bookings      = ! empty( $vendor_data['enable_bookings'] ) ? $vendor_data['enable_bookings'] : 'no';
		$admins               = ! empty( $vendor_data['admins'] ) ? $vendor_data['admins'] : array();
		$tzstring             = ! empty( $vendor_data['timezone'] ) ? $vendor_data['timezone'] : '';
		$pass_shipping        = ! empty( $vendor_data['pass_shipping'] ) ? $vendor_data['pass_shipping'] : 'no';
		$taxes                = ! empty( $vendor_data['taxes'] ) ? $vendor_data['taxes'] : 'keep-tax';

		$selected_admins = array();

		if ( empty( $tzstring ) ) {
			$tzstring = WC_Product_Vendors_Utils::get_default_timezone_string();
		}

		foreach ( $admins as $admin_id ) {
			$admin = get_user_by( 'id', $admin_id );

			if ( is_object( $admin ) ) {
				$selected_admins[ $admin_id ] = esc_html( $admin->display_name ) . ' (#' . absint( $admin->ID ) . ') &ndash; ' . esc_html( $admin->user_email );
			}
		}

		$hide_remove_image_link = '';

		$logo_image_url = wp_get_attachment_image_src( $logo, 'thumbnail' );

		if ( empty( $logo_image_url ) ) {
			$hide_remove_image_link = 'display:none;';
		}

		/**
		 * Optionally override the views/html-edit-vendor-fields-page.php view: filters must return a string to be passed into include_once.
		 *
		 * @since 2.1.77
		 *
		 * @param string $path Default path to the view.
		 */
		include_once( apply_filters( 'wcpv_edit_vendor_fields_page_template', 'views/html-edit-vendor-fields-page.php' ) );

		return true;
	}

	/**
	 * Saves additional fields for product vendor term
	 *
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param object $term_id
	 * @return bool
	 */
	public function save_vendor_fields( $term_id ) {
		if ( ! empty( $_POST['vendor_data'] ) ) {
			$posted_vendor_data    = wp_unslash( $_POST['vendor_data'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$sanitized_vendor_data = array();

			foreach ( $posted_vendor_data as $data_key => $data_value ) {
				if ( 'admins' === $data_key ) {
					$sanitized_vendor_data[ $data_key ] = array_map( 'absint', $data_value );

					continue;
				}

				if ( 'description' === $data_key ) {
					$sanitized_vendor_data[ $data_key ] = ! empty( $data_value ) ? sanitize_text_field( $data_value ) : '';

					continue;
				}

				if ( 'profile' === $data_key ) {
					// Sanitize html editor content.
					$sanitized_vendor_data[ $data_key ] = ! empty( $data_value ) ? wp_kses_post( $data_value ) : '';

					continue;
				}

				if ( 'commission' === $data_key ) {
					// validate commission as it takes an absolute number
					$sanitized_vendor_data[ $data_key ] = WC_Product_Vendors_Utils::sanitize_commission( $data_value );

					continue;
				}

				$sanitized_vendor_data[ $data_key ] = sanitize_text_field( $data_value );
				$sanitized_vendor_data[ $data_key ] = stripslashes( $data_value );
			}

			$sanitized_vendor_data['pass_shipping'] = ! isset( $posted_vendor_data['pass_shipping'] ) ? 'no' : 'yes';

			// account for checkbox fields
			$sanitized_vendor_data['enable_bookings'] = ! isset( $posted_vendor_data['enable_bookings'] ) ? 'no' : 'yes';

			// account for checkbox fields
			$sanitized_vendor_data['per_product_shipping'] = ! isset( $posted_vendor_data['per_product_shipping'] ) ? 'no' : 'yes';

			// account for checkbox fields
			$sanitized_vendor_data['instant_payout'] = ! isset( $posted_vendor_data['instant_payout'] ) ? 'no' : 'yes';

			if ( empty( $posted_vendor_data['admins'] ) ) {
				$sanitized_vendor_data['admins'] = array();
			}

			WC_Product_Vendors_Utils::set_vendor_data( $term_id, $sanitized_vendor_data );
		}

		WC_Product_Vendors_Utils::clear_vendor_error_list_transient();

		return true;
	}

	/**
	 * Run validation on the vendor email field
	 *
	 * @since 2.1.52
	 * @version 2.1.52
	 * @param string $term The term name to add.
	 * @param string $taxonomy Taxonomy slug.
	 * @return string|WP_Error
	 */
	public function validate_email_field( $term, $taxonomy ) {
		if (
			WC_PRODUCT_VENDORS_TAXONOMY !== $taxonomy
			|| empty( $_POST['vendor_data'] )
			|| ! isset( $_POST['vendor_data']['email'] )
		) {
			return $term;
		}

		// Ensure we have proper email data
		$emails = explode( ',', wp_unslash( $_POST['vendor_data']['email'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$errors = 0;

		foreach ( $emails as $email ) {
			if ( $email && ! filter_var( $email, FILTER_VALIDATE_EMAIL ) ) {
				$errors ++;
			}
		}

		// If any email is not valid, return an error message
		if ( $errors === 1 ) {
			return new WP_Error( 'error', __( 'Please enter a valid Vendor Email', 'woocommerce-product-vendors' ) );
		} else if ( $errors > 1 ) {
			return new WP_Error( 'error', __( 'Please ensure all Vendor Emails are valid', 'woocommerce-product-vendors' ) );
		}

		return $term;
	}

	/**
	 * Output a notice if a vendor is missing their email
	 *
	 * @since 2.1.52
	 * @version 2.1.52
	 * @return void
	 */
	public function invalid_vendor_data_notice() {
		// only show to admin users
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// see if we have errors already cached
		$errors = ( false !== get_transient( 'wcpv_vendor_error_list' ) ) ? get_transient( 'wcpv_vendor_error_list' ) : array();

		// if nothing cached, run our checks
		if ( empty( $errors ) ) {
			// get all vendors
			$vendors = WC_Product_Vendors_Utils::get_all_vendors();

			if ( empty( $vendors ) ) {
				set_transient( 'wcpv_vendor_error_list', array( 'none' ), WEEK_IN_SECONDS );
				return;
			}

			// check all vendors to ensure they have an email associated
			foreach ( $vendors as $vendor ) {
				$vendor_data = WC_Product_Vendors_Utils::get_vendor_data_by_id( $vendor->term_id );

				if ( empty( $vendor_data['email'] ) ) {
					$errors[] = sprintf( '<a href="%1$s">%2$s</a>', get_edit_term_link( $vendor->term_id ), $vendor->name );
				}

				// only show 10 vendors at a time in the notice
				if ( count( $errors ) === 10 ) {
					break;
				}
			}

			// cache our errors, if any
			if ( empty( $errors ) ) {
				set_transient( 'wcpv_vendor_error_list', array( 'none' ), WEEK_IN_SECONDS );
			} else {
				set_transient( 'wcpv_vendor_error_list', $errors, WEEK_IN_SECONDS );
			}
		}

		// if we have errors, output those in a notice
		if ( ! empty( $errors ) && 'none' !== $errors[0] ) :
			?>

			<div class="notice notice-error is-dismissible">
				<p>
					<?php
					esc_html_e( 'Some Vendors don\'t have a valid email associated with them. This means notification emails will not be sent for these Vendors. Please edit these Vendors and add proper emails: ', 'woocommerce-product-vendors' );

					echo wp_kses_post( implode( ', ', $errors ) );
					?>
				</p>
			</div>

		<?php
		endif;
	}

	/**
	 * Modifies the vendor columns
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return array $columns modified columns
	 */
	public function modify_vendor_columns( $columns ) {
		unset( $columns['description'] );

		// add admins to column
		$columns['admins'] = __( 'Admins', 'woocommerce-product-vendors' );

		// rename count column
		$columns['posts'] = __( 'Products', 'woocommerce-product-vendors' );

		// add notes to column
		$columns['notes'] = __( 'Notes', 'woocommerce-product-vendors' );

		// add vendor id to column
		$columns['vendor_id'] = __( 'Vendor ID', 'woocommerce-product-vendors' );

		return $columns;
	}

	/**
	 * Renders the modified vendor column
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param string $value current value
	 * @param string $column_name current column name
	 * @param int $term_id current term id
	 * @return string $value
	 */
	public function render_vendor_columns( $value, $column_name, $term_id ) {
		$vendor_data = WC_Product_Vendors_Utils::get_vendor_data_by_id( $term_id );

		if ( 'vendor_id' === $column_name ) {
			$value .= esc_html( $term_id );
		}

		if ( 'admins' === $column_name && ! empty( $vendor_data['admins'] ) ) {
			if ( is_array( $vendor_data['admins'] ) ) {
				$admin_ids = array_map( 'absint', $vendor_data['admins'] );
			} else {
				$admin_ids = array_filter( array_map( 'absint', explode( ',', $vendor_data['admins'] ) ) );
			}

			foreach ( $admin_ids as $admin_id ) {
				$admin = get_user_by( 'id', $admin_id );

				if ( is_object( $admin ) ) {
					$value .= '<a href="' . get_edit_user_link( $admin_id ) . '" class="wcpv-vendor-column-user">' . esc_html( $admin->display_name ) . ' (#' . absint( $admin->ID ) . ' &ndash; ' . esc_html( $admin->user_email ) . ')</a><br />';
				}
			}
		}

		if ( 'notes' === $column_name && ! empty( $vendor_data['notes'] ) ) {
			$value .= esc_html( $vendor_data['notes'] );
		}

		return $value;
	}

	/**
	 * Add column to the user taxonomy columns
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param array $columns
	 * @return bool
	 */
	public function add_custom_user_column( $columns ) {
		$columns['vendors'] = __( 'Managed Vendors', 'woocommerce-product-vendors' );

		return $columns;
	}

	/**
	 * Modifies the user taxonomy columns
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param string $content
	 * @return string $vendors
	 */
	public function add_user_column_data( $content, $column_slug, $user_id ) {
		$vendor_data = WC_Product_Vendors_Utils::get_all_vendor_data( $user_id );

		if ( 'vendors' === $column_slug && ! empty( $vendor_data ) ) {
			$vendor_names = array();

			foreach ( $vendor_data as $data ) {
				$vendor_names[] = $data['name'];
			}

			return implode( '<br />', $vendor_names );
		}

		return $content;
	}

	/**
	 * Add vendor section fields to user profile
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.1.0
	 * @param object $user
	 * @return bool
	 */
	public function add_product_vendor_user_profile_section( $user ) {
		// display section only if current user is an admin and the editing user is a vendor
		if ( ! WC_Product_Vendors_Utils::is_vendor( $user->ID ) || ! current_user_can( 'manage_vendors' ) ) {
			return;
		}

		$publish_products = 'disallow';
		$manage_customers = 'disallow';

		// check for user publish products capability
		if ( $user->has_cap( 'publish_products' ) ) {
			$publish_products = 'allow';
		}

		// check for create users capability
		if ( $user->has_cap( 'create_users' ) && $user->has_cap( 'edit_users' ) ) {
			$manage_customers = 'allow';
		}

		/**
		 * Optionally override the views/html-edit-user-profile-page.php view: filters must return a string to be passed into include_once.
		 *
		 * @since 2.1.77
		 *
		 * @param string $path Default path to the view.
		 */
		include_once( apply_filters( 'wcpv_edit_user_profile_page_template', 'views/html-edit-user-profile-page.php' ) );

		return true;
	}

	/**
	 * Save vendor section fields to user profile
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param object $user
	 * @return bool
	 */
	public function save_product_vendor_user_profile_section( $user_id ) {
		$publish_products = ! empty( $_POST['wcpv_publish_products'] ) ? sanitize_text_field( $_POST['wcpv_publish_products'] ) : 'disallow';
		$manage_customers = ! empty( $_POST['wcpv_manage_customers'] ) ? sanitize_text_field( $_POST['wcpv_manage_customers'] ) : 'disallow';

		$roles_caps = new WC_Product_Vendors_Roles_Caps;

		// update user capability
		if ( 'disallow' === $publish_products ) {
			$roles_caps->remove_publish_products( $user_id );
		} else {
			$roles_caps->add_publish_products( $user_id );
		}

		// update user capability
		if ( 'disallow' === $manage_customers ) {
			$roles_caps->remove_manage_users( $user_id );
		} else {
			$roles_caps->add_manage_users( $user_id );
		}

		return true;
	}

	/**
	 * Register the commission menu item
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function register_commissions_menu_item() {
		$hook = add_menu_page( __( 'Vendor Commission', 'woocommerce-product-vendors' ), __( 'Commission', 'woocommerce-product-vendors' ), 'manage_vendors', 'wcpv-commissions', array( $this, 'render_commission_page' ), 'dashicons-chart-pie', 56 );

		add_action( "load-$hook", array( $this, 'add_screen_options' ) );

		return true;
	}

	/**
	 * Register the navigation items in the WooCommerce navigation.
	 */
	public function register_navigation_items() {
		/*
		 * WooCommerce 9.3 removed the Navigation feature making this code obsolete.
		 * Registering menu items will only work on stores running WooCommerce 9.2 or older.
		 */
		if ( ! defined( 'WC_VERSION' ) || version_compare( WC_VERSION, '9.3', '>=' ) ) {
			return;
		}

		if (
			! method_exists( Screen::class, 'register_taxonomy' ) ||
			! method_exists( Menu::class, 'add_plugin_item' ) ||
			! method_exists( Menu::class, 'add_plugin_category' ) ||
			! method_exists( Menu::class, 'get_taxonomy_items' )
		) {
			return;
		}

		Menu::add_plugin_category(
			array(
				'id'    => 'wcpv',
				'title' => __( 'Product Vendors', 'woocommerce-product-vendors' ),
			)
		);

		Menu::add_plugin_item(
			array(
				'id'         => 'wcpv-commissions',
				'title'      => __( 'Commission', 'woocommerce-product-vendors' ),
				'capability' => 'manage_vendors',
				'url'        => 'admin.php?page=wcpv-commissions',
				'parent'     => 'wcpv',
				'order'      => 1,
			)
		);

		$vendor_taxonomy_items = Menu::get_taxonomy_items(
			WC_PRODUCT_VENDORS_TAXONOMY,
			array(
				'parent' => 'wcpv',
				'order'  => 2,
			)
		);

		Screen::register_taxonomy( WC_PRODUCT_VENDORS_TAXONOMY );
		Menu::add_plugin_item( $vendor_taxonomy_items['default'] );
	}

	/**
	 * Adds screen options for this page
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function add_screen_options() {
		$option = 'per_page';

		$args = array(
			'label'   => __( 'Commissions', 'woocommerce-product-vendors' ),
			'default' => apply_filters( 'wcpv_commission_list_default_item_per_page', 20 ),
			'option'  => 'commissions_per_page',
		);

		add_screen_option( $option, $args );

		return true;
	}

	/**
	 * Sets screen options for this page
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return mixed
	 */
	public function set_screen_option( $status, $option, $value ) {
		if ( 'commissions_per_page' === $option ) {
			return $value;
		}

		return $status;
	}

	/**
	 * Renders the commission page
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function render_commission_page() {
		$this->maybe_render_bulk_update_notifications();
		$commissions_list = new WC_Product_Vendors_Store_Admin_Commission_List( new WC_Product_Vendors_Commission( new WC_Product_Vendors_PayPal_MassPay ) );

		$commissions_list->prepare_items();
		?>
		<div class="wrap">

			<h2><?php esc_html_e( 'Vendor Commission', 'woocommerce-product-vendors' ); ?>
				<?php
				if ( ! empty( $_REQUEST['s'] ) ) {
					echo '<span class="subtitle">' . esc_html__( 'Search results for', 'woocommerce-product-vendors' ) . ' "' . esc_html( wc_clean( wp_unslash( $_REQUEST['s'] ) ) ) . '"</span>';
				}
				?>
			</h2>

			<?php $commissions_list->views(); ?>

			<form id="wcpv-commission-list" action="" method="get">
				<input type="hidden" name="page" value="wcpv-commissions" />
				<?php $commissions_list->search_box( esc_html__( 'Search Order #', 'woocommerce-product-vendors' ), 'search_id' ); ?>
				<?php $commissions_list->display(); ?>
			</form>
		</div>
		<?php
		return true;
	}

	/**
	 * After performing bulk updates, show a notification to the admin.
	 *
	 * @since  2.1.38
	 * @return void
	 */
	private function maybe_render_bulk_update_notifications() {
		if ( ! empty( $_REQUEST['processed'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$processed               = intval( $_REQUEST['processed'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$items_processed_message = sprintf( _n( '%d item processed.', '%d items processed', $processed, 'woocommerce-product-vendors' ), $processed );
			WC_Admin_Settings::add_message( $items_processed_message );

			if ( ! empty( $_REQUEST['pay'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$paid_status_message = esc_html__( 'Paid status will be updated in a few minutes.', 'woocommerce-product-vendors' );
				WC_Admin_Settings::add_message( $paid_status_message );
			}
			WC_Admin_Settings::show_messages();
		}
	}

	/**
	 * Adds extra vendor field to attachment so we know who the attachment belongs to
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param array $form_fields
	 * @param object $post
	 * @return array $form_fields
	 */
	public function add_attachments_field( $form_fields, $post ) {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			$post_vendor = get_post_meta( $post->ID, '_wcpv_vendor', true );

			$form_fields['vendor'] = array(
				'label' => __( 'Vendor', 'woocommerce-product-vendors' ),
				'input' => 'text',
				'value' => $post_vendor,
			);
		}

		return $form_fields;
	}

	/**
	 * Saves attachment extra fields
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param object $post
	 * @param array $data
	 * @return array $post
	 */
	public function save_attachments_field( $post, $data ) {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			if ( ! empty( $data['vendor'] ) ) {
				// save vendor id to attachment post meta
				update_post_meta( $post['ID'], '_wcpv_vendor', absint( $data['vendor'] ) );
			}
		}

		return $post;
	}

	/**
	 * Add vendor settings section to products tab
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param array $sections existing sections
	 * @return array $sections modified sections
	 */
	public function add_vendor_settings_section( $sections ) {
		$sections['wcpv_vendor_settings'] = __( 'Vendors', 'woocommerce-product-vendors' );

		return $sections;
	}

	/**
	 * Add vendor settings to vendor settings section
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param array $settings existing settings
	 * @param string $current_section current section name
	 * @return array $settings
	 */
	public function add_vendor_settings( $settings, $current_section ) {
		if ( 'wcpv_vendor_settings' === $current_section ) {
			$new_settings = array(
				array(
					'title' => 'Global Settings',
					'desc'  => __( 'Configure global settings, applied for all vendors.', 'woocommerce-product-vendors' ),
					'id'    => 'wcpv_vendor_settings',
					'type'  => 'title',
					'css'   => 'font-size:1.5em',
				),

				array(
					'title' => __( 'Default Commission Configuration', 'woocommerce-product-vendors' ),
					'id'    => 'wcpv_vendor_settings_commission',
					'type'  => 'title',
					'style' => 'color: red;',
					/* translators: %s: URL to vendors page */
					'desc'  => sprintf( __( 'These settings serve as global defaults and can be overridden per vendor or per product. To set vendor-specific commission rates, visit the <a href="%s" target="_blank">Vendors page</a> and edit a vendor.', 'woocommerce-product-vendors' ), admin_url( 'edit-tags.php?taxonomy=wcpv_product_vendors&post_type=product' ) ),
				),

				array(
					'title'    => __( 'Commission Type', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Choose how the commission will be calculated for each product sale.', 'woocommerce-product-vendors' ),
					'desc_tip' => __( 'Percentage: Commission is a percentage of the product price. Fixed: Commission is a set amount regardless of the product price.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_default_commission_type',
					'default'  => 'percentage',
					'type'     => 'select',
					'options'  => array(
						'percentage' => __( 'Percentage of Product Price', 'woocommerce-product-vendors' ),
						'fixed'      => __( 'Fixed Amount per Product Sale', 'woocommerce-product-vendors' ),
					),
					'autoload' => false,
				),

				array(
					'title'             => __( 'Commission Amount', 'woocommerce-product-vendors' ),
					'desc_tip'          => __( 'The default commission amount.', 'woocommerce-product-vendors' ),
					'desc'              => __( 'For percentage: enter 1-100 (no % symbol needed).', 'woocommerce-product-vendors' ),
					'id'                => 'wcpv_vendor_settings_default_commission',
					'default'           => '0',
					'type'              => 'number',
					'custom_attributes' => array(
						'step' => 'any',
						'min'  => '0',
					),
					'autoload'          => false,
				),

				array(
					'type' => 'sectionend',
					'id'   => 'wcpv_vendor_settings_commission',
				),

				array(
					'title' => __( 'Commission Payout Configuration', 'woocommerce-product-vendors' ),
					'id'    => 'wcpv_vendor_settings_payout',
					'type'  => 'title',
					'desc'  => __( 'Define how and when vendors receive their earnings. Set up automated PayPal payouts or manage payments manually.', 'woocommerce-product-vendors' ),
				),

				array(
					'title' => '',
					'id'    => 'wcpv_vendor_settings_payout_notice',
					'type'  => 'title',
					/* translators: %s: URL to review unpaid commissions page */
					'desc'  => '<div class="notice notice-warning inline"><p><strong>' . __( 'Notice:', 'woocommerce-product-vendors' ) . '</strong> ' . sprintf( __( 'Switching Payment Frequency from manual to automatic payouts will trigger immediate payment of all unpaid commissions, starting a new payment cycle today. Before any changes please <a href="%s">review all unpaid commissions</a>.', 'woocommerce-product-vendors' ), admin_url( 'admin.php?page=wcpv-commissions&commission_status=unpaid' ) ) . '</p></div>',
				),

				array(
					'title'    => __( 'Payment Frequency', 'woocommerce-product-vendors' ),
					'desc'     => '',
					'desc_tip' => __( 'Choose between manual payments or automated PayPal schedules. Automatic payouts help maintain consistent vendor payments.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_payout_schedule',
					'default'  => 'manual',
					'type'     => 'select',
					'options'  => array(
						'manual'   => __( 'Manual Processing', 'woocommerce-product-vendors' ),
						'weekly'   => __( 'Weekly PayPal Automatic Payments', 'woocommerce-product-vendors' ),
						'biweekly' => __( 'Bi-Weekly PayPal Automatic Payments', 'woocommerce-product-vendors' ),
						'monthly'  => __( 'Monthly PayPal Automatic Payments', 'woocommerce-product-vendors' ),
					),
					'class'    => 'wc-enhanced-select payment-method-select',
					'css'      => 'min-width: 350px;',
					'autoload' => false,
				),

				array(
					'title'    => __( 'PayPal Integration Mode', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_paypal_masspay_environment',
					'desc'     => __( 'Use Sandbox mode to test your PayPal payout system without processing real transactions.<br>Switch to Live mode when ready to process actual payments.', 'woocommerce-product-vendors' ),
					/* translators: %s: URL to PayPal Developer Portal */
					'desc_tip' => sprintf( __( 'Need API credentials? Get them from the %1$sPayPal Developer Portal%2$s', 'woocommerce-product-vendors' ), '<a href="https://developer.paypal.com/developer/applications/" target="_blank">', '</a>' ),
					'type'     => 'select',
					'default'  => 'sandbox',
					'options'  => array(
						'sandbox' => __( 'Sandbox Mode (Testing)', 'woocommerce-product-vendors' ),
						'live'    => __( 'Live Mode (Production)', 'woocommerce-product-vendors' ),
					),
					'class'    => 'paypal-settings-section paypal-integration-select wc-enhanced-select',
					'autoload' => false,
				),

				array(
					'title'    => __( 'Sandbox API Client ID', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Your PayPal REST API Client ID for testing. Required for sandbox mode.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_paypal_masspay_client_id_sandbox',
					'default'  => '',
					'type'     => 'text',
					'class'    => 'paypal-settings-section',
					'desc_tip' => true,
					'autoload' => false,
				),

				array(
					'title'    => __( 'Sandbox API Secret', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Your PayPal REST API Secret key for testing. Required for sandbox mode.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_paypal_masspay_client_secret_sandbox',
					'default'  => '',
					'type'     => 'password',
					'class'    => 'paypal-settings-section',
					'desc_tip' => true,
					'autoload' => false,
				),

				array(
					'title'    => __( 'Live API Client ID', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Your PayPal REST API Client ID for processing real transactions. Required for live mode.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_paypal_masspay_client_id_live',
					'default'  => '',
					'type'     => 'text',
					'class'    => 'paypal-settings-section',
					'desc_tip' => true,
					'autoload' => false,
				),

				array(
					'title'    => __( 'Live API Secret', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Your PayPal REST API Secret key for processing real transactions. Required for live mode.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_paypal_masspay_client_secret_live',
					'default'  => '',
					'type'     => 'password',
					'class'    => 'paypal-settings-section',
					'desc_tip' => true,
					'autoload' => false,
				),

				array(
					'type' => 'sectionend',
					'id'   => 'wcpv_vendor_settings_payout',
				),

				array(
					'title' => __( 'Store Display Options', 'woocommerce-product-vendors' ),
					'id'    => 'wcpv_vendor_settings_display',
					'type'  => 'title',
					'desc'  => __( 'Control how vendor information appears on your store. These settings affect vendor visibility and branding.', 'woocommerce-product-vendors' ),
				),

				array(
					'title'    => __( 'Vendor Attribution', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Display "Sold By [Vendor Name]" on product pages', 'woocommerce-product-vendors' ),
					'desc_tip' => __( 'Shows customers which vendor sells each product. This helps build trust and allows customers to find more products from vendors they like.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_display_show_by',
					'type'     => 'checkbox',
					'default'  => 'yes',
					'autoload' => false,
				),

				array(
					'title'    => __( 'Vendor Ratings', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Show vendor ratings and review count on store pages', 'woocommerce-product-vendors' ),
					'desc_tip' => __( 'Shows vendor performance metrics to help customers make informed decisions. Displays overall rating and total reviews received.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_vendor_review',
					'type'     => 'checkbox',
					'default'  => 'yes',
					'autoload' => false,
				),

				array(
					'title'    => __( 'Store Branding', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Display logos uploaded by vendors on store pages', 'woocommerce-product-vendors' ),
					'desc_tip' => __( 'Let vendors show their brand identity through their logo. Professional branding helps establish credibility.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_vendor_display_logo',
					'type'     => 'checkbox',
					'default'  => 'yes',
					'autoload' => false,
				),

				array(
					'title'    => __( 'Store Details', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Show vendor profile and store information', 'woocommerce-product-vendors' ),
					'desc_tip' => __( 'Shows vendor descriptions, policies, and other store details written by the vendor. This helps vendors share their story and build customer trust.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_vendor_display_profile',
					'type'     => 'checkbox',
					'default'  => 'yes',
					'autoload' => false,
				),

				array(
					'type' => 'sectionend',
					'id'   => 'wcpv_vendor_settings_display',
				),

				array(
					'title' => __( 'Advanced Configuration', 'woocommerce-product-vendors' ),
					'id'    => 'wcpv_vendor_settings_advanced',
					'type'  => 'title',
					'desc'  => __( 'Technical settings for debugging and system maintenance. Only modify these if you understand their impact.', 'woocommerce-product-vendors' ),
				),

				array(
					'title'    => __( 'System Logging', 'woocommerce-product-vendors' ),
					'desc'     => __( 'Enable detailed event logging', 'woocommerce-product-vendors' ),
					'desc_tip' => __( 'Creates detailed logs of plugin operations for troubleshooting. Warning: Logs may contain sensitive information - enable only when needed and remove logs after debugging.', 'woocommerce-product-vendors' ),
					'id'       => 'wcpv_vendor_settings_logging',
					'type'     => 'checkbox',
					'default'  => 'no',
				),

				array(
					'type' => 'sectionend',
					'id'   => 'wcpv_vendor_settings_advanced',
				),
			);

			$settings = apply_filters( 'wcpv_vendor_settings', $new_settings );
		}

		return $settings;
	}

	/**
	 * Save vendor general/global settings
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function save_vendor_settings( $value, $option, $raw_value ) {
		global $current_section;

		if ( 'wcpv_vendor_settings' !== $current_section ) {
			return $value;
		}

		if ( 'wcpv_vendor_settings_default_commission' === $option['id'] ) {
			return WC_Product_Vendors_Utils::sanitize_commission( $value );
		}

		return $value;
	}

	/**
	 * Add sold by vendor to order item
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function add_sold_by_order_item_detail( $item_id, $item, $product ) {
		$sold_by = get_option( 'wcpv_vendor_settings_display_show_by', 'yes' );

		if ( 'yes' === $sold_by && ! empty( $item['product_id'] ) && WC_Product_Vendors_Utils::is_vendor_product( $item['product_id'] ) ) {

			$sold_by = WC_Product_Vendors_Utils::get_sold_by_link( $item['product_id'] );

			echo '<em class="wcpv-sold-by-order-details">';
			echo apply_filters( 'wcpv_sold_by_text', esc_html__( 'Sold By:', 'woocommerce-product-vendors' ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo ' <a href="' . esc_url( $sold_by['link'] ) . '" title="' . esc_attr( $sold_by['name'] ) . '">' . esc_html( $sold_by['name'] ) . '</a></em>';
		}
	}

	/**
	 * Add a commission field to the product general tab
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function add_product_commission_field_general() {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			global $post;

			$vendor_id        = WC_Product_Vendors_Utils::get_vendor_id_from_product( $post->ID );
			$vendor_data      = WC_Product_Vendors_Utils::get_vendor_data_by_id( $vendor_id );
			$commission_data  = WC_Product_Vendors_Utils::get_product_commission( $post->ID, $vendor_data );

			$commission_placeholder = ! empty( $commission_data['commission'] ) ? $commission_data['commission'] : '';

			$commission_type = __( 'Fixed', 'woocommerce-product-vendors' );

			if ( 'percentage' === $commission_data['type'] ) {
				$commission_type = '%';
			}

			echo '<div class="options_group show_if_simple show_if_variable show_if_booking">';

			woocommerce_wp_text_input( array(
				'id'                => '_wcpv_product_commission',
				'label'             => sprintf( __( 'Commission %s:', 'woocommerce-product-vendors' ), '(' . $commission_type . ')' ),
				'desc_tip'          => 'true',
				'description'       => __( 'Enter a default commission for this product. Enter a positive number.', 'woocommerce-product-vendors' ),
				'placeholder'       => $commission_placeholder,
				'type'              => 'number',
				'custom_attributes' => array( 'step' => 'any', 'min' => '0' ),
			) );

			echo '</div>';
		}

		return true;
	}

	/**
	 * Save the commission field for the product general tab
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param int $post_id
	 * @return bool
	 */
	public function save_product_commission_field_general( $post_id ) {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			if ( empty( $post_id ) ) {
				return;
			}

			$commission = WC_Product_Vendors_Utils::sanitize_commission( wp_unslash( $_POST['_wcpv_product_commission'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			update_post_meta( $post_id, '_wcpv_product_commission', $commission );
		}

		return true;
	}

	/**
	 * Add a commission field to the product variation
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param int $loop
	 * @return bool
	 */
	public function add_product_commission_field_variation( $loop, $variation_data, $variation ) {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			$commission = get_post_meta( $variation->ID, '_wcpv_product_commission', true );

			global $post;

			$vendor_id        = WC_Product_Vendors_Utils::get_vendor_id_from_product( $post->ID );
			$vendor_data      = WC_Product_Vendors_Utils::get_vendor_data_by_id( $vendor_id );
			$commission_data  = WC_Product_Vendors_Utils::get_product_commission( $post->ID, $vendor_data );

			$commission_placeholder = ! empty( $commission_data['commission'] ) ? $commission_data['commission'] : '';

			$commission_type = __( 'Fixed', 'woocommerce-product-vendors' );

			if ( 'percentage' === $commission_data['type'] ) {
				$commission_type = '%';
			}

			echo '<div class="options_group show_if_variable show_if_booking">';
			?>
			<p class="wcpv-commission form-row form-row-first">
				<label><?php echo esc_html__( 'Commission', 'woocommerce-product-vendors' ) . ' (' . esc_html( $commission_type ) . ')'; ?>: <?php echo wc_help_tip( esc_html__( 'Enter a commission for this product variation.  Enter a positive number.', 'woocommerce-product-vendors' ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?></label>

				<input type="number" name="_wcpv_product_variation_commission[<?php echo esc_attr( $loop ); ?>]" value="<?php echo esc_attr( $commission ); ?>" placeholder="<?php echo esc_attr( $commission_placeholder ); ?>" step="any" min="0" />
			</p>
			<?php
			echo '</div>';
		}

		return true;
	}

	/**
	 * Save the commission field for the product variable
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param int $product_id
	 * @return bool
	 */
	public function save_product_commission_field_variable( $product_id ) {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			if ( empty( $product_id ) ) {
				return;
			}

			$commission = WC_Product_Vendors_Utils::sanitize_commission( wp_unslash( $_POST['_wcpv_product_commission'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			update_post_meta( $product_id, '_wcpv_product_commission', $commission );
		}

		return true;
	}

	/**
	 * Save the commission field for the product variation
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param int $variation_id
	 * @param int $i loop count
	 * @return bool
	 */
	public function save_product_commission_field_variation( $variation_id, $i ) {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			if ( empty( $variation_id ) ) {
				return;
			}

			$commission = WC_Product_Vendors_Utils::sanitize_commission( wp_unslash( $_POST['_wcpv_product_variation_commission'][ $i ] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			update_post_meta( $variation_id, '_wcpv_product_commission', $commission );
		}

		return true;
	}

	/**
	 * Add a pass shipping field to the product general tab
	 *
	 * @access  public
	 * @since 2.1.72 Implement _wcpv_customize_product_vendor_settings setting.
	 * @since   2.0.0
	 * @return void
	 * @version 2.0.0
	 *
	 */
	public function add_product_pass_shipping_tax_field_general() {
		/**
		 * @var WC_Product $product_object
		 */
		global $product_object;

		if ( WC_Product_Vendors_Utils::is_vendor() || ! current_user_can( 'manage_vendors' ) ) {
			return;
		}

		$vendor_id        = WC_Product_Vendors_Utils::get_vendor_id_from_product( $product_object->get_id() );
		$vendor_data      = $vendor_id ? WC_Product_Vendors_Utils::get_vendor_data_by_id( $vendor_id ) : null;
		$product_settings = WC_Product_Vendors_Utils::get_product_vendor_settings( $product_object, $vendor_data );

		$customize_product_vendor_settings = $product_object->get_meta( '_wcpv_customize_product_vendor_settings', true, 'edit'  );

		// Backward compatibility.
		// _wcpv_customize_product_vendor_settings is newly added meta key.
		// If this meta key does not exist then
		// Default value depends upon whether product vendor settings override previously.
		if ( $vendor_id ) {
			if ( ! $customize_product_vendor_settings ) {
				$customize_product_vendor_settings = 'yes';

				$pass_shipping = $product_object->get_meta( '_wcpv_product_pass_shipping', true, 'edit' );
				$taxes         = $product_object->get_meta( '_wcpv_product_taxes', true, 'edit' );

				if ( $pass_shipping === $vendor_data['taxes'] && ( empty($taxes) || $taxes === $vendor_data['taxes'] ) ) {
					$customize_product_vendor_settings = 'no';
				}
			}
		}
		?>
		<div id="wcpv-product-vendor-tax-general-settings"
			 class="options_group show_if_simple show_if_variable show_if_booking">
			<h2><?php esc_html_e( 'Product Vendors', 'woocommerce-product-vendors' ); ?></h2>
			<p class="wcpv-vendor-selection-notice">
				<i><?php esc_html_e( 'Please select vendor first to edit settings.', 'woocommerce-product-vendors' ); ?></i>
			</p>
			<p class="form-field wcpv_customize_settings_field">
				<label for="wcpv_customize_product_vendor_settings">
					<?php esc_html_e( 'Override product vendor settings', 'woocommerce-product-vendors' ); ?>
				</label>
				<input type="checkbox" name="_wcpv_customize_product_vendor_settings"
					   id="wcpv_customize_product_vendor_settings"
					   value="yes" <?php checked( 'yes', $customize_product_vendor_settings ); ?> />
			</p>
			<p class="form-field wcpv_product_default_pass_shipping_tax_field">
				<label for="wcpv_product_pass_shipping">
					<?php esc_html_e( 'Pass shipping', 'woocommerce-product-vendors' ) ?>
				</label>
				<input type="checkbox" name="_wcpv_product_pass_shipping" id="wcpv_product_pass_shipping"
					   value="yes" <?php checked( 'yes', $product_settings['pass_shipping'] ); ?> />
				<span class="description">
						<?php esc_html_e( 'Check box to pass the shipping charges for this product to the vendor.',
							'woocommerce-product-vendors' ); ?>
					</span>
			</p>
			<h2><?php esc_html_e( 'Tax Handling', 'woocommerce-product-vendors' ); ?></h2>
			<p class="form-field wcpv_product_taxes_field">
				<label for="wcpv-keep-tax">
					<?php esc_html_e( 'Keep Taxes', 'woocommerce-product-vendors' ); ?>
				</label>
				<input type="radio" value="keep-tax" id="wcpv-keep-tax"
					   name="_wcpv_product_taxes" <?php $vendor_id && checked( 'keep-tax', $product_settings['taxes'] ); ?>/>
				<span class="description">
						<?php esc_html_e( 'Calculate commission based on product price only.',
							'woocommerce-product-vendors' ); ?>
					</span>
			</p>
			<p class="form-field wcpv_product_taxes_field">
				<label for="wcpv-pass-tax">
					<?php esc_html_e( 'Pass Taxes', 'woocommerce-product-vendors' ); ?>
				</label>
				<input type="radio" value="pass-tax" id="wcpv-pass-tax"
					   name="_wcpv_product_taxes" <?php $vendor_id && checked( 'pass-tax', $product_settings['taxes'] ); ?>/>
				<span class="description">
						<?php esc_html_e( 'All tax charges will be included in the vendor\'s commission.',
							'woocommerce-product-vendors' ); ?>
					</span>
			</p>
			<p class="form-field wcpv_product_taxes_field">
				<label for="wcpv-split-tax">
					<?php esc_html_e( 'Split Taxes', 'woocommerce-product-vendors' ); ?>
				</label>
				<input type="radio" value="split-tax" id="wcpv-split-tax"
					   name="_wcpv_product_taxes" <?php $vendor_id && checked( 'split-tax', $product_settings['taxes'] ); ?>/>
				<span class="description">
						<?php esc_html_e( 'The full price including taxes will be used to calculate commission.',
							'woocommerce-product-vendors' ); ?>
					</span>
			</p>
		</div>
		<?php
	}

	/**
	 * Save the pass shipping field for the product general tab
	 *
	 * @since 2.1.72 Implement logic to save _wcpv_customize_product_vendor_settings and _wcpv_product_taxes setting.
	 * @since   2.0.0
	 *
	 * @param int $post_id
	 *
	 * @version 2.0.0
	 */
	public function save_product_pass_shipping_tax_field_general( $post_id ) {
		if ( empty( $post_id ) ) {
			return;
		}

		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			$product = wc_get_product( $post_id );
			$product->update_meta_data(
				'_wcpv_product_pass_shipping',
				! empty( $_POST['_wcpv_product_pass_shipping'] ) ? 'yes' : 'no'
			);
			$product->update_meta_data(
				'_wcpv_product_taxes',
				wc_clean( wp_unslash( $_POST['_wcpv_product_taxes'] ?? '' ) )
			);
			$product->update_meta_data(
				'_wcpv_product_split_tax',
				! empty( $_POST['_wcpv_product_split_tax'] ) ? 'yes' : 'no'
			);
			$product->update_meta_data(
				'_wcpv_customize_product_vendor_settings',
				! empty( $_POST['_wcpv_customize_product_vendor_settings'] ) ? 'yes' : 'no'
			);
			$product->save();
		}
	}

	/**
	 * Adds bulk edit action for vendors
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function add_variation_vendor_bulk_edit() {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			?>
			<optgroup label="<?php esc_attr_e( 'Vendor', 'woocommerce-product-vendors' ); ?>">
				<option value="variable_vendor_commission"><?php esc_html_e( 'Commission', 'woocommerce-product-vendors' ); ?></option>
			</optgroup>
			<?php
		}
	}

	/**
	 * Ajax search for vendors
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return json $found_vendors
	 */
	public function vendor_search_ajax() {
		$nonce = wc_clean( wp_unslash( $_GET['security'] ) );

		// bail if nonce don't check out
		if ( ! wp_verify_nonce( $nonce, '_wcpv_vendor_search_nonce' ) ) {
			wp_die( esc_html__( 'Cheatin&#8217; huh?', 'woocommerce-product-vendors' ) );
		}

		if ( empty( $_GET['term'] ) ) {
			wp_die( esc_html__( 'Cheatin&#8217; huh?', 'woocommerce-product-vendors' ) );
		}

		$term = (string) wc_clean( stripslashes( $_GET['term']['term'] ) );
		$args = array(
			'hide_empty' => false,
			'name__like' => $term,
		);

		$vendor_terms = get_terms( WC_PRODUCT_VENDORS_TAXONOMY, $args ); // phpcs:ignore

		$found_vendors = array();

		if ( ! empty( $vendor_terms ) ) {
			foreach ( $vendor_terms as $term ) {
				$found_vendors[ $term->term_id ] = $term->name;
			}
		}

		wp_send_json( $found_vendors );
	}

	/**
	 * Add pass shipping/tax setting to product bulk edit menu
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return bool
	 */
	public function add_product_bulk_edit_pass_shipping_tax() {
		?>
		<label>
			<span class="title"><?php esc_html_e( 'Pass Shipping to Vendor?', 'woocommerce-product-vendors' ); ?></span>
			<span class="input-text-wrap">
					<select class="pass-shipping-tax" name="_wcpv_product_pass_shipping">
					<?php
					$options = array(
						''    => __( '— No Change —', 'woocommerce-product-vendors' ),
						'yes' => __( 'Yes', 'woocommerce-product-vendors' ),
						'no'  => __( 'No', 'woocommerce-product-vendors' ),
					);

					foreach ( $options as $key => $value ) {
						echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $value ) . '</option>';
					}
					?>
				</select>
			</span>
		</label>
		<label>
			<span class="title"><?php esc_html_e( 'Tax Handling', 'woocommerce-product-vendors' ); ?></span>
			<span class="input-text-wrap">
					<select class="pass-shipping-tax" name="_wcpv_product_taxes">
					<?php
					$options = array(
						''          => __( '— No Change —', 'woocommerce-product-vendors' ),
						'keep-tax'  => __( 'Keep Taxes', 'woocommerce-product-vendors' ),
						'pass-tax'  => __( 'Pass Taxes', 'woocommerce-product-vendors' ),
						'split-tax' => __( 'Split Taxes', 'woocommerce-product-vendors' ),
					);

					foreach ( $options as $key => $value ) {
						echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $value ) . '</option>';
					}
					?>
				</select>
			</span>
		</label>
		<label>
			<span class="title"><?php esc_html_e( 'Inclined Tax in commission?', 'woocommerce-product-vendors' ); ?></span>
			<span class="input-text-wrap">
					<select class="pass-shipping-tax" name="_wcpv_product_split_tax">
					<?php
					$options = array(
						''    => __( '— No Change —', 'woocommerce-product-vendors' ),
						'yes' => __( 'Yes', 'woocommerce-product-vendors' ),
						'no'  => __( 'No', 'woocommerce-product-vendors' ),
					);

					foreach ( $options as $key => $value ) {
						echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $value ) . '</option>';
					}
					?>
				</select>
			</span>
		</label>
		<?php
	}

	/**
	 * Filters the order item meta label without underscore.
	 *
	 * @since 2.1.8 User correct WC core filter.
	 * @param string $key
	 * @return bool
	 */
	public function filter_order_attribute_label( $key ) {
		if ( '_fulfillment_status' === $key ) {
			return __( 'Fulfillment Status', 'woocommerce-product-vendors' );
		}

		if ( '_commission_status' === $key ) {
			return __( 'Commission Status', 'woocommerce-product-vendors' );
		}

		return $key;
	}

	/**
	 * Filters the order item meta value.
	 *
	 * @since 2.1.8
	 * @param string $value
	 * @param object $meta
	 * @return bool
	 */
	public function filter_order_attribute_value( $value, $meta ) {
		if ( 'unfulfilled' === $value ) {
			return __( 'Unfulfilled', 'woocommerce-product-vendors' );
		}

		if ( 'fulfilled' === $value ) {
			return __( 'Fulfilled', 'woocommerce-product-vendors' );
		}

		if ( 'paid' === $value ) {
			return __( 'Paid', 'woocommerce-product-vendors' );
		}

		if ( 'unpaid' === $value ) {
			return __( 'Unpaid', 'woocommerce-product-vendors' );
		}

		return $value;
	}

	/**
	 * Save pass shipping/tax setting to product bulk edit
	 *
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param object $product
	 */
	public function save_product_bulk_edit_pass_shipping_tax( $product ) {
		if ( ! WC_Product_Vendors_Utils::is_vendor() && current_user_can( 'manage_vendors' ) ) {
			if ( empty( $product ) ) {
				return;
			}

			if ( ! empty( $_REQUEST['_wcpv_product_pass_shipping'] ) ) {
				$product->update_meta_data(
					'_wcpv_product_pass_shipping',
					wc_clean( wp_unslash( $_REQUEST['_wcpv_product_pass_shipping'] ) )
				);
			}
			if ( ! empty( $_REQUEST['_wcpv_product_taxes'] ) ) {
				$product->update_meta_data(
					'_wcpv_product_taxes',
					wc_clean( wp_unslash( $_POST['_wcpv_product_taxes'] ) )
				);
			}
			if ( ! empty( $_REQUEST['_wcpv_product_split_tax'] ) ) {
				$product->update_meta_data(
					'_wcpv_product_split_tax',
					wc_clean( wp_unslash( $_REQUEST['_wcpv_product_split_tax'] ) )
				);
			}
			$product->save();
		}

		return;
	}

	/**
	 * Handles saving of the vendor to product.
	 *
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param int $post_id
	 * @return bool
	 */
	public function save_product_vendor( $post_id ) {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		// don't continue if it is bulk/quick edit
		if ( ! empty( $_REQUEST['woocommerce_quick_edit'] ) || ! empty( $_REQUEST['woocommerce_bulk_edit'] ) || ! isset( $_POST['wcpv_product_term'] ) ) {
			return;
		}

		// if not a product bail
		if ( 'product' !== get_post_type( $post_id ) ) {
			return;
		}

		$term = ! empty( $_POST['wcpv_product_term'] ) ? absint( $_POST['wcpv_product_term'] ) : '';

		wp_set_object_terms( $post_id, $term, WC_PRODUCT_VENDORS_TAXONOMY );

		return true;
	}

	/**
	 * Add vendor selection on quick and bulk edit
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param string $column_name the name of the column to add it to
	 * @param string $post_type
	 * @return bool
	 */
	public function quick_edit( $column_name, $post_type ) {
		if ( WC_Product_Vendors_Utils::is_vendor() || ! current_user_can( 'manage_vendors' ) ) {
			return;
		}

		if ( 'taxonomy-wcpv_product_vendors' !== $column_name || 'product' !== $post_type ) {
			return;
		}

		$args = array(
			'hide_empty'   => false,
			'hierarchical' => false,
		);

		$terms = get_terms( WC_PRODUCT_VENDORS_TAXONOMY, $args ); // phpcs:ignore

		if ( ! empty( $terms ) ) {
			$output = '<fieldset class="inline-edit-col-center"><div class="inline-edit-group"><label class="alignleft"><span class="title">' . esc_html__( 'Vendors', 'woocommerce-product-vendors' ) . '</span>';

			$output .= '<select class="wcpv-product-vendor-terms-dropdown" name="wcpv_qe_product_term">';

			$output .= '<option value="no">' . esc_html__( 'No Change', 'woocommerce-product-vendors' ) . '</option>';
			$output .= '<option value="novendor">' . esc_html__( 'No Vendor', 'woocommerce-product-vendors' ) . '</option>';

			foreach ( $terms as $term ) {
				$output .= '<option value="' . esc_attr( $term->term_id ) . '">' . esc_html( $term->name ) . '</option>';
			}

			$output .= '</select>';

			$output .= '</label></div></fieldset>';

			echo $output; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}

	/**
	 * Handles quick and bulk edit saves
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @param int $post_id
	 * @param object $post
	 * @return int
	 */
	public function bulk_and_quick_edit_save_post( $post_id, $post ) {
		if ( WC_Product_Vendors_Utils::is_vendor() || ! current_user_can( 'manage_vendors' ) ) {
			return $post_id;
		}

		// If this is an autosave, our form has not been submitted, so we don't want to do anything.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return $post_id;
		}

		// Don't save revisions and autosaves
		if ( wp_is_post_revision( $post_id ) || wp_is_post_autosave( $post_id ) ) {
			return $post_id;
		}

		// Check post type is product
		if ( 'product' !== $post->post_type ) {
			return $post_id;
		}

		// Check user permission
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return $post_id;
		}

		if ( empty( $_REQUEST['wcpv_qe_product_term'] ) || 'no' === $_REQUEST['wcpv_qe_product_term'] ) {
			return $post_id;
		}

		$term = ! empty( $_REQUEST['wcpv_qe_product_term'] ) ? absint( $_REQUEST['wcpv_qe_product_term'] ) : '';

		if ( 'novendor' === $term ) {
			$term = '';
		}

		// check if it is a quick edit or bulk edit
		if ( ! empty( $_REQUEST['woocommerce_quick_edit'] ) ) {
			// update the product term
			wp_set_object_terms( $post_id, $term, WC_PRODUCT_VENDORS_TAXONOMY );

			// Clear transient
			wc_delete_product_transients( $post_id );

		} elseif ( ! empty( $_REQUEST['woocommerce_bulk_edit'] ) && ! empty( $_REQUEST['post'] ) ) {
			foreach ( array_map( 'absint', wp_unslash( $_REQUEST['post'] ) ) as $post ) {
				// update the product term
				wp_set_object_terms( $post, $term, WC_PRODUCT_VENDORS_TAXONOMY );

				// Clear transient
				wc_delete_product_transients( $post );
			}
		}

		return $post_id;
	}

	/**
	 * Generates the CSV ( commissions ) download of current view
	 *
	 * @access public
	 * @since 2.0.0
	 * @version 2.0.0
	 * @return array $query
	 */
	public function export_commissions_ajax() {
		$order_id          = ! empty( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : '';
		$year              = ! empty( $_POST['year'] ) ? sanitize_text_field( $_POST['year'] ) : '';
		$month             = ! empty( $_POST['month'] ) ? sanitize_text_field( $_POST['month'] ) : '';
		$commission_status = ! empty( $_POST['commission_status'] ) ? sanitize_text_field( $_POST['commission_status'] ) : '';
		$vendor_id         = ! empty( $_POST['vendor'] ) ? absint( $_POST['vendor'] ) : '';
		$nonce             = ! empty( $_POST['nonce'] ) ? sanitize_text_field( $_POST['nonce'] ) : '';

		// bail if nonce don't check out
		if ( ! wp_verify_nonce( $nonce, '_wcpv_export_commissions_nonce' ) ) {
			wp_die( esc_html__( 'Cheatin&#8217; huh?', 'woocommerce-product-vendors' ) );
		}

		$commission = new WC_Product_Vendors_Commission( new WC_Product_Vendors_PayPal_MassPay );

		$query = $commission->csv_filtered_query( $order_id, $year, $month, $commission_status, $vendor_id );

		// This will end up as URIencoded CSV file content that will be pushed as an anchor's href via jQuery.
		echo $query; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

	/**
	 * Handles export of unpaid commissions
	 *
	 * @access public
	 * @since 2.0.6
	 * @version 2.0.6
	 * @return bool
	 */
	public function export_unpaid_commissions_ajax() {
		$nonce = ! empty( $_POST['nonce'] ) ? sanitize_text_field( $_POST['nonce'] ) : '';

		// bail if nonce don't check out
		if ( ! wp_verify_nonce( $nonce, '_wcpv_export_unpaid_commissions_nonce' ) ) {
			wp_die( esc_html__( 'Cheatin&#8217; huh?', 'woocommerce-product-vendors' ) );
		}

		$currency = get_woocommerce_currency();
		$commission = new WC_Product_Vendors_Commission( new WC_Product_Vendors_PayPal_MassPay );

		$unpaid_commissions = $commission->get_unpaid_commission_data();

		$commissions = array();

		foreach ( $unpaid_commissions as $commission ) {
			if ( ! isset( $commissions[ $commission->vendor_id ] ) ) {
				$commissions[ $commission->vendor_id ] = wc_format_decimal( 0, 2 );
			}

			$commissions[ $commission->vendor_id ] += wc_format_decimal( $commission->total_commission_amount, 2 );
		}

		$payout_note = apply_filters( 'wcpv_export_unpaid_commissions_note', sprintf( __( 'Total commissions earned from %1$s as of %2$s on %3$s', 'woocommerce-product-vendors' ), get_bloginfo( 'name', 'display' ), date( 'H:i:s' ), date( 'd-m-Y' ) ) );

		$commissions_data = array();

		foreach ( $commissions as $vendor_id => $total ) {
			$vendor = WC_Product_Vendors_Utils::get_vendor_data_by_id( $vendor_id );

			$recipient = $vendor['name'];

			if ( ! empty( $vendor['paypal'] ) ) {
				$recipient = $vendor['paypal'];
			}

			$commissions_data[] = array(
				$recipient,
				$total,
				$currency,
				$vendor_id,
				$payout_note,
			);
		}

		// prepare CSV
		$headers = array(
			'Recipient',
			'Payment',
			'Currency',
			'Customer ID',
			'Note',
		);

		array_unshift( $commissions_data, $headers );

		// convert the array to string recursively
		$commissions_data = implode( PHP_EOL, array_map( array( 'WC_Product_Vendors_Utils', 'convert2string' ), $commissions_data ) );

		echo $commissions_data; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

	/**
	 * Add debug tool button.
	 *
	 * @since 2.0.0
	 * @since 2.2.0 Update "wcpv_clear_transients" action callback.
	 * @version 2.0.35
	 * @return array $tools
	 */
	public function add_debug_tool( $tools ) {
		$tools['wcpv_clear_transients'] = array(
			'name'     => __( 'Product Vendors Transients', 'woocommerce-product-vendors' ),
			'button'   => __( 'Clear all transients/cache', 'woocommerce-product-vendors' ),
			'desc'     => __( 'This will clear all Product Vendors related transients/caches such as reports.', 'woocommerce-product-vendors' ),
			'callback' => [ $this, 'clear_all_transients' ],
		);

		$tools['wcpv_delete_webhook'] = array(
			'name'     => __( 'Product Vendors Delete Webhook', 'woocommerce-product-vendors' ),
			'button'   => __( 'Delete Webhook', 'woocommerce-product-vendors' ),
			'desc'     => __( 'Troubleshoot PayPal Payouts. Delete current webhook and receive new webhook id.', 'woocommerce-product-vendors' ),
			'callback' => 'WC_Product_Vendors_Utils::delete_paypal_webhook_id',
		);

		return $tools;
	}

	/**
	 * Clears all report transients
	 *
	 * @access public
	 * @since 2.0.0
	 * @return bool
	 * @version 2.0.0
	 */
	public function clear_reports_transients() {
		WC_Product_Vendors_Utils::clear_reports_transients();

		return true;
	}

	/**
	 * Delete all transients created by this add-on.
	 *
	 * This function handles "wcpv_clear_transients" action.
	 *
	 * Note: This function is for internal usage.
	 *
	 * @since 2.2.0
	 * @return void
	 */
	public function clear_all_transients() {
		$vendor_ids = get_terms( // phpcs:ignore
			WC_PRODUCT_VENDORS_TAXONOMY,
			[
				'fields'     => 'ids',
				'hide_empty' => false
			]
		);

		if ( ! $vendor_ids ) {
			return;
		}

		foreach ( $vendor_ids as $vendor_id ) {
			$transient_manager = WC_Product_Vendor_Transient_Manager::make( (int) $vendor_id );
			$transient_manager->delete();
		}

		if ( class_exists( 'WC_Bookings' ) ) {
			WC_Bookings_Cache::clear_cache();
		}

		WC_Product_Vendors_Utils::clear_vendor_error_list_transient();
	}

	/**
	 * Maybe show a migrate admin storage notice.
	 */
	public function migrate_admin_storage_notice() {
		// only show to admin users
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		if ( WC_Product_Vendors_Utils::is_vendor_admin_meta_storage_enabled() ) {
			return;
		}

		$admin_storage     = new WC_Product_Vendors_Admin_Storage_Compatibility();
		$migration_started = $admin_storage->is_migration_scheduled();

		if ( isset( $_POST['wcpv_migrate_admin_storage_migrate'] ) && ! $migration_started ) {
			$migration_started = $admin_storage->schedule_migration();
		}

		include_once( 'views/html-migrate-admin-storage-notice.php' );
	}
}

WC_Product_Vendors_Store_Admin::init();
