<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class WC_Product_Vendors_Order_Email_To_Vendor extends WC_Email {
	/**
	 * Vendor ID
	 *
	 * @var int
	 */
	public $vendor;

	/**
	 * Recipients for the email.
	 *
	 * @var string
	 */
	public $recipient = '';

	/**
	 * Constructor
	 *
	 * @since 2.1.70 Move subscription renewal email notification related code to WC_Product_Vendors_Order_Email_To_Vendor.
	 *
	 * @access public
	 * @return bool
	 */
	public function __construct() {
		$this->id               = 'order_email_to_vendor';
		$this->title            = __( 'Order Email (Vendor)', 'woocommerce-product-vendors' );
		$this->description      = __( 'When an order is placed with vendor products, this email will be sent out to the vendors.', 'woocommerce-product-vendors' );

		$this->heading          = __( 'New Customer Order', 'woocommerce-product-vendors' );
		$this->subject          = __( '[{site_title}] New customer order ({order_number}) - {order_date}', 'woocommerce-product-vendors' );

		$this->template_base    = WC_PRODUCT_VENDORS_TEMPLATES_PATH;
		$this->template_html    = 'emails/order-email-to-vendor.php';
		$this->template_plain   = 'emails/plain/order-email-to-vendor.php';

		// Triggers for this email
		add_action( 'woocommerce_order_status_pending_to_processing_notification', array( $this, 'trigger' ) );
		add_action( 'woocommerce_order_status_pending_to_completed_notification', array( $this, 'trigger' ) );
		add_action( 'woocommerce_order_status_pending_to_on-hold_notification', array( $this, 'trigger' ) );
		add_action( 'woocommerce_order_status_failed_to_processing_notification', array( $this, 'trigger' ) );
		add_action( 'woocommerce_order_status_failed_to_completed_notification', array( $this, 'trigger' ) );
		add_action( 'woocommerce_order_status_failed_to_on-hold_notification', array( $this, 'trigger' ) );

		// Add notification for deposits orders.
		add_action( 'woocommerce_order_status_pending_to_partial-payment_notification', array( $this, 'trigger' ) );

		// Call parent constructor
		parent::__construct();

		return true;
	}

	/**
	 * Function to append email ID and phone number to address.
	 *
	 * @param string $address     Formatted address.
	 * @param array  $raw_address Raw address.
	 *
	 * @return mixed|string
	 */
	public function formatted_address( $address, $raw_address ) {

		/**
		 * Filters if email and phone of the customer should be removed from vendor email.
		 *
		 * @since 2.1.62
		 *
		 * @param bool $remove Remove email and phone from email.
		 */
		if ( ! apply_filters( 'wcpv_order_email_to_vendor_remove_email_phone', false ) ) {
			if ( array_key_exists( 'phone', $raw_address ) && ! empty( $raw_address['phone'] ) ) {
				$address .= '<br/>' . wc_make_phone_clickable( $raw_address['phone'] );
			}
			if ( array_key_exists( 'email', $raw_address ) && ! empty( $raw_address['email'] ) ) {
				$address .= '<br/>' . esc_html( $raw_address['email'] );
			}
		}

		return $address;
	}

	/**
	 * trigger function.
	 *
	 * @access public
	 * @param int $order_id
	 * @return bool
	 */
	public function trigger( $order_id ) {
		if ( ! $this->is_enabled() ) {
			return;
		}

		if ( $order_id ) {

			$this->object = wc_get_order( $order_id );
			$order_date   = $this->object->get_date_created();

			$this->find['order-date']      = '{order_date}';
			$this->find['order-number']    = '{order_number}';

			$this->replace['order-date']   = date_i18n( wc_date_format(), strtotime( $order_date ) );
			$this->replace['order-number'] = $this->object->get_order_number();

			if ( is_a( $this->object, 'WC_Order' ) ) {

				$vendors = WC_Product_Vendors_Utils::get_vendors_from_order( $this->object );

				add_filter( 'woocommerce_get_order_item_totals', array( $this, 'filter_order_totals' ), 10, 2 );

				add_filter( 'woocommerce_email_customer_details_fields', array( $this, 'filter_customer_fields' ), 10, 3 );

				add_filter( 'wc_get_template', array( $this, 'filter_customer_addresses' ), 10, 3 );

				add_action( 'wc_product_vendors_email_order_meta', array( $this, 'show_commission_information' ), 10 , 4 );

				add_filter( 'woocommerce_order_get_formatted_billing_address', array( $this, 'formatted_address' ), 10, 2 );

				$sent = false;

				// Send email to each vendor.
				foreach ( $vendors as $vendor_id => $data ) {
					/**
					 * Shall we send an order email to this vendor?
					 * Defaults to true, but filters can override based on data e.g. the vendor term or order data.
					 *
					 * @since 2.1.80
					 *
					 * @param bool $send_email Send email to vendor?
					 * @param array $data Vendor term data.
					 * @param WC_Order $this->object Order object.
					 */
					if ( ! apply_filters( 'wcpv_email_order_details_to_vendor', true, $data, $this->object ) ) {
						continue;
					}

					$this->vendor = $vendor_id;

					$this->recipient = ! empty( $data['email'] ) ? $data['email'] : '';

					if ( empty( $this->recipient ) ) {
						continue;
					}

					$sent = $this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
				}

				remove_filter( 'woocommerce_order_get_formatted_billing_address', array( $this, 'formatted_address' ) );

				remove_action( 'wc_product_vendors_email_order_meta', array( $this, 'show_commission_information' ), 10 );

				remove_filter( 'wc_get_template', array( $this, 'filter_customer_addresses' ), 10 );

				remove_filter( 'woocommerce_email_customer_details_fields', array( $this, 'filter_customer_fields' ), 10 );

				remove_filter( 'woocommerce_get_order_item_totals', array( $this, 'filter_order_totals' ), 10 );

				if ( $sent ) {
					// add order note that email was sent to vendor
					$note = __( 'New Order email sent to vendor(s).', 'woocommerce-product-vendors' );

					$this->object->add_order_note( $note );
				}
			}
		}

		return true;
	}

	/**
	 * get_content_html function.
	 *
	 * @access public
	 * @return string
	 */
	public function get_content_html() {
		ob_start();
		wc_get_template( $this->template_html, array(
			'order'         => $this->object,
			'this_vendor'   => $this->vendor,
			'email_heading' => $this->get_heading(),
			'sent_to_admin' => true,
			'plain_text'    => false,
			'email'         => $this,
		), 'woocommerce-product-vendors/', $this->template_base );

		return ob_get_clean();
	}

	/**
	 * get_content_plain function.
	 *
	 * @access public
	 * @return string
	 */
	public function get_content_plain() {
		ob_start();
		wc_get_template( $this->template_plain, array(
			'order'         => $this->object,
			'this_vendor'   => $this->vendor,
			'email_heading' => $this->get_heading(),
			'sent_to_admin' => true,
			'plain_text'    => true,
			'email'         => $this,
		), 'woocommerce-product-vendors/', $this->template_base );

		return ob_get_clean();
	}

	/**
	 * Filters the order totals
	 *
	 * @access public
	 * @param array $total_rows;
	 * @param object $order
	 * @return string
	 */
	public function filter_order_totals( $total_rows, $order ) {
		// don't show payment method to vendors
		unset( $total_rows['payment_method'] );
		unset( $total_rows['order_total'] );

		return $total_rows;
	}

	/**
	 * Filters the customer fields
	 *
	 * @access public
	 * @param array $fields
	 * @param bool $sent_to_admin
	 * @param object $order
	 * @return string
	 */
	public function filter_customer_fields( $fields, $sent_to_admin, $order ) {
		if ( ! apply_filters( 'wcpv_order_email_to_vendor_remove_email_phone', false ) ) {
			return $fields;
		}

		unset( $fields['billing_phone'], $fields['billing_email'] );

		return $fields;
	}

	/**
	 * Filters the customer addresses
	 *
	 * @access public
	 * @param array $template_path the path of the original template
	 * @param string $template_name the name to the original template
	 * @param array $args
	 * @return string
	 */
	public function filter_customer_addresses( $template_path, $template_name, $args ) {
		if ( 'emails/plain/email-addresses.php' !== $template_name && 'emails/email-addresses.php' !== $template_name ) {
			return $template_path;
		}

		if ( 'html' === $this->get_email_type() ) {
			$template_path = WC_PRODUCT_VENDORS_TEMPLATES_PATH . 'emails/order-email-addresses-to-vendor.php';
		} elseif ( 'plain' === $this->get_email_type() ) {
			$template_path = WC_PRODUCT_VENDORS_TEMPLATES_PATH . 'emails/plain/order-email-addresses-to-vendor.php';
		}

		return $template_path;
	}

	/**
	 * Renders the order table
	 *
	 * @since 2.0.2
	 * @version 2.0.2
	 * @access public
	 * @param object $order
	 * @param bool $sent_to_admin
	 * @param bool $plain_text
	 * @param object $email
	 * @return bool
	 */
	public function render_order_details_table( $order, $sent_to_admin, $plain_text, $email ) {
		if ( $plain_text ) {
			wc_get_template( 'emails/plain/email-order-details.php', array(
				'order'         => $order,
				'this_vendor'   => $this->vendor,
				'sent_to_admin' => $sent_to_admin,
				'plain_text'    => $plain_text,
				'email'         => $email,
			), 'woocommerce-product-vendors/', $this->template_base );
		} else {
			wc_get_template( 'emails/email-order-details.php', array(
				'order'         => $order,
				'this_vendor'   => $this->vendor,
				'sent_to_admin' => $sent_to_admin,
				'plain_text'    => $plain_text,
				'email'         => $email,
			), 'woocommerce-product-vendors/', $this->template_base );
		}

		return true;
	}

	/**
	 * Initialize Settings Form Fields
	 *
	 * @access public
	 * @return bool
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled' => array(
				'title'         => __( 'Enable/Disable', 'woocommerce-product-vendors' ),
				'type'          => 'checkbox',
				'label'         => __( 'Enable this email notification', 'woocommerce-product-vendors' ),
				'default'       => 'yes',
			),
			'subject' => array(
				'title'         => __( 'Subject', 'woocommerce-product-vendors' ),
				'type'          => 'text',
				/* translators: email subject */
				'description'   => sprintf( __( 'This controls the email subject line. Leave blank to use the default subject: <code>%s</code>.', 'woocommerce-product-vendors' ), $this->subject ),
				'placeholder'   => '',
				'default'       => '',
			),
			'heading' => array(
				'title'         => __( 'Email Heading', 'woocommerce-product-vendors' ),
				'type'          => 'text',
				/* translators: email heading */
				'description'   => sprintf( __( 'This controls the main heading contained within the email notification. Leave blank to use the default heading: <code>%s</code>.', 'woocommerce-product-vendors' ), $this->heading ),
				'placeholder'   => '',
				'default'       => '',
			),
			'email_type' => array(
				'title'         => __( 'Email type', 'woocommerce-product-vendors' ),
				'type'          => 'select',
				'description'   => __( 'Choose which format of email to send.', 'woocommerce-product-vendors' ),
				'default'       => 'html',
				'class'         => 'email_type wc-enhanced-select',
				'options'       => $this->get_email_type_options(),
			),
		);

		return true;
	}

	/**
	 * Show the commission details on the order.
	 *
	 * @since 2.1.0
	 * @version 2.1.3
	 *
	 * @param WC_Order $order
	 * @param Boolean $sent_to_admin
	 * @param Boolean $plain_text
	 * @param String $email
	 */
	public function show_commission_information( $order, $sent_to_admin, $plain_text, $email ) {
		$commission_manager = new WC_Product_Vendors_Commission( new WC_Product_Vendors_PayPal_MassPay() );
		$commission         = $commission_manager->get_vendor_earned_commission_by_order_id( $this->vendor, $order->get_id() );

		if ( empty( $commission ) ) {
			return;
		}

		$show_commission = apply_filters( 'wcpv_email_to_vendor_show_commission', true, $order, $commission );

		if ( ! $show_commission ) {
			return;
		}

		?>
		<p>
			<?php
			/* translators: the amount in commission price */
			echo sprintf( esc_html__( 'Your commission for this order is %s.', 'woocommerce-product-vendors' ), wc_price( $commission, array( 'currency' => get_woocommerce_currency() ) ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			?>
		</p>
		<?php
	}
}
