<?php

/**
 * Coupon section of the plugin
 *
 * @link           
 *
 * @package  Wt_Import_Export_For_Woo 
 */
if (!defined('ABSPATH')) {
    exit;
}

class Wt_Import_Export_For_Woo_Coupon {

    public $module_id = '';
    public static $module_id_static = '';
    public $module_base = 'coupon';
    public $module_name = 'Coupon Import Export for WooCommerce';
    public $min_base_version= '1.0.0'; /* Minimum `Import export plugin` required to run this add on plugin */

    private $importer = null;
    private $exporter = null;
    private $all_meta_keys = array();
    private $found_meta = array();
    private $found_hidden_meta = array();
	private $selected_column_names = null;

    public function __construct()
    {
        /**
        *   Checking the minimum required version of `Import export plugin` plugin available
        */
        if(!Wt_Import_Export_For_Woo_Common_Helper::check_base_version($this->module_base, $this->module_name, $this->min_base_version))
        {
            return;
        }
        if(!function_exists('is_plugin_active'))
        {
            include_once(ABSPATH.'wp-admin/includes/plugin.php');
        }
        if(!is_plugin_active('woocommerce/woocommerce.php'))
        {
            return;
        }

        $this->module_id = Wt_Import_Export_For_Woo::get_module_id($this->module_base);
        self::$module_id_static = $this->module_id;
       
        add_filter('wt_iew_exporter_post_types', array($this, 'wt_iew_exporter_post_types'), 10, 1);
        add_filter('wt_iew_importer_post_types', array($this, 'wt_iew_exporter_post_types'), 10, 1);

        add_filter('wt_iew_exporter_alter_filter_fields', array($this, 'exporter_alter_filter_fields'), 10, 3);
        
        add_filter('wt_iew_exporter_alter_mapping_fields', array($this, 'exporter_alter_mapping_fields'), 10, 3);        
        add_filter('wt_iew_importer_alter_mapping_fields', array($this, 'get_importer_post_columns'), 10, 3);  
        
        add_filter('wt_iew_importer_alter_advanced_fields', array($this, 'importer_alter_advanced_fields'), 10, 3);

        add_filter('wt_iew_exporter_alter_meta_mapping_fields', array($this, 'exporter_alter_meta_mapping_fields'), 10, 3);
        add_filter('wt_iew_importer_alter_meta_mapping_fields', array($this, 'importer_alter_meta_mapping_fields'), 10, 3);

        add_filter('wt_iew_exporter_alter_mapping_enabled_fields', array($this, 'exporter_alter_mapping_enabled_fields'), 10, 3);
        add_filter('wt_iew_importer_alter_mapping_enabled_fields', array($this, 'exporter_alter_mapping_enabled_fields'), 10, 3);

        add_filter('wt_iew_exporter_do_export', array($this, 'exporter_do_export'), 10, 7);
        add_filter('wt_iew_importer_do_import', array($this, 'importer_do_import'), 10, 8);

        add_filter('wt_iew_importer_steps', array($this, 'importer_steps'), 10, 2);
		
		
		add_action('admin_footer-edit.php', array($this, 'add_coupons_bulk_actions'));
        add_action('load-edit.php', array($this, 'process_coupons_bulk_actions'));
    }


    public function add_coupons_bulk_actions() {
        global $post_type, $post_status;

        if ($post_type == 'shop_coupon' && $post_status != 'trash') {
            ?>
            <script type="text/javascript">
                jQuery(document).ready(function ($) {
                    var $downloadToCSV = $('<option>').val('wt_iew_download_coupons').text('<?php _e('Download as CSV', 'wt-import-export-for-woo') ?>');
                    $('select[name^="action"]').append($downloadToCSV);
                });
            </script>
            <?php
        }
    }

    public function process_coupons_bulk_actions() {
        global $typenow;
        if ($typenow == 'shop_coupon') {
            // get the action list
            $wp_list_table = _get_list_table('WP_Posts_List_Table');
            $action = $wp_list_table->current_action();
            if (!in_array($action, array('wt_iew_download_coupons'))) {
                return;
            }
            check_admin_referer('bulk-posts');

            if (isset($_REQUEST['post'])) {
                $coupon_ids = array_map('absint', $_REQUEST['post']);
            }
            if (empty($coupon_ids)) {
                return;
            }
            @set_time_limit(0);

            if ($action == 'wt_iew_download_coupons') {
                include_once( 'export/class-wt-cpnimpexpcsv-exporter.php' );
                Wt_Import_Export_For_Woo_Coupon_Bulk_Export::do_export('shop_coupon', $coupon_ids);
            }
        }
    }	
	
	
	
    /**
    *   Altering advanced step description
    */
    public function importer_steps($steps, $base)
    {
        if($this->module_base==$base)
        {
            $steps['advanced']['description'] = __('Use options from below to decide updates to existing coupons, batch import count or schedule an import. You can also save the template file for future imports.', 'wt-import-export-for-woo');
        }
        return $steps;
    }
    
    public function importer_do_import($import_data, $base, $step, $form_data, $selected_template_data, $method_import, $batch_offset, $is_last_batch) {        
        if ($this->module_base != $base) {
            return $import_data;
        }
        
        if(0 == $batch_offset){                        
            $memory = size_format(wt_let_to_num(ini_get('memory_limit')));
            $wp_memory = size_format(wt_let_to_num(WP_MEMORY_LIMIT));                      
            Wt_Import_Export_For_Woo_Logwriter::write_log($this->module_base, 'import', '---[ New import started at '.date('Y-m-d H:i:s').' ] PHP Memory: ' . $memory . ', WP Memory: ' . $wp_memory);
        }
        
        include plugin_dir_path(__FILE__) . 'import/import.php';
        $import = new Wt_Import_Export_For_Woo_Coupon_Import($this);
        
        $response = $import->prepare_data_to_import($import_data,$form_data,$batch_offset,$is_last_batch);
        
        if($is_last_batch){
            Wt_Import_Export_For_Woo_Logwriter::write_log($this->module_base, 'import', '---[ Import ended at '.date('Y-m-d H:i:s').']---');
        }
                
        return $response;
    }
    
    
    public function exporter_do_export($export_data, $base, $step, $form_data, $selected_template_data, $method_export, $batch_offset) {
        if ($this->module_base != $base) {
            return $export_data;
        }
                
        switch ($method_export) {
            case 'quick':
                $this->set_export_columns_for_quick_export($form_data);  
                break;

            case 'template':            
            case 'new':
                $this->set_selected_column_names($form_data);
                break;
            
            default:
                break;
        }
        
        include plugin_dir_path(__FILE__) . 'export/export.php';
        $export = new Wt_Import_Export_For_Woo_Coupon_Export($this);

        $header_row = $export->prepare_header();

        $data_row = $export->prepare_data_to_export($form_data, $batch_offset);

        $export_data = array(
            'head_data' => $header_row,
            'body_data' => $data_row['data'],
            'total' => $data_row['total'],
        );
        if(isset($data_row['no_post'])){
            $export_data['no_post'] = $data_row['no_post'];
        }
        return $export_data;
    }
    
    /*
     * Setting default export columns for quick export
     */

    public function set_export_columns_for_quick_export($form_data) {

        $post_columns = self::get_coupon_post_columns();

        $this->selected_column_names = array_combine(array_keys($post_columns), array_keys($post_columns));

        if (isset($form_data['method_export_form_data']['mapping_enabled_fields']) && !empty($form_data['method_export_form_data']['mapping_enabled_fields'])) {
            foreach ($form_data['method_export_form_data']['mapping_enabled_fields'] as $value) {
                $additional_quick_export_fields[$value] = array('fields' => array());
            }

            $export_additional_columns = $this->exporter_alter_meta_mapping_fields($additional_quick_export_fields, $this->module_base, array());
            foreach ($export_additional_columns as $value) {
                $this->selected_column_names = array_merge($this->selected_column_names, $value['fields']);
            }
        }
    }

    
    /**
     * Adding current post type to export list
     *
     */
    public function wt_iew_exporter_post_types($arr) {
        $arr['coupon'] = __('Coupon');
        return $arr;
    }

    public static function get_coupon_types() {
        $coupon_types   = wc_get_coupon_types();
        return apply_filters('wt_iew_export_coupon_types',  $coupon_types);
        
    }

    public static function get_coupon_statuses() {
        $statuses = array('publish', 'private', 'draft', 'pending', 'future'); 
        return apply_filters('wt_iew_export_coupon_statuses', array_combine($statuses, $statuses));
    }

    public static function get_coupon_sort_columns() {                
        $sort_columns = array('ID', 'post_parent', 'post_title', 'post_date', 'post_modified', 'post_author', 'menu_order', 'comment_count');
        return apply_filters('wt_iew_export_coupon_sort_columns', array_combine($sort_columns, $sort_columns));
    }

    public static function get_coupon_post_columns() {
        return include plugin_dir_path(__FILE__) . 'data/data-coupon-post-columns.php';
    }
    
    public function get_importer_post_columns($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }
        $colunm = include plugin_dir_path(__FILE__) . 'data/data/data-wf-reserved-fields-pair.php';
//        $colunm = array_map(function($vl){ return array('title'=>$vl, 'description'=>$vl); }, $arr); 
        return $colunm;
    }
    
    public function exporter_alter_mapping_enabled_fields($mapping_enabled_fields, $base, $form_data_mapping_enabled_fields) {
        if ($base == $this->module_base) {
            $mapping_enabled_fields = array();
            $mapping_enabled_fields['meta'] = array(__('Additional meta', 'wt-import-export-for-woo'), 1);
            $mapping_enabled_fields['hidden_meta'] = array(__('Hidden meta', 'wt-import-export-for-woo'), 0);
             
        }
        return $mapping_enabled_fields;
    }

    
    public function exporter_alter_meta_mapping_fields($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }

        foreach ($fields as $key => $value) {
            switch ($key) {
                
                case 'meta':
                    $meta_attributes = array();
                    $found_meta = $this->wt_get_found_meta();
                    foreach ($found_meta as $meta) {
                        $fields[$key]['fields']['meta:' . $meta] = 'meta:' . $meta;
                    }

                    break;               

                case 'hidden_meta':
                    $found_hidden_meta = $this->wt_get_found_hidden_meta();
                    foreach ($found_hidden_meta as $hidden_meta) {
                        $fields[$key]['fields']['meta:' . $hidden_meta] = 'meta:' . $hidden_meta;
                    }
                    break;
                default:
                    break;
            }
        }

        return $fields;
    }
    
    public function importer_alter_meta_mapping_fields($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }
        $fields=$this->exporter_alter_meta_mapping_fields($fields, $base, $step_page_form_data);
        $out=array();
        foreach ($fields as $key => $value) 
        {
            $value['fields']=array_map(function($vl){ return array('title'=>$vl, 'description'=>$vl); }, $value['fields']);
            $out[$key]=$value;
        }
        return $out;
    }
    
    public function wt_get_found_meta() {

        if (!empty($this->found_meta)) {
            return $this->found_meta;
        }

        // Loop products and load meta data
        $found_meta = array();
        // Some of the values may not be usable (e.g. arrays of arrays) but the worse
        // that can happen is we get an empty column.

        $all_meta_keys = $this->wt_get_all_meta_keys();


        $csv_columns = self::get_coupon_post_columns();


        foreach ($all_meta_keys as $meta) {

            if (!$meta || (substr((string) $meta, 0, 1) == '_') || in_array($meta, array_keys($csv_columns)) || in_array('meta:' . $meta, array_keys($csv_columns)))
                continue;

            $found_meta[] = $meta;
        }
        
        $found_meta = array_diff($found_meta, array_keys($csv_columns));
        $this->found_meta = $found_meta;
        return $this->found_meta;
    }

    public function wt_get_found_hidden_meta() {

        if (!empty($this->found_hidden_meta)) {
            return $this->found_hidden_meta;
        }

        // Loop products and load meta data
        $found_hidden_meta = array();
        // Some of the values may not be usable (e.g. arrays of arrays) but the worse
        // that can happen is we get an empty column.

        $all_meta_keys = $this->wt_get_all_meta_keys();
        $csv_columns = self::get_coupon_post_columns();
        foreach ($all_meta_keys as $meta) {

            if (!$meta || (substr((string) $meta, 0, 1) != '_') || in_array($meta, array_keys($csv_columns)) || in_array('meta:' . $meta, array_keys($csv_columns)))
                continue;

            $found_hidden_meta[] = $meta;
        }

        $found_hidden_meta = array_diff($found_hidden_meta, array_keys($csv_columns));

        $this->found_hidden_meta = $found_hidden_meta;
        return $this->found_hidden_meta;
    }


    public function wt_get_all_meta_keys() {

        if (!empty($this->all_meta_keys)) {
            return $this->all_meta_keys;
        }

        $all_meta_keys = self::get_all_metakeys('shop_coupon');

        $this->all_meta_keys = $all_meta_keys;
        return $this->all_meta_keys;
    }

        
    public static function get_all_metakeys($post_type = 'shop_coupon') {
        global $wpdb;

        $meta = $wpdb->get_col($wpdb->prepare(
                        "SELECT DISTINCT pm.meta_key
            FROM {$wpdb->postmeta} AS pm
            LEFT JOIN {$wpdb->posts} AS p ON p.ID = pm.post_id
            WHERE p.post_type = %s
            AND p.post_status IN ( 'publish', 'pending', 'private', 'draft' ) LIMIT 2010", $post_type
                ));

        sort($meta);
		
        return $meta;
    }
    
    
    public function set_selected_column_names($full_form_data) {        
        if (is_null($this->selected_column_names)) {
            if (isset($full_form_data['mapping_form_data']['mapping_selected_fields']) && !empty($full_form_data['mapping_form_data']['mapping_selected_fields'])) {
                $this->selected_column_names = $full_form_data['mapping_form_data']['mapping_selected_fields'];
            }
            if (isset($full_form_data['meta_step_form_data']['mapping_selected_fields']) && !empty($full_form_data['meta_step_form_data']['mapping_selected_fields'])) {
                $export_additional_columns = $full_form_data['meta_step_form_data']['mapping_selected_fields'];
                foreach ($export_additional_columns as $value) {
                    $this->selected_column_names = array_merge($this->selected_column_names, $value);
                }
            }
        }

        return $full_form_data;
    }
    
    public function get_selected_column_names() {
            
        return $this->selected_column_names;
    }    

    public function exporter_alter_mapping_fields($fields, $base, $mapping_form_data) {
        if ($base == $this->module_base) {
            $fields = self::get_coupon_post_columns();
        }
        return $fields;
    }
    
    public function importer_alter_advanced_fields($fields, $base, $advanced_form_data) {
        if ($this->module_base != $base) {
            return $fields;
        }
        $out = array(); 
        
        $out['skip_new'] = array(
            'label' => __("Update Only", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'skip_new',
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('The store is updated with the data from the input file only for matching/existing records from the file.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_skip_new', 'value'=>1)
                    )
                ),
                array(
                    'help_text'=> __('The entire data from the input file is processed for an update or insert as the case maybe.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_skip_new', 'value'=>0)
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'wt_iew_skip_new',
            )
        ); 
        
        $out['merge_with'] = array(
            'label' => __("Match coupons by their", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                'id' => __('ID'),
                'code' => __('Coupon Code'),
                
            ),
            'value' => 'id',
            'field_name' => 'merge_with',
            'help_text' => __('The products are either looked up based on their ID or coupon code as per the selection.', 'wt-import-export-for-woo'),
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('If the post ID of the coupon being imported exists already(for any of the other post types like coupon, order, user, pages, media etc) skip the coupon from being inserted into the store.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_merge_with', 'value'=>'id'),
                        'AND',
                        array('field'=>'wt_iew_skip_new', 'value'=>1)
                    )
                )
            )
        );
        
        $out['found_action_merge'] = array(
            'label' => __("If the coupon exists in the store", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
//                'import' => __('Import as new item'),
                'skip' => __('Skip', 'wt-import-export-for-woo'),
                'update' => __('Update', 'wt-import-export-for-woo'),                
            ),
            'value' => 'skip',
            'field_name' => 'found_action',
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('Retains the coupon in the store as is and skips the matching coupon from the input file.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_found_action', 'value'=>'skip')
                    )
                ),
                array(
                    'help_text'=> __('Update coupon as per data from the input file', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_found_action', 'value'=>'update')
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'wt_iew_found_action'
            )
        );       
        
        $out['merge_empty_cells'] = array(
            'label' => __("Update even if empty values", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'merge_empty_cells',
            'help_text' => __('Updates the coupon data respectively even if some of the columns in the input file contains empty value.', 'wt-import-export-for-woo'),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'wt_iew_found_action',
                'val'=>'update',
            )
        );
        
        $out['conflict_with_existing_post'] = array(
            'label' => __("If conflict with an existing Post ID", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(                
                'skip' => __('Skip item', 'wt-import-export-for-woo'),
                'import' => __('Import as new item', 'wt-import-export-for-woo'),
                
            ),
            'value' => 'skip',
            'field_name' => 'id_conflict',
            'help_text' => __('All the items within WooCommerce/WordPress are treated as posts and assigned a unique ID as and when they are created in the store. The post ID uniquely identifies an item irrespective of the post type be it coupon/product/pages/attachments/revisions etc.', 'wt-import-export-for-woo'),
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('If the post ID of the coupon being imported exists already(for any of the posts like coupon, order, user, pages, media etc) skip the coupon from being inserted into the store.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_id_conflict', 'value'=>'skip')
                    )
                ),
                array(
                    'help_text'=> __('Insert the coupon into the store with a new coupon ID(next available post ID) different from the value in the input file.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_id_conflict', 'value'=>'import')
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'wt_iew_skip_new',
                'val'=>'0',
                'depth'=>0,
            )
        );
        
        
        
        
        
        
//        $out['merge'] = array(
//            'label' => __("Update coupon if exists"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                '1' => __('Yes'),
//                '0' => __('No')
//            ),
//            'value' => '0',
//            'field_name' => 'merge',
//            'help_text' => __('Existing coupons are identified by their IDs or Coupon Codes'),
//            'form_toggler'=>array(
//                'type'=>'parent',
//                'target'=>'wt_iew_merge'
//            )
//        );   
//        
//        $out['merge_with'] = array(
//            'label' => __("Update Coupon with ID or Coupon Code"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                'id' => __('Update with id'),
//                'coupon_code' => __('Update Coupon with Coupon Code'),
//                
//            ),
//            'value' => 'id',
//            'field_name' => 'merge_with',
//            'help_text' => __('Update Coupon with ID or Coupon Code.'),
//            'form_toggler'=>array(
//                'type'=>'child',
//                'id'=>'wt_iew_merge',
//                'val'=>'1',
//            )
//        );
//        
//        $out['skip_new'] = array(
//            'label' => __("Skip new coupon"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                '1' => __('Yes'),
//                '0' => __('No')
//            ),
//            'value' => '0',
//            'field_name' => 'skip_new',
//            'help_text' => __('While updating existing coupons, enable this to skip coupons which are not already present in the store.'),
//            'form_toggler'=>array(
//                'type'=>'child',
//                'id'=>'wt_iew_merge',
//                'val'=>'1',
//            )
//        );
//        
//        $out['found_action_merge'] = array(
//            'label' => __("Skip or import if Coupon not find"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                'import' => __('Import as new item'),
//                'skip' => __('Skip item'),
//                
//            ),
//            'value' => 'import',
//            'field_name' => 'found_action',
//            'help_text' => __('Skip or import if Coupon not find.'),
//            'form_toggler'=>array(
//                'type'=>'',
//                'id'=>'wt_iew_merge',
//                'val'=>'1',
//                'target'=>'wt_iew_use_same_id'
//            )
//        );
//        
//        $out['found_action_import'] = array(
//            'label' => __("Skip or import"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                'import' => __('Import as new item'),
//                'skip' => __('Skip item'),
//                
//            ),
//            'value' => 'import',
//            'field_name' => 'found_action',
//            'help_text' => __('Skip or import if found a non coupon post type in given ID.'),
//            'form_toggler'=>array(
//                'type'=>'',
//                'id'=>'wt_iew_merge',
//                'val'=>'0',
//                'target'=>'wt_iew_use_same_id'
//            )
//        );
//        
//        $out['use_same_id'] = array(
//            'label' => __("Use the same ID for Coupon on import"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                '1' => __('Yes'),
//                '0' => __('No')
//            ),
//            'value' => '0',
//            'field_name' => 'use_same_id',
//            'help_text' => __('Use the same ID for Coupon on import.'),
//            'form_toggler'=>array(
//                'type'=>'',
//                'id'=>'wt_iew_use_same_id',
//                'val'=>'import',
//                'depth'=>2,
//            )
//        );
//        
//        $out['merge_empty_cells'] = array(
//            'label' => __("Merge empty cells"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                '1' => __('Yes'),
//                '0' => __('No')
//            ),
//            'value' => '0',
//            'field_name' => 'merge_empty_cells',
//            'help_text' => __('Check to merge the empty cells in CSV, otherwise empty cells will be ignored.'),
//            'form_toggler'=>array(
//                'type'=>'child',
//                'id'=>'wt_iew_merge',
//                'val'=>'1',
//            )
//        );
        
        
        
        $out['delete_existing'] = array(
            'label' => __("Delete non-matching coupons from store", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'delete_existing',
            'help_text' => __('Select ‘Yes’ if you need to remove the coupons from your store which are not present in the input file. For e.g, if you have a coupon A in your store and your import file has coupons B, C; the coupon A is deleted from the store prior to importing B and C.', 'wt-import-export-for-woo'),
        );
        
        $out['use_sku'] = array(
            'label' => __("Use product SKU for coupon restriction settings", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'use_sku',
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('Link the products by their SKUs under coupon restrictions for the imported coupons.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_use_sku', 'value'=>1)
                    )
                ),
                array(
                    'help_text'=> __('Link the products by their product IDs under coupon restrictions for the imported coupons. In case of a conflict with IDs of other existing post types the link will be empty.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_use_sku', 'value'=>0)
                    )
                )
            ),
        );
        
        
        foreach ($fields as $fieldk => $fieldv) {
            $out[$fieldk] = $fieldv;
        }
        return $out;
    }

     /**
     *  Customize the items in filter export page
     * 
     *  1.3.1 - updated filter fields
     */
    public function exporter_alter_filter_fields($fields, $base, $filter_form_data) {
        if ( $this->module_base !== $base ) {
            return $fields;
        }

         // Initialize filter structure
         $fields = $this->initialize_filter_structure($fields);

        // Get saved template data and prepare data
        $saved_template_data = $this->get_saved_template_data($filter_form_data);
        $all_filter_options = $this->get_all_filter_options();
        $selected_elements = $this->extract_selected_elements($saved_template_data);     
        
        if ( isset( $filter_form_data['old_template_data'] ) && is_array($filter_form_data['old_template_data']) ) {
            $fields = $this->create_old_template_rows($fields, $base, $all_filter_options, $selected_elements, $filter_form_data);
        } else {
            // Create filter rows based on data type
            if ( ! empty( $saved_template_data ) ) {
                $fields = $this->create_saved_template_rows($fields, $saved_template_data, $all_filter_options, $selected_elements);
                // Apply saved template values
                $fields = $this->alter_filter_fields_with_saved_template($fields, $saved_template_data);
            } else {
                $fields = $this->create_default_rows($fields, $all_filter_options);
            }
        }
        
        // Add extra fields if needed
        $fields = $this->add_extra_filter_fields($fields, $base, $filter_form_data);
        
        // Add available filter options for dropdown
        $fields['multi_field_row']['available_filter_options'] = $all_filter_options;
		
        return $fields;
    }

    /**
     * Initialize the basic filter structure
     * 
     * @since 1.3.1
     */
    private function initialize_filter_structure($fields) {
        if ( isset( $fields['limit'] ) ) {
            unset( $fields['limit'] );
        }
        if ( isset( $fields['offset'] ) ) {
            unset( $fields['offset'] ); 
        }
        $fields['multi_field_row']['type'] = 'multi_field_row';
        $fields['multi_field_row']['fields'] = array();
        $fields['multi_field_row']['fields'][] = $this->get_filter_column_headers();
            return $fields;
    }


    /**
     * Get saved template data from filter form data
     * 
     * @since 1.3.1
     */
    private function get_saved_template_data($filter_form_data) {
        return ( isset( $filter_form_data['saved_template_data'] ) && is_array( $filter_form_data['saved_template_data'] ) ) ? $filter_form_data['saved_template_data'] : array();
    }
    
    /**
     * Get all available filter options
     * 
     * @since 1.3.1
     */
    private function get_all_filter_options() {
        $default_filter_options = $this->get_default_filter_options();
        $extra_filter_options = $this->get_extra_filter_options();
        
        // Ensure both are arrays before merging
        $default_filter_options = is_array( $default_filter_options ) ? $default_filter_options : array();
        $extra_filter_options = is_array( $extra_filter_options ) ? $extra_filter_options : array();
        
        return array_merge( $default_filter_options, $extra_filter_options );
    }
    
    /**
     * Extract selected elements from saved template data
     * 
     * @since 1.3.1
     */
    private function extract_selected_elements($saved_template_data) {
        $selected_elements = array();
        if ( ! empty( $saved_template_data ) && is_array( $saved_template_data ) ) {
            foreach ( $saved_template_data as $template_data ) {
                if ( ! empty( $template_data['element'] ) ) {
                    $selected_elements[] = $template_data['element'];
                }
            }
        }
        return $selected_elements;
    }
    
    /**
     * Create saved template rows
     * 
     * @since 1.3.1
     */
    private function create_saved_template_rows($fields, $saved_template_data, $all_filter_options, $selected_elements) {
        $valid_template_count = $this->count_valid_template_entries( $saved_template_data );
        $index = 0;
        $current_row = 0;
        
        foreach ( $saved_template_data as $template_key => $template_data ) {
            $element = isset($template_data['element']) ? $template_data['element'] : '';
            
            if ( empty( $element ) ) {
                continue;
            }
            
            $row_fields = $this->create_filter_row_structure($index, $element, $all_filter_options, $current_row === $valid_template_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $element, $all_filter_options, $selected_elements);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
            $current_row++;
        }

        return $fields;
    }

    /**
     * Create old template rows
     * 
     * @since 1.3.1
     */
    private function create_old_template_rows($fields, $base, $all_filter_options, $selected_elements, $filter_form_data) {
        if ( $this->module_base != $base ) {
            return $fields;
        }

        // Get the old template data from filter_form_data
        $old_template_data = isset($filter_form_data['old_template_data']) ? $filter_form_data['old_template_data'] : array();
        


        if ( empty($old_template_data) || ! is_array( $old_template_data ) ) {
            return $fields;
        }

        // Convert old template data to new structure
        $converted_template_data = $this->convert_old_template_to_new_structure($old_template_data);

        if ( empty($converted_template_data) ) {
            return $this->create_default_rows($fields, $all_filter_options);
        }
        
        $valid_template_count = count($converted_template_data);
        $index = 0;
        $current_row = 0;
        
        foreach ( $converted_template_data as $template_data ) {
            $element = isset($template_data['element']) ? $template_data['element'] : '';
            
            if ( empty( $element ) ) {
                continue;
            }
            
            $row_fields = $this->create_filter_row_structure($index, $element, $all_filter_options, $current_row === $valid_template_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $element, $all_filter_options, $selected_elements);
            
            // Set the values from old template data
            $row_fields = $this->set_old_template_values($row_fields, $template_data, $index);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
            $current_row++;
        }

        return $fields;
    }

    /**
     * Convert old template data to new structure
     * 
     * @since 1.3.1
     */
    private function convert_old_template_to_new_structure($old_template_data) {
        
        // Static field mapping for better performance - based on actual old template field names
        static $field_mapping = null;
        
        if ( null === $field_mapping ) {
            $field_mapping = array(
                'wt_iew_statuses' => array('new_field' => 'coupon_status', 'condition' => 'include'),
                'wt_iew_types' => array('new_field' => 'coupon_type', 'condition' => 'include'),
                'wt_iew_coupon_amount_from' => array('new_field' => 'coupon_amount', 'condition' => 'in_between', 'is_range_start' => true),
                'wt_iew_coupon_amount_to' => array('new_field' => 'coupon_amount', 'condition' => 'in_between', 'is_range_end' => true),
                'wt_iew_coupon_exp_date_from' => array('new_field' => 'expiry_date', 'condition' => 'in_between', 'is_range_start' => true),
                'wt_iew_coupon_exp_date_to' => array('new_field' => 'expiry_date', 'condition' => 'in_between', 'is_range_end' => true),
            );
        }
        
        $converted_data = array();
        $range_fields = array();
        
        // Handle simple field mappings
        foreach ( $field_mapping as $old_field => $field_config ) {
            if ( isset( $old_template_data[$old_field] ) && ! empty( $old_template_data[$old_field] ) ) {
                $value = $old_template_data[$old_field];
                $new_field = $field_config['new_field'];
                $condition = $field_config['condition'];

                // Handle range fields (from/to pairs)
                if (isset($field_config['is_range_start']) || isset($field_config['is_range_end'])) {
                    $range_fields[$new_field][$field_config['is_range_start'] ? 'from' : 'to'] = $value;
                } else {
                    // Simple field mapping
                    $converted_data[] = array(
                        'element' => $new_field,
                        'value' => $value,
                        'condition' => $condition,
                        'value_for' => $new_field
                    );
                }
            }
        }
        
        // Process range fields
        foreach ($range_fields as $field_name => $range_data) {
            if (isset($range_data['from']) && isset($range_data['to'])) {
                $converted_data[] = array(
                    'element' => $field_name,
                    'condition' => 'in_between',
                    'value' => array(
                        'from' => $range_data['from'],
                        'to' => $range_data['to']
                    ),
                    'value_for' => $field_name
                );
            }
        }       
        
        return $converted_data;
    }

    /**
     * Set old template values in the row fields
     * 
     * @since 1.3.1
     */
    private function set_old_template_values($row_fields, $template_data, $index) {
        if ( ! isset( $row_fields['rows'] ) || ! is_array( $row_fields['rows'] ) ) {
            return $row_fields;
        }
        
        $element = $template_data['element'];
        $value = $template_data['value'];
        $condition = $template_data['condition'];
        $element_field_name = 'wt_iew_filter_element[' . $index . ']';
        
        foreach ( $row_fields['rows'] as $row_index => $row ) {
            $field_name = $row['field_name'] ?? '';
            
            // Set element selector value
            if ( $field_name === $element_field_name ) {
                $row_fields['rows'][$row_index]['value'] = $element;
            }
            
            // Set condition field value
            if ( isset( $row['condition_field'] ) && $row['condition_field'] === true ) {
                $condition_type = $row['data_value_for'] ?? '';
                
                if ( 'greater-than-less-than' === $condition_type ) {
                    $row_fields['rows'][$row_index]['value'] = $condition;
                }
            }
            
            // Set value field
            if ( isset( $row['data_value_for'] ) && $row['data_value_for'] === $element && ! empty( $value ) ) {
                $row_fields['rows'][$row_index]['value'] = $value;
            }
        }
        
        return $row_fields;
    }

    /**
     * Create default filter rows
     * 
     * @since 1.3.1
     */
    private function create_default_rows($fields, $all_filter_options) {
        $default_filter_options = $this->get_default_filter_options();
        $default_filter_count = count($default_filter_options);
        $index = 0;
        
        foreach ( $default_filter_options as $filter_key => $filter_value ) {
            $row_fields = $this->create_filter_row_structure($index, $filter_key, $all_filter_options, $index === $default_filter_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $filter_key, $all_filter_options, array(), $default_filter_options);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
        }
        
        return $fields;
    }
    
    /**
     * Count valid template entries
     * 
     * @since 1.3.1
     */
    private function count_valid_template_entries($saved_template_data) {
        $count = 0;
        foreach ( $saved_template_data as $template_data ) {
            $element = isset( $template_data['element'] ) ? $template_data['element'] : '';
            if ( ! empty( $element ) ) {
                $count++;
            }
        }
        return $count;
    }
    
    /**
     * Create the basic structure for a filter row
     * 
     * @since 1.3.1
     */
    private function create_filter_row_structure($index, $element, $all_filter_options, $is_last_row) {
        return array(
            'identifier' => 'multi_field_row_' . ($index + 1),
            'add_new_btn' => $is_last_row,
            'row_delete_btn' => true,
            'default_value' => $element,
            'default_condition' => isset($all_filter_options[$element]) ? $all_filter_options[$element]['condition'] : 'equals-not-equals',
            'row_container_attr' => array('style' => 'display:flex;flex-direction:row;gap:40px;'),
            'row_main_attr' => array('style' => 'display:flex;flex-direction:row;gap:4px;align-items:center;padding: 10px 0px;margin-left: 24px;'),
            'rows' => array()
        );
    }
    
    /**
     * Build the fields for a filter row
     * 
     * @since 1.3.1
     */
    private function build_filter_row_fields($index, $element, $all_filter_options, $selected_elements = array(), $default_filter_options = array() ) {
        $rows = array();
        
        // Element selector field
        $available_options = $this->get_available_options( $element, $all_filter_options, $selected_elements, $index, $default_filter_options );
        $rows[] = array(
            'field_name' => 'wt_iew_filter_element[' . $index . ']',
            'type' => 'select',
            'value' => $element,
            'sele_vals' => $available_options,
            'css_class' => '',
            'attr' => $this->get_common_style_for_element_selector(),
        );
        
        // Add condition fields
        $rows = array_merge( $rows, $this->get_condition_fields( $index ) );
        
        // Add value fields
        $value_fields = $this->get_value_fields( $index );
        $rows = array_merge( $rows, array_values( $value_fields ) );
        
        return $rows;
    }
    
    /**
     * Get available options for element selector
     * 
     * @since 1.3.1
     */
    private function get_available_options($current_element, $all_filter_options, $selected_elements, $index = 0, $default_filter_options = array() ) {
        $available_options = array();
        
        if ( ! empty( $selected_elements ) ) {
            // For saved template data - exclude already selected elements
            foreach ( $all_filter_options as $option_key => $option_value ) {
                if ( ! in_array( $option_key, $selected_elements ) || $option_key === $current_element ) {
                    $available_options[$option_key] = $option_value;
                }
            }
        } else {
            // For default rows - all rows show the same available options
            // Exclude only the default elements that are used in other rows
            $default_elements = array_keys( $default_filter_options );
            
            foreach ( $all_filter_options as $option_key => $option_value ) {
                // Include all options except the ones used as defaults in other rows
                if ( ! in_array( $option_key, $default_elements ) || $option_key === $current_element ) {
                    $available_options[$option_key] = $option_value;
                }
            }
        }
        
        return $available_options;
    }

    /**
     * Apply saved template values to fields
     * 
     * @since 1.3.1
     */
    private function alter_filter_fields_with_saved_template($fields, $saved_template_data) {
        // Early validation
        if ( empty($fields) || ! is_array( $fields ) || 
            empty( $saved_template_data ) || ! is_array( $saved_template_data ) ) {
            return $fields;
        }
        
        // Cache filter options to avoid repeated array_merge calls
        $all_filter_options = array_merge( $this->get_default_filter_options(), $this->get_extra_filter_options() );

        foreach ( $saved_template_data as $template_key => $template_data ) {
            // Extract data with null coalescing for cleaner code
            $element = $template_data['element'] ?? '';
            $condition = $template_data['condition'] ?? '';
            $value = $template_data['value'] ?? '';

            // Skip empty elements early
            if ( empty( $element ) ) {
                continue;
            }

            // Get condition type and skip if not found
            $element_condition_type = $all_filter_options[$element]['condition'] ?? '';
            if ( empty( $element_condition_type ) ) {
                continue;
            }

            // Extract row index with validation
            if ( ! preg_match( '/multi_field_row_(\d+)/', $template_key, $matches ) ) {
                continue;
            }
            $row_index = max(0, intval($matches[1]) - 1);

            // Validate field exists and has rows
            if ( ! isset( $fields['multi_field_row']['fields'][$row_index]['rows'] ) || 
                ! is_array( $fields['multi_field_row']['fields'][$row_index]['rows'] ) ) {
                continue;
            }

            $rows = &$fields['multi_field_row']['fields'][$row_index]['rows'];

            // Process each row field
            foreach ( $rows as $row_idx => &$row ) {
                $this->set_field_value_if_matches($row, $row_idx, $element, $condition, $value, $element_condition_type, $row_index);
            }
        }

        return $fields;
    }

    /**
     * Helper method to set field values if they match the criteria
     * 
     * @since 1.3.1
     */
    private function set_field_value_if_matches(&$row, $row_idx, $element, $condition, $value, $element_condition_type, $row_index) {
        // Set element selector value
        if ( isset($row['field_name'] ) && 
            preg_match('/wt_iew_filter_element\[(\d+)\]/', $row['field_name'], $matches) &&
            intval($matches[1]) === $row_index && 
            isset($row['sele_vals'][$element])) {
            $row['value'] = $element;
        }

        // Set condition field value
        if ( isset( $row['condition_field']) && true === $row['condition_field'] &&
            isset( $row['data_value_for'] ) && $row['data_value_for'] === $element_condition_type && 
            ! empty( $condition ) ) {
            $row['value'] = $condition;
        }

        // Set value field
        if ( isset( $row['data_value_for'] ) && $row['data_value_for'] === $element && ! empty( $value ) ) {
            $row['value'] = $value;
        }
    }

    /**
     * Get filter column headers
     * 
     * @since 1.3.1
     */
    private function get_filter_column_headers() {
        return array(
            'row_container_attr' => array('style' => 'display:flex;flex-direction:row;gap:40px;'),
            'row_main_attr' => array('style' => 'display:flex;flex-direction:row;gap:4px;align-items:center;padding: 10px 0px;margin-left: 24px;'),
            'type' => 'header_row',
            'rows' => array(
                array(
                    'type' => 'label',
                    'label' => __('Element', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 250px;'),
                ),
                array(
                    'type' => 'label',
                    'label' => __('Condition', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 132px;'),
                ),
                array(
                    'type' => 'label',
                    'label' => __('Value', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 350px;'),
                ),
            ),
        );
    }

    /**
     * Get default filter options
     * 
     * @since 1.3.1
     */
    private function get_default_filter_options() {
        return array(
            'coupons' => array('label' => __('Coupons', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Coupons', 'wt-import-export-for-woo')),
            'coupon_status' => array('label' => __('Status', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Status', 'wt-import-export-for-woo')),
            'coupon_type' => array('label' => __('Coupon Type', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Coupon Type', 'wt-import-export-for-woo')),
            'coupon_amount' => array('label' => __('Coupon Amount', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Coupon Amount', 'wt-import-export-for-woo')),
            'expiry_date' => array('label' => __('Expiry Date', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Expiry Date', 'wt-import-export-for-woo')),
        );
    }
    
    /**
     * Get extra filter options
     * 
     * @since 1.3.1
     */
    private function get_extra_filter_options() {
        return array(
            'usage_count' => array('label' => __('Usage Count', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Usage Count', 'wt-import-export-for-woo')),
        );
    }

    /**
     * Get common style for element selector
     * 
     * @since 1.3.1
     */
    private function get_common_style_for_element_selector() {
        return array('style' => 'width: 250px; height: 40px; border-radius: 6px;');
    }

    /**
     * Get common style for value fields
     * 
     * @since 1.3.1
     */
    private function get_common_style_for_value_fields() {
        return array('style' => 'width: 350px; height: 40px; border-radius: 6px;');
    }

    /**
     * Get condition fields
     * 
     * @since 1.3.1
     */
    private function get_condition_fields($index) {
        $style = array('style' => 'width: 132px; height: 40px; border-radius: 6px;');
        return array(
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'include' => __('Include', 'wt-import-export-for-woo'),
                    'exclude' => __('Exclude', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'exc-inc',
                'visible_if' => 'exc-inc',
                'condition_field' => true,
            ),
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'equals' => __('Equals', 'wt-import-export-for-woo'),
                    'not_equals' => __('Not equals', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'equals-not-equals',
                'visible_if' => 'equals-not-equals',
                'condition_field' => true,
            ),
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'greater_than' => __('Greater than', 'wt-import-export-for-woo'),
                    'less_than' => __('Less than', 'wt-import-export-for-woo'),
                    'equals' => __('Equals', 'wt-import-export-for-woo'),
                    'not_equals' => __('Not equals', 'wt-import-export-for-woo'),
                    'in_between' => __('In between', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'greater-than-less-than',
                'visible_if' => 'greater-than-less-than',
                'condition_field' => true,
            ),
        );
    }

    /**
     * Get value fields
     * 
     * @since 1.3.1
     */
    private function get_value_fields($index) {
        $style = $this->get_common_style_for_value_fields();
    
        return array(
            'coupons' => array(
                'field_name' => "coupons[$index]",
                'data_value_for' => 'coupons',
                'type' => 'multi_select',
                'placeholder' => __('Search for coupons', 'wt-import-export-for-woo'),
                'sele_vals' => array(),
                'css_class' => 'wt-coupon-search',
                'attr' => $style,
                'visible_if' => 'coupons',
            ),
            'coupon_status' => array(
                'field_name' => "coupon_status[$index]",
                'data_value_for' => 'coupon_status',
                'type' => 'multi_select',
                'placeholder' => __('All statuses', 'wt-import-export-for-woo'),
                'sele_vals' => self::get_coupon_statuses(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'coupon_status',
            ),
            'coupon_type' => array(
                'field_name' => "coupon_type[$index]",
                'data_value_for' => 'coupon_type',
                'type' => 'multi_select',
                'placeholder' => __('All types', 'wt-import-export-for-woo'),
                'sele_vals' => self::get_coupon_types(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'coupon_type',
            ),
            'coupon_amount' => array(
                'field_name' => "coupon_amount[$index]",
                'data_value_for' => 'coupon_amount',
                'type' => 'numeric_range',
                'placeholder' => __('Enter amount', 'wt-import-export-for-woo'),
                'from_placeholder' => __('From amount', 'wt-import-export-for-woo'),
                'to_placeholder' => __('To amount', 'wt-import-export-for-woo'),
                'step' => '0.01',
                'css_class' => 'price',
                'min' => '0',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'coupon_amount',
                'index' => $index,
            ),
            'expiry_date' => array(
                'field_name' => "expiry_date[$index]",
                'data_value_for' => 'expiry_date',
                'type' => 'custom_date_range',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'expiry_date',
                'index' => $index,
            ),
            'usage_count' => array(
                'field_name' => "usage_count[$index]",
                'data_value_for' => 'usage_count',
                'type' => 'numeric_range',
                'placeholder' => __('Enter count', 'wt-import-export-for-woo'),
                'from_placeholder' => __('From count', 'wt-import-export-for-woo'),
                'to_placeholder' => __('To count', 'wt-import-export-for-woo'),
                'step' => '1',
                'css_class' => 'number',
                'min' => '0',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'usage_count',
                'index' => $index,
            ),
      
        );
    }

    /**
     * Add extra filter fields
     * 
     * @since 1.3.1
     */
    public function add_extra_filter_fields($fields, $base, $filter_form_data) {

        if ( ! apply_filters( 'wt_iew_exporter_add_extra_filter_fields', false, $base, $filter_form_data ) ) {
            return $fields;
        }

         /* altering help text of default fields */
			$fields['limit']['label']=__('Total number of coupons to export', 'wt-import-export-for-woo'); 
			$fields['limit']['help_text']=__('Exports specified number of coupons. e.g. Entering 500 with a skip count of 10 will export coupons from 11th to 510th position.', 'wt-import-export-for-woo');
			$fields['offset']['label']=__('Skip first <i>n</i> coupons', 'wt-import-export-for-woo');
			$fields['offset']['help_text']=__('Skips specified number of coupons from the beginning. e.g. Enter 10 to skip first 10 coupons from export.', 'wt-import-export-for-woo');

            $fields['sort_columns'] = array(
                'label' => __('Sort columns', 'wt-import-export-for-woo'),
                'placeholder' => __('ID'),
                'field_name' => 'sort_columns',
                'sele_vals' => self::get_coupon_sort_columns(),
                'help_text' => __('Sort the exported data based on the selected columns in order specified. Defaulted to ascending order.', 'wt-import-export-for-woo'),
                'type' => 'multi_select',
                'css_class' => 'wc-enhanced-select',
                'validation_rule' => array('type'=>'text_arr')
            );

            $fields['order_by'] = array(
                'label' => __('Sort By', 'wt-import-export-for-woo'),
                'placeholder' => __('ASC'),
                'field_name' => 'order_by',
                'sele_vals' => array('ASC' => 'Ascending', 'DESC' => 'Descending'),
                'help_text' => __('Defaulted to Ascending. Applicable to above selected columns in the order specified.', 'wt-import-export-for-woo'),
                'type' => 'select',
            );


        return $fields;
    }

    public function get_item_by_id($id) {
        $post['edit_url']=get_edit_post_link($id);
        $post['title'] = get_the_title($id);
        return $post; 
    }
	public static function get_item_link_by_id($id) {
        $post['edit_url']=get_edit_post_link($id);
        $post['title'] = get_the_title($id);
        return $post; 
    }

}
new Wt_Import_Export_For_Woo_Coupon();
