<?php

if (!defined('WPINC')) {
    exit;
}

class Wt_Import_Export_For_Woo_Coupon_Export {
    
    /**
     * SQL fragments for building complex queries
     * 
     * @var array
     */
    private $sql_fragments = array();

    public $parent_module = null;

    public $export_shortcodes = false;

    public function __construct($parent_object) {

        $this->parent_module = $parent_object;
    }

    public function prepare_header() {

        $export_columns = $this->parent_module->get_selected_column_names();

        return apply_filters('hf_alter_coupon_csv_header', $export_columns);
    }


     /**
     * Prepare data that will be exported.
     */
    public function prepare_data_to_export($form_data, $batch_offset) {
        
        // Initialize SQL fragment collectors
        $this->sql_fragments = array(
            'left_joins' => array(),
            'where_conditions' => array(),
            'subqueries' => array(),
            'base_conditions' => array('1') // Always true base condition
        );
        
        // Extract basic parameters
        $export_limit = !empty($form_data['filter_form_data']['wt_iew_limit']) ? intval($form_data['filter_form_data']['wt_iew_limit']) : 999999999;
        $current_offset = !empty($form_data['filter_form_data']['wt_iew_offset']) ? intval($form_data['filter_form_data']['wt_iew_offset']) : 0;
        $batch_count = !empty($form_data['advanced_form_data']['wt_iew_batch_count']) ? $form_data['advanced_form_data']['wt_iew_batch_count'] : Wt_Import_Export_For_Woo_Common_Helper::get_advanced_settings('default_export_batch');
        $export_sortby = !empty($form_data['filter_form_data']['wt_iew_sort_columns']) ? implode(' ', $form_data['filter_form_data']['wt_iew_sort_columns']) : 'ID';
        $export_sort_order = !empty($form_data['filter_form_data']['wt_iew_order_by']) ? $form_data['filter_form_data']['wt_iew_order_by'] : 'ASC';
        
        $this->export_shortcodes = (!empty($form_data['advanced_form_data']['wt_iew_export_shortcode_tohtml']) && $form_data['advanced_form_data']['wt_iew_export_shortcode_tohtml'] == 'Yes') ? true : false;
        
        /**
         * Process filters and build the final optimized SQL query.
         * @since 1.3.1
         */
        $this->filter_coupons_to_export($form_data);
        
        $coupon_sql_query = $this->build_final_sql_query($export_limit, $current_offset, $export_sortby, $export_sort_order);

        // Execute single query to get coupon IDs
        global $wpdb;
        $total_queried_coupons = $wpdb->get_col($coupon_sql_query);
        
        // Use coupon IDs with batch processing
        $coupon_ids = apply_filters('wt_couponimpexpcsv_alter_coupon_ids', $total_queried_coupons);
        $total_records = count($coupon_ids);
        
        // Calculate batch limits
        if ($batch_count <= $export_limit) {
            if (($batch_offset + $batch_count) > $export_limit) { //last offset
                $limit = $export_limit - $batch_offset;
            } else {
                $limit = $batch_count;
            }
        } else {
            $limit = $export_limit - $batch_offset;
        }
        
        $data_array = array();
        if ($batch_offset < $export_limit) {
            
            // Process batch of coupon IDs
            $batch_coupon_ids = array_slice($coupon_ids, $batch_offset, $limit);
            
            foreach ($batch_coupon_ids as $coupon_id) {
                $coupon = get_post($coupon_id);
                if ($coupon && 'shop_coupon' === $coupon->post_type ) {
                    $data_array[] = $this->generate_row_data($coupon);
                }
            }

            $data_array = apply_filters('wt_ier_alter_coupon_data_befor_export', $data_array);
        }
        
        $return = array();
        $return['total'] = $total_records;
        $return['data'] = $data_array;
        if (0 === $batch_offset && 0 === $total_records) {
            $return['no_post'] = __('Nothing to export under the selected criteria. Please check and try adjusting the filters.', 'wt-import-export-for-woo');
        }
        return $return;
    }


    /**
     * Filter coupons based on the new filter system
     * 
     * @param array $form_data Complete form data from frontend
     */
    private function filter_coupons_to_export($form_data) {
        // Check if multi-field filter data exists
        if (empty($form_data['filter_form_data']['wt_iew_multi_field_row_array'])) {
            return;
        }
       
        // Decode the JSON string to get the filter array
        $filter_data = json_decode($form_data['filter_form_data']['wt_iew_multi_field_row_array'], true);
            
        if (empty($filter_data) || !is_array($filter_data)) {
            return;
        }

        // Process each filter row efficiently
        foreach ($filter_data as $row_data) {
            $element = $row_data['element'] ?? '';
            $value = $row_data['value'] ?? '';
            $condition = $row_data['condition'] ?? '';
            
            if (!empty($element) && !empty($condition)) {
                $this->apply_filter_by_element($element, $value, $condition);
            }
        }
    }


    /**
     * Apply filter based on element type
     * 
     * @param string $element Filter element type
     * @param mixed $value Filter value
     * @param string $condition Filter condition
     */
    private function apply_filter_by_element($element, $value, $condition) {
        // Static filter method mapping for better performance
        static $filter_methods = array(
            'coupons' => 'apply_coupons_filter',
            'coupon_status' => 'apply_coupon_status_filter',
            'coupon_type' => 'apply_coupon_type_filter',
            'coupon_amount' => 'apply_coupon_amount_filter',
            'expiry_date' => 'apply_expiry_date_filter',
            'usage_count' => 'apply_usage_count_filter',
        );

        if (isset($filter_methods[$element])) {
            $method_name = $filter_methods[$element];
            $this->$method_name($value, $condition);
        }

    }

    /**
     * Apply coupons filter (include/exclude by ID or name)
     * 
     * @param mixed $value Coupon IDs or names
     * @param string $condition Include or exclude
     */
    private function apply_coupons_filter($value, $condition) {
        if (empty($value) || !is_array($value)) {
            return;
        }
        
        // Separate IDs and names efficiently
        $coupon_ids = array();
        $coupon_names = array();
        
        foreach ($value as $item) {
            $trimmed_item = trim($item);
            if (is_numeric($trimmed_item)) {
                $coupon_ids[] = intval($trimmed_item);
            } else {
                $coupon_names[] = $trimmed_item;
            }
        }


        
        // Build optimized query conditions
        $conditions = array();
        
        // Handle IDs
        if (!empty($coupon_ids)) {
            $id_condition = 'include' === $condition ? 'IN' : 'NOT IN';
            $conditions[] = "ID {$id_condition} (" . implode(',', $coupon_ids) . ")";
        }
        
        // Handle names - optimized for exact matches
        if (!empty($coupon_names)) {
            global $wpdb;
            $names_sql = array();
            foreach ($coupon_names as $name) {
                $names_sql[] = $wpdb->prepare('%s', $name);
            }
            $names_sql = implode(',', $names_sql);
            
            $name_condition = 'include' === $condition ? 'IN' : 'NOT IN';
            $conditions[] = "post_title {$name_condition} ({$names_sql})";
        }
        
        // Add combined condition if we have any
        if (!empty($conditions)) {
            $this->sql_fragments['base_conditions'][] = "(" . implode(' OR ', $conditions) . ")";
        }
    }

    /**
     * Apply coupon status filter
     * 
     * @param mixed $value Status values
     * @param string $condition Include or exclude
     */
    private function apply_coupon_status_filter($value, $condition) {
        global $wpdb;
        
        if (empty($value)) {
            return;
        }
        
        if (is_string($value)) {
            $value = array_filter(array_map('trim', explode(',', $value)));
        }
        
        if (empty($value) || !is_array($value)) {
            return;
        }
        
        // Use wpdb->prepare for each value
        $statuses_sql = array();
        foreach ($value as $status) {
            $statuses_sql[] = $wpdb->prepare('%s', $status);
        }
        $statuses_sql = implode(',', $statuses_sql);
        
        if ('include' === $condition) {
            $this->sql_fragments['base_conditions'][] = "post_status IN ($statuses_sql)";
        } elseif ('exclude' === $condition) {
            $this->sql_fragments['base_conditions'][] = "post_status NOT IN ($statuses_sql)";
        }
    }

    /**
     * Apply coupon type filter
     * 
     * @param mixed $value Coupon type values
     * @param string $condition Include or exclude
     */
    private function apply_coupon_type_filter($value, $condition) {
        global $wpdb;
        
        if (empty($value)) {
            return;
        }
        
        if (is_string($value)) {
            $value = array_filter(array_map('trim', explode(',', $value)));
        }
        
        if (empty($value) || !is_array($value)) {
            return;
        }
        
        // Use wpdb->prepare for each value
        $types_sql = array();
        foreach ($value as $type) {
            $types_sql[] = $wpdb->prepare('%s', $type);
        }
        $types_sql = implode(',', $types_sql);
        
        if ('include' === $condition) {
            $this->sql_fragments['base_conditions'][] = "ID IN (
                SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                WHERE meta_key = 'discount_type' 
                AND meta_value IN ($types_sql)
            )";
        } elseif ('exclude' === $condition) {
            $this->sql_fragments['base_conditions'][] = "ID NOT IN (
                SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                WHERE meta_key = 'discount_type' 
                AND meta_value IN ($types_sql)
            )";
        }
    }

    /**
     * Apply coupon amount filter
     * 
     * @param mixed $value Amount value or range
     * @param string $condition Filter condition
     */
    private function apply_coupon_amount_filter($value, $condition) {
        global $wpdb;
        
        if (empty($value)) {
            return;
        }
        
        if (is_array($value) && isset($value['from']) && isset($value['to'])) {
            // Handle amount range (Between)
            $from_amount = floatval($value['from']);
            $to_amount = floatval($value['to']);
            
            $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                WHERE meta_key = 'coupon_amount' 
                AND CAST(meta_value AS DECIMAL(10,2)) BETWEEN %f AND %f", $from_amount, $to_amount);
            
            $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
        } else {
            // Handle single amount
            $amount_value = floatval($value);
            
            switch ($condition) {
                case 'greater_than':
                    $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                        WHERE meta_key = 'coupon_amount' 
                        AND CAST(meta_value AS DECIMAL(10,2)) > %f", $amount_value);
                    $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                    break;
                case 'less_than':
                    $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                        WHERE meta_key = 'coupon_amount' 
                        AND CAST(meta_value AS DECIMAL(10,2)) < %f", $amount_value);
                    $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                    break;
                case 'equals':
                    $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                        WHERE meta_key = 'coupon_amount' 
                        AND CAST(meta_value AS DECIMAL(10,2)) = %f", $amount_value);
                    $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                    break;
                case 'not_equals':
                    $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                        WHERE meta_key = 'coupon_amount' 
                        AND CAST(meta_value AS DECIMAL(10,2)) != %f", $amount_value);
                    $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                    break;
            }
        }
    }

    /**
     * Apply expiry date filter
     * 
     * @param mixed $value Date value or range
     * @param string $condition Filter condition
     */
    private function apply_expiry_date_filter($value, $condition) {
        global $wpdb;

        if (empty($value)) {
            return;
        }
    
        if (is_array($value) && isset($value['from']) && isset($value['to'])) {
            $from_date = trim($value['from']);
            $to_date = trim($value['to']);
    
            if (!empty($from_date) && !empty($to_date)) {
                $from_timestamp = strtotime($from_date);
                $to_timestamp = strtotime($to_date);
                $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                    WHERE meta_key = 'date_expires' 
                    AND meta_value BETWEEN %d AND %d", $from_timestamp, $to_timestamp);
                $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
            } elseif (!empty($from_date)) {
                $from_timestamp = strtotime($from_date);
                $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                    WHERE meta_key = 'date_expires' 
                    AND meta_value >= %d", $from_timestamp);
                $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
            } elseif (!empty($to_date)) {
                $to_timestamp = strtotime($to_date);
                $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                    WHERE meta_key = 'date_expires' 
                    AND meta_value <= %d", $to_timestamp);
                $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
            }
    
            return;
        }
    
        // Single date handling
        $date_value = $value;
        $timestamp = strtotime($date_value);
    
        switch ($condition) {
            case 'greater_than':
                $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                    WHERE meta_key = 'date_expires' 
                    AND meta_value > %d", $timestamp);
                $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                break;
            case 'less_than':
                $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                    WHERE meta_key = 'date_expires' 
                    AND meta_value < %d", $timestamp);
                $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                break;
            case 'equals':
                $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                    WHERE meta_key = 'date_expires' 
                    AND DATE(FROM_UNIXTIME(meta_value)) = %s", $date_value);
                $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                break;
            case 'not_equals':
                $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                    WHERE meta_key = 'date_expires' 
                    AND DATE(FROM_UNIXTIME(meta_value)) != %s", $date_value);
                $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                break;
        }
    }
    

    /**
     * Apply usage count filter
     * 
     * @param mixed $value Usage count value or range
     * @param string $condition Filter condition
     */
    private function apply_usage_count_filter($value, $condition) {
        global $wpdb;
        
        if (empty($value)) {
            return;
        }
        
        if (is_array($value) && isset($value['from']) && isset($value['to'])) {
            // Handle count range (Between)
            $from_count = intval($value['from']);
            $to_count = intval($value['to']);
            
            $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                WHERE meta_key = 'usage_count' 
                AND CAST(meta_value AS UNSIGNED) BETWEEN %d AND %d", $from_count, $to_count);
            
            $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
        } else {
            // Handle single count
            $count_value = intval($value);
            
            switch ($condition) {
                case 'greater_than':
                    $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                        WHERE meta_key = 'usage_count' 
                        AND CAST(meta_value AS UNSIGNED) > %d", $count_value);
                    $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                    break;
                case 'less_than':
                    $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                        WHERE meta_key = 'usage_count' 
                        AND CAST(meta_value AS UNSIGNED) < %d", $count_value);
                    $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                    break;
                case 'equals':
                    $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                        WHERE meta_key = 'usage_count' 
                        AND CAST(meta_value AS UNSIGNED) = %d", $count_value);
                    $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                    break;
                case 'not_equals':
                    $subquery = $wpdb->prepare("SELECT post_id FROM {$GLOBALS['wpdb']->postmeta} 
                        WHERE meta_key = 'usage_count' 
                        AND CAST(meta_value AS UNSIGNED) != %d", $count_value);
                    $this->sql_fragments['base_conditions'][] = "ID IN ({$subquery})";
                    break;
            }
        }
    }

    /**
     * Build the final optimized SQL query from collected fragments
     */
    private function build_final_sql_query($export_limit, $current_offset, $sort_by, $sort_order) {
        global $wpdb;
        
        // Build base query efficiently
        $where_conditions = array("post_type = 'shop_coupon'");
        
        // Add custom filter conditions
        if (!empty($this->sql_fragments['base_conditions'])) {
            $where_conditions = array_merge($where_conditions, $this->sql_fragments['base_conditions']);
        }
        
        // Add default status filter if none specified
        if (!strpos(implode(' ', $where_conditions), 'post_status')) {
            $where_conditions[] = "post_status IN ('publish', 'pending', 'private', 'draft', 'future')";
        }
        
        // Build final WHERE clause
        $final_where = "WHERE " . implode(' AND ', $where_conditions);
        
        // Apply filter hook to the final query
        $sql = "SELECT DISTINCT ID 
                FROM {$wpdb->posts} 
                {$final_where}
                ORDER BY {$sort_by} {$sort_order} 
                LIMIT {$export_limit} OFFSET {$current_offset}";
        
        return apply_filters("wt_ier_coupon_sql_query", $sql, $where_conditions);
    }
    
    public function generate_row_data($coupon) {
        $csv_columns = $this->parent_module->get_selected_column_names();                
        $row = array();
        
        foreach ($csv_columns as $column => $value) {  
                                    
            if (isset($coupon->$column)) {
                if (is_array($coupon->$column)) {
                    $coupon->$column = implode(",", $coupon->$column);
                }
                if($column == 'product_ids'){
                    $hf_val = $coupon->$column;
                    $sku = self::get_sku_from_id($hf_val);
                    $row[$column] = str_replace('|', ',', $hf_val);
                    continue;
                }
                if($column == 'exclude_product_ids'){
                    $ex_val = $coupon->$column;
                    $exsku = self::get_sku_from_id($ex_val);
                    $row[$column] = str_replace('|', ',', $ex_val);
                    continue;
                }
                
               if($column == 'product_categories' || $column == 'exclude_product_categories'){

                    $cpn_product_category_ids = explode(',', $coupon->$column);
                    $cpn_product_category_name = array();
                    foreach ($cpn_product_category_ids as $cpn_product_category_id) {
                       $cpn_product_category_name[] = get_term( $cpn_product_category_id )->name;
                    }
                    $row[$column] = implode(',', $cpn_product_category_name);
                    continue;
                }                
                
                if('date_expires' == $column && !empty($coupon->$column)){
                    if(strtotime($coupon->$column) == false){
                        $date_expires_in_time_stamp = $coupon->$column;
                    }else{
                        $date_expires_in_time_stamp = strtotime($coupon->$column);
                    }
                    // Use wp_date() to properly handle WordPress timezone settings
                    $row[$column] = wp_date('Y-m-d', $date_expires_in_time_stamp);

                    continue;
                }

                $row[$column] = $coupon->$column;
                continue;
            } elseif (isset($coupon->$column) && !is_array($coupon->$column)) {
                if ($column === 'post_title') {
                    $row[$column] = sanitize_text_field($coupon->$column);
                } else {
                    $row[$column] = $coupon->$column;
                }
                continue;
            }
            elseif ($column === 'product_SKUs') {
                $row[$column] = !empty($sku) ? $sku : '';
                unset($sku);
                continue;
            }
            elseif ($column === 'exclude_product_SKUs') {
                $row[$column] = !empty($exsku) ? $exsku : '';
                unset($exsku);
                continue;
            }
            
            if ($this->export_shortcodes && ( 'post_content' == $column || 'post_excerpt' == $column )) {
                //Convert Shortcodes to html for Description and Short Description
                $row[$column] = do_shortcode($coupon->$column);
                continue;
            }

			if ('wt_iew_coupon_categories' === $column) {

				$categories = get_the_terms($coupon->ID, 'shop_coupon_cat');
				$category_items = array();
				if (is_array($categories) && !empty($categories)) {
					foreach ($categories as $category) {
						$category_items[] = trim( $category->name );
					}
				}
				$row[$column] = implode(',', $category_items);
				continue;
			}

			// Export meta data                
           if ('meta:' == substr($column, 0, 5)) {

               $meta = substr($column, 5);
               if (isset($coupon->$meta)) {
                   
				   $coupon_serialized_metas = apply_filters('wt_iew_coupon_serialized_metas', array('meta:wt_order_Status_need_to_count'));
				   if( in_array( $column, $coupon_serialized_metas ) ){
					   $row[$column] = json_encode( maybe_unserialize( $coupon->$meta ) );
				   }else{
					   $row[$column] = maybe_serialize($coupon->$meta);
				   }
               } else {
                   $row[$column] = '';
               }

               continue;
           }
           
            $row[$column] = '';
        }
        return apply_filters('hf_alter_coupon_csv_data', $row, $csv_columns);

    }
    
    public static function get_sku_from_id($val){
        $pro_id = explode(",", $val);
        $sku_arr = array();
        if($pro_id){
            foreach ($pro_id as $value){
                $product_exist = get_post_type($value);
                if ($product_exist == 'product' || $product_exist == 'product_variation'){
                    $psku = get_post_meta($value,'_sku',TRUE);
                    if(!empty($psku)){
                        $sku_arr[] = $psku;
                    }
                }
            }
        }
        $new_sku = implode("|", $sku_arr);
        return $new_sku;
    }



}
