<?php

if (!defined('WPINC')) {
	exit;
}
use Automattic\WooCommerce\Utilities\OrderUtil;
class Wt_Import_Export_For_Woo_Order_Export {

	public $parent_module = null;
	public $table_name ;
	public $is_hpos_enabled ;
	public $hpos_sync ;
	private $line_items_max_count = 0;
	private $export_to_separate_columns = false;
	private $export_to_separate_rows = false;
	private $exclude_line_items = false;
	
	private $sql_fragments = array(
		'left_joins' => array(),
		'where_conditions' => array(),
		'subqueries' => array(),
		'base_conditions' => array()
	);

	public function __construct($parent_object) {

		$this->parent_module = $parent_object;
        $hpos_data = Wt_Import_Export_For_Woo_Common_Helper::is_hpos_enabled();
        $this->table_name = $hpos_data['table_name'];
		$this->hpos_sync = $hpos_data['sync'];
        if( strpos($hpos_data['table_name'],'wc_orders') !== false ){
			$this->is_hpos_enabled = true;
        }
	}

	public function prepare_header() {

		$export_columns = $this->parent_module->get_selected_column_names();

		if ($this->exclude_line_items) {
			return apply_filters('hf_alter_csv_header', $export_columns, 0);
		}

		$max_line_items = $this->line_items_max_count;

		for ($i = 1; $i <= $max_line_items; $i++) {
			$export_columns["line_item_{$i}"] = "line_item_{$i}";
		}
		if ($this->export_to_separate_columns) {
			for ($i = 1; $i <= $max_line_items; $i++) {
				$export_columns["line_item_{$i}_name"] = "Product Item {$i} Name";
				$export_columns["line_item_{$i}_product_id"] = "Product Item {$i} id";
				$export_columns["line_item_{$i}_sku"] = "Product Item {$i} SKU";
				$export_columns["line_item_{$i}_quantity"] = "Product Item {$i} Quantity";
				$export_columns["line_item_{$i}_total"] = "Product Item {$i} Total";
				$export_columns["line_item_{$i}_subtotal"] = "Product Item {$i} Subtotal";
			}
		}

		if ($this->export_to_separate_rows) {
			$export_columns = $this->wt_line_item_separate_row_csv_header($export_columns);
		}

		return apply_filters('hf_alter_csv_header', $export_columns, $max_line_items);
	}

	public function wt_line_item_separate_row_csv_header($export_columns) {


		foreach ($export_columns as $s_key => $value) {
			if (strstr($s_key, 'line_item_')) {
				unset($export_columns[$s_key]);
			}
		}

		$export_columns["line_item_product_id"] = "item_product_id";
		$export_columns["line_item_name"] = "item_name";
		$export_columns["line_item_sku"] = "item_sku";
		$export_columns["line_item_quantity"] = "item_quantity";
		$export_columns["line_item_subtotal"] = "item_subtotal";
		$export_columns["line_item_subtotal_tax"] = "item_subtotal_tax";
		$export_columns["line_item_total"] = "item_total";
		$export_columns["line_item_total_tax"] = "item_total_tax";
		
		$export_columns["item_refunded"] = "item_refunded";
		$export_columns["item_refunded_qty"] = "item_refunded_qty";
		$export_columns["item_meta"] = "item_meta";
		return $export_columns;
	}

	public function wt_line_item_separate_row_csv_data($order, $order_export_data, $order_data_filter_args) {
		if ($order) {
			if($order->get_items()){
				foreach ($order->get_items() as $item_key => $item) {
					foreach ($order_export_data as $key => $value) {
						if (strpos($key, 'line_item_') !== false) {
							continue;
						} else {
							$data1[$key] = $value;
						}
					}
					$item_data = $item->get_data();
					$product = $item->get_product();

					$data1["line_item_product_id"] = !empty($item_data['product_id']) ? $item_data['product_id'] : '';
					$data1["line_item_name"] = !empty($item_data['name']) ? $item_data['name'] : '';
					$data1["line_item_sku"] = !empty($product) ? $product->get_sku() : '';
					$data1["line_item_quantity"] = !empty($item_data['quantity']) ? $item_data['quantity'] : '';
					$data1["line_item_subtotal"] = !empty($item_data['subtotal']) ? $item_data['subtotal'] : 0;
					$data1["line_item_subtotal_tax"] = !empty($item_data['subtotal_tax']) ? $item_data['subtotal_tax'] : 0;
					$data1["line_item_total"] = !empty($item_data['total']) ? $item_data['total'] : 0;
					$data1["line_item_total_tax"] = !empty($item_data['total_tax']) ? $item_data['total_tax'] : 0;

					$data1["item_refunded"] = !empty($order->get_total_refunded_for_item($item_key)) ? $order->get_total_refunded_for_item($item_key) : '';
					$data1["item_refunded_qty"] = !empty($order->get_qty_refunded_for_item($item_key)) ? absint($order->get_qty_refunded_for_item($item_key)) : '';
					$meta_data_array = array();
					foreach($item_data['meta_data'] as $meta_data){
						$meta_data_array[] = $meta_data->key . ':' .  $meta_data->value ;
					}
					$meta_data_array = implode('|', $meta_data_array);
					$data1["item_meta"] = !empty($meta_data_array) ? $meta_data_array : '';
					$row[] = $data1;
				}
			}else{
				$row[] = $order_export_data;
			}
			return $row;
		}
	}
	public function wt_ier_alter_order_data_befor_export_for_separate_row($data_array) {
		$new_data_array = array();
		foreach ($data_array as $key => $avalue) {
			if (is_array($avalue)) {
				if (count($avalue) == 1) {
					$new_data_array[] = $avalue[0];
				} elseif (count($avalue) > 1) {
					foreach ($avalue as $arrkey => $arrvalue) {
						$new_data_array[] = $arrvalue;
					}
				}
			}
		}
		return $new_data_array;
	}

	/**
	 * Prepare data that will be exported.
	 */
	public function prepare_data_to_export($form_data, $batch_offset) {
		
		// Initialize SQL fragment collectors
		$this->sql_fragments = array(
			'left_joins' => array(),
			'where_conditions' => array(),
			'subqueries' => array(),
			'base_conditions' => array('1') // Always true base condition
		);
		
		// Extract basic parameters (same as before)
		$export_limit = !empty($form_data['filter_form_data']['wt_iew_limit']) ? intval($form_data['filter_form_data']['wt_iew_limit']) : 999999999;
		$current_offset = !empty($form_data['filter_form_data']['wt_iew_offset']) ? intval($form_data['filter_form_data']['wt_iew_offset']) : 0;
		$batch_count = !empty($form_data['advanced_form_data']['wt_iew_batch_count']) ? $form_data['advanced_form_data']['wt_iew_batch_count'] : Wt_Import_Export_For_Woo_Common_Helper::get_advanced_settings('default_export_batch');
		$exclude_already_exported = (!empty($form_data['advanced_form_data']['wt_iew_exclude_already_exported']) && $form_data['advanced_form_data']['wt_iew_exclude_already_exported'] == 'Yes') ? true : false;
		
		// Handle exclude already exported orders
		if ($exclude_already_exported) {
			$this->add_exclude_already_exported_filter();
		}
		
		/**
		 * Process filters.
		 * * Build the final optimized SQL query.
		 * @since 1.3.1
		 */
		$this->filter_orders_to_export($form_data);
		$order_sql_query = $this->build_final_sql_query($export_limit, $current_offset);

		$filter_form_data = isset($form_data['filter_form_data']) && is_array($form_data['filter_form_data']) ? $form_data['filter_form_data'] : array();
		$advanced_form_data = is_array($form_data['advanced_form_data']) ? $form_data['advanced_form_data'] : array();
		$transient_key = 'wt_iew_orders_export_hybrid_' . md5(wp_json_encode(array_merge($filter_form_data, $advanced_form_data)));
		
		$total_queried_orders = get_transient($transient_key);
		if (false === $total_queried_orders) {
			// Execute single query and cache results
			global $wpdb;
			$total_queried_orders = $wpdb->get_col($order_sql_query);
			set_transient($transient_key, $total_queried_orders, 60); // Cache for 60 seconds
		}
		
		// Use cached order IDs with batch processing
		$order_ids = apply_filters('wt_orderimpexpcsv_alter_order_ids', $total_queried_orders);
		$total_records = count($order_ids);
		
		// Setup export options
		$this->export_to_separate_columns = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate_columns']) && $form_data['advanced_form_data']['wt_iew_export_to_separate_columns'] == 'Yes') ? true : false;
		if (!$this->export_to_separate_columns) {
			$this->export_to_separate_columns = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate']) && $form_data['advanced_form_data']['wt_iew_export_to_separate'] == 'column') ? true : false;
		}

		$this->export_to_separate_rows = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate_rows']) && $form_data['advanced_form_data']['wt_iew_export_to_separate_rows'] == 'Yes') ? true : false;
		if (!$this->export_to_separate_rows) {
			$this->export_to_separate_rows = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate']) && $form_data['advanced_form_data']['wt_iew_export_to_separate'] == 'row') ? true : false;
		}
		$this->exclude_line_items = (!empty($form_data['advanced_form_data']['wt_iew_exclude_line_items']) && $form_data['advanced_form_data']['wt_iew_exclude_line_items'] == 'Yes') ? true : false;
		
		// Calculate batch limits
		if ($batch_count <= $export_limit) {
			if (($batch_offset + $batch_count) > $export_limit) { //last offset
				$limit = $export_limit - $batch_offset;
			} else {
				$limit = $batch_count;
			}
		} else {
			$limit = $export_limit - $batch_offset;
		}
		
		$data_array = array();
		if ($batch_offset < $export_limit) {
			
			// Set up line items max count for first batch
			if ($batch_offset == 0) {
				$this->line_items_max_count = $this->get_max_line_items();
				update_option('wt_order_line_items_max_count', $this->line_items_max_count);
			}

			if (empty($this->line_items_max_count)) {
				$this->line_items_max_count = get_option('wt_order_line_items_max_count');
			}
			
			// Process batch of order IDs
			$batch_order_ids = array_slice($order_ids, $batch_offset, $limit);
			
			foreach ($batch_order_ids as $order_id) {
				if (wc_get_order($order_id)) {
					$data_array[] = $this->generate_row_data($order_id, $form_data);
					
					// Update exported status
					if ($this->is_hpos_enabled) {
						if ($this->hpos_sync) {
							update_post_meta($order_id, 'wf_order_exported_status', TRUE);
						}
						global $wpdb;
						$wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->prefix}wc_orders_meta WHERE meta_key = 'wf_order_exported_status' AND order_id = %d;", $order_id));
						$wpdb->query($wpdb->prepare("INSERT INTO {$wpdb->prefix}wc_orders_meta (order_id, meta_key, meta_value) VALUES (%d, %s, %s)", $order_id, 'wf_order_exported_status', TRUE));
					} else {
						update_post_meta($order_id, 'wf_order_exported_status', TRUE);
					}
				}
			}

			if ($this->export_to_separate_rows) {
				$data_array = $this->wt_ier_alter_order_data_befor_export_for_separate_row($data_array);
			}

			$data_array = apply_filters('wt_ier_alter_order_data_befor_export', $data_array);
		}
		
		// Clean up cache on last batch
		if (($batch_offset + $batch_count) >= $export_limit) {
			delete_transient($transient_key);
		}
		
		$return = array();
		$return['total'] = $total_records;
		$return['data'] = $data_array;
		if (0 === $batch_offset && 0 === $total_records) {
			$return['no_post'] = __('Nothing to export under the selected criteria. Please check and try adjusting the filters.','wt-import-export-for-woo');
		}
		return $return;
	}

	/**
	 * Main filter processing function
	 */
	public function filter_orders_to_export($form_data) {
		// Check if multi-field filter data exists
		if ( empty( $form_data['filter_form_data']['wt_iew_multi_field_row_array'] ) ) {
			return;
		}
		
		// Decode the JSON string to get the filter array
		$filter_array = json_decode($form_data['filter_form_data']['wt_iew_multi_field_row_array'], true);
		
		if (empty($filter_array)) {
			return;
		}
		
		// Process each filter row and collect SQL fragments
		foreach ($filter_array as $key => $filter_row) {
			$element = isset($filter_row['element']) ? $filter_row['element'] : '';
			$value = isset($filter_row['value']) ? $filter_row['value'] : '';
			$condition = isset($filter_row['condition']) ? $filter_row['condition'] : '';
			
			// Skip empty filters
			if (empty($element)) {
				continue;
			}
			
			// Validate values based on filter type
			if (!$this->validate_filter_value($element, $value, $condition)) {
				continue;
			}
			
			// Apply filter and collect SQL fragments
			$this->apply_filter_by_element($element, $value, $condition);
		}
	}

	/**
	 * Element mapping system - returns SQL fragments instead of executing queries
	 */
	private function apply_filter_by_element($element, $value, $condition) {
		$filter_methods = array(
			'orders' => 'apply_order_filter',
			'order_date' => 'apply_order_date_filter',
			'order_status' => 'apply_order_status_filter',
			'customer' => 'apply_customer_filter',
			'user_role' => 'apply_user_role_filter',
			'billing_address' => 'apply_billing_address_filter',
			'shipping_address' => 'apply_shipping_address_filter',
			'order_total' => 'apply_order_total_filter',
			'payment_method' => 'apply_payment_method_filter',
			'shipping_method' => 'apply_shipping_method_filter',
			'item_ordered' => 'apply_item_ordered_filter',
			'product_category' => 'apply_product_category_filter',
			'vendor' => 'apply_vendor_filter',
			'coupons' => 'apply_coupons_filter',
			'completed_date' => 'apply_completed_date_filter',
			'paid_date' => 'apply_paid_date_filter',
			'modification_date' => 'apply_modification_date_filter'
		);
		
		if (isset($filter_methods[$element]) && method_exists($this, $filter_methods[$element])) {
			$method_name = $filter_methods[$element];
			$this->$method_name($value, $condition);
		}
	}

	/**
	 * Order Filter (Include/Exclude by ID or order number)
	 */
	private function apply_order_filter($value, $condition) {
		// Handle different input formats
		if (is_array($value)) {
			$order_ids = $value;
		} elseif (is_string($value)) {
			// Split comma-separated string into array
			$order_ids = array_filter(array_map('trim', explode(',', $value)));
		} else {
			$order_ids = array($value);
		}
		
		// Remove empty values and ensure integers
		$order_ids = array_filter(array_map('intval', $order_ids));
		
		// Convert order numbers to order IDs if needed
		$order_ids = apply_filters('wt_oiew_alter_filter_by_order_ids', $order_ids);
		$order_ids = is_array($order_ids) ? $order_ids : array();
		
		if (!empty($order_ids)) {
			$orders_sql = self::prepare_sql_inarray($order_ids);
						
			if ('include' === $condition) {
				$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} IN ($orders_sql)";
			} elseif ('exclude' === $condition) {
				$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} NOT IN ($orders_sql)";
			}
		}
	}

	/**
	 * Order Date Filter (Default)
	 */
	private function apply_order_date_filter($value, $condition) {
		global $wpdb;
		$date_field = $this->is_hpos_enabled ? 'date_created_gmt' : 'post_date';
		
		if (is_array($value) && isset($value['from']) && isset($value['to'])) {
			// Handle date range (Between)
			$from_date = apply_filters('wt_iew_alter_order_start_date', $value['from']);
			$to_date = apply_filters('wt_iew_alter_order_end_date', $value['to']);
			
			$from_date = $from_date . ' 00:00:00';
			$to_date = $to_date . ' 23:59:59.99';
			
			// Use wpdb->prepare for safe SQL
			$prepared_condition = $wpdb->prepare("orders.{$date_field} BETWEEN %s AND %s", $from_date, $to_date);
			$this->sql_fragments['base_conditions'][] = $prepared_condition;
		} else {
			// Handle single date
			$date_value = $value;
			
			switch ($condition) {
				case 'greater_than':
					$prepared_condition = $wpdb->prepare("orders.{$date_field} > %s", $date_value . ' 23:59:59.99');
					$this->sql_fragments['base_conditions'][] = $prepared_condition;
					break;
				case 'less_than':
					$prepared_condition = $wpdb->prepare("orders.{$date_field} < %s", $date_value . ' 00:00:00');
					$this->sql_fragments['base_conditions'][] = $prepared_condition;
					break;
				case 'equals':
					$prepared_condition = $wpdb->prepare("DATE(orders.{$date_field}) = %s", $date_value);
					$this->sql_fragments['base_conditions'][] = $prepared_condition;
					break;
				case 'not_equals':
					$prepared_condition = $wpdb->prepare("DATE(orders.{$date_field}) != %s", $date_value);
					$this->sql_fragments['base_conditions'][] = $prepared_condition;
					break;
			}
		}
	}

	/**
	 * Order Status Filter
	 */
	private function apply_order_status_filter($value, $condition) {
		$statuses = is_array($value) ? $value : array($value);
		$statuses_sql = self::prepare_sql_inarray($statuses);
		
		if ('include' === $condition || 'equals' === $condition) {
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_status_field()} IN ($statuses_sql)";
		} elseif ('exclude' === $condition || 'not_equals' === $condition) {
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_status_field()} NOT IN ($statuses_sql)";
		}
	}

	/**
	 * Customer Filter (General customer search using customer IDs)
	 */
	private function apply_customer_filter($value, $condition) {
		// Handle customer IDs - $value should be an array of customer IDs from wc-customer-search
		$customer_ids = is_array($value) ? $value : array($value);
		$customers_sql = self::prepare_sql_inarray($customer_ids);
		
		if ('include' === $condition) {
			if ($this->is_hpos_enabled) {
				$this->sql_fragments['base_conditions'][] = "orders.customer_id IN ({$customers_sql})";
			} else {
				$this->sql_fragments['left_joins'][] = "LEFT JOIN {$this->get_meta_table()} AS ordermeta_customer ON ordermeta_customer.{$this->get_post_id_field()} = orders.{$this->get_order_id_field()}";
				$this->sql_fragments['where_conditions'][] = "(ordermeta_customer.meta_key = '_customer_user' AND ordermeta_customer.meta_value IN ({$customers_sql}))";
			}
		} elseif ('exclude' === $condition) {
			if ($this->is_hpos_enabled) {
				$this->sql_fragments['base_conditions'][] = "orders.customer_id NOT IN ({$customers_sql})";
			} else {
				$this->sql_fragments['left_joins'][] = "LEFT JOIN {$this->get_meta_table()} AS ordermeta_customer ON ordermeta_customer.{$this->get_post_id_field()} = orders.{$this->get_order_id_field()}";
				$this->sql_fragments['where_conditions'][] = "(ordermeta_customer.meta_key = '_customer_user' AND ordermeta_customer.meta_value NOT IN ({$customers_sql}))";
			}
		}
	}

	/**
	 * User Role Filter
	 */
	private function apply_user_role_filter($value, $condition) {
		global $wpdb;
		
		// Handle user roles - $value should be an array of role names
		$user_roles = is_array($value) ? $value : array($value);
		
		// Build subquery to get users with specific roles using prepared statements
		$subquery = "SELECT DISTINCT u.ID 
		             FROM {$wpdb->users} u
		             INNER JOIN {$wpdb->usermeta} um ON u.ID = um.user_id
		             WHERE um.meta_key = %s
		             AND (";
		
		$role_conditions = array();
		$prepare_values = array($wpdb->prefix . 'capabilities');
		
		foreach ($user_roles as $role) {
			$role_conditions[] = "um.meta_value LIKE %s";
			$prepare_values[] = '%"' . $wpdb->esc_like($role) . '"%';
		}
		$subquery .= implode(' OR ', $role_conditions) . ")";
		
		// Prepare the entire subquery
		$prepared_subquery = $wpdb->prepare($subquery, $prepare_values);
		
		if ('equals' === $condition || 'include' === $condition) {
			if ($this->is_hpos_enabled) {
				$this->sql_fragments['base_conditions'][] = "orders.customer_id IN ({$prepared_subquery})";
			} else {
				$this->sql_fragments['left_joins'][] = "LEFT JOIN {$this->get_meta_table()} AS ordermeta_customer_role ON ordermeta_customer_role.{$this->get_post_id_field()} = orders.{$this->get_order_id_field()}";
				$this->sql_fragments['where_conditions'][] = "(ordermeta_customer_role.meta_key = '_customer_user' AND ordermeta_customer_role.meta_value IN ({$prepared_subquery}))";
			}
		} elseif ('not_equals' === $condition || 'exclude' === $condition) {
			if ($this->is_hpos_enabled) {
				$this->sql_fragments['base_conditions'][] = "orders.customer_id NOT IN ({$prepared_subquery})";
			} else {
				$this->sql_fragments['left_joins'][] = "LEFT JOIN {$this->get_meta_table()} AS ordermeta_customer_role ON ordermeta_customer_role.{$this->get_post_id_field()} = orders.{$this->get_order_id_field()}";
				$this->sql_fragments['where_conditions'][] = "(ordermeta_customer_role.meta_key = '_customer_user' AND ordermeta_customer_role.meta_value NOT IN ({$prepared_subquery}))";
			}
		}
	}

	/**
	 * Shared method for product-based filtering (vendor, category, item ordered)
	 * 
	 * @param array $product_ids Array of product IDs to filter by
	 * @param string $condition Filter condition (include/exclude/equals/not_equals)
	 * @param string $filter_type Type of filter for debugging (optional)
	 */
	private function apply_product_based_filter($product_ids, $condition, $filter_type = '') {
		if (empty($product_ids)) {
			// If no products provided, handle appropriately
			if (in_array($condition, array('include', 'equals'))) {
				// No orders should match (return impossible condition)
				$this->sql_fragments['base_conditions'][] = "1 = 0";
			}
			// For exclude/not_equals, we don't need to add any condition (all orders match)
			return;
		}
		
		$products_sql = self::prepare_sql_inarray($product_ids);
		
		// Create subquery to find orders containing the specified products
		$subquery = "SELECT DISTINCT order_items.order_id 
					 FROM {$this->get_wpdb_prefix()}woocommerce_order_items as order_items
					 LEFT JOIN {$this->get_wpdb_prefix()}woocommerce_order_itemmeta AS orderitemmeta_product 
						 ON orderitemmeta_product.order_item_id = order_items.order_item_id
					 WHERE order_items.order_item_type = 'line_item' 
					 AND orderitemmeta_product.meta_key IN ('_variation_id', '_product_id') 
					 AND orderitemmeta_product.meta_value IN ({$products_sql})";
		
		// Apply the filter condition
		if (in_array($condition, array('include', 'equals'))) {
			// Include orders that contain these products
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} IN ({$subquery})";
		} elseif (in_array($condition, array('exclude', 'not_equals'))) {
			// Exclude orders that contain these products
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} NOT IN ({$subquery})";
		}

	}

	/**
	 * Order Total Filter with range support
	 */
	private function apply_order_total_filter($value, $condition) {
		global $wpdb;

		if (is_array($value) && isset($value['from']) && isset($value['to'])) {
			// Handle range (Between condition)
			$from_total = number_format(floatval($value['from']), 4, '.', '');
			$to_total = number_format(floatval($value['to']), 4, '.', '');
			
			if ($this->is_hpos_enabled) {
				// Use ROUND for consistent precision matching
				$prepared_condition = $wpdb->prepare("ROUND(orders.total_amount, 4) BETWEEN %f AND %f", $from_total, $to_total);
				$this->sql_fragments['base_conditions'][] = $prepared_condition;
			} else {
				$this->sql_fragments['left_joins'][] = "LEFT JOIN {$this->get_meta_table()} AS ordermeta_total ON ordermeta_total.{$this->get_post_id_field()} = orders.{$this->get_order_id_field()}";
				$prepared_condition = $wpdb->prepare("(ordermeta_total.meta_key = '_order_total' AND ROUND(CAST(ordermeta_total.meta_value AS DECIMAL(10,4)), 4) BETWEEN %f AND %f)", $from_total, $to_total);
				$this->sql_fragments['where_conditions'][] = $prepared_condition;
			}
			
		} else {
			// Handle single value
			$total_value = number_format(floatval($value), 4, '.', '');
			$operator = $this->get_numeric_compare_operator($condition);
			
			if ($this->is_hpos_enabled) {
				// Use ROUND for consistent precision matching
				$prepared_condition = $wpdb->prepare("ROUND(orders.total_amount, 4) {$operator} %f", $total_value);
				$this->sql_fragments['base_conditions'][] = $prepared_condition;
			} else {
				$this->sql_fragments['left_joins'][] = "LEFT JOIN {$this->get_meta_table()} AS ordermeta_total ON ordermeta_total.{$this->get_post_id_field()} = orders.{$this->get_order_id_field()}";
				$prepared_condition = $wpdb->prepare("(ordermeta_total.meta_key = '_order_total' AND ROUND(CAST(ordermeta_total.meta_value AS DECIMAL(10,4)), 4) {$operator} %f)", $total_value);
				$this->sql_fragments['where_conditions'][] = $prepared_condition;
			}
			
		}
	}

	/**
	 * Item Ordered Filter (Products) - Direct product selection
	 */
	private function apply_item_ordered_filter($value, $condition) {

		// Direct product IDs from product search
		$product_ids = is_array($value) ? $value : array($value);
		$this->apply_product_based_filter($product_ids, $condition, 'item_ordered');
	}

	/**
	 * Coupons Filter
	 */
	private function apply_coupons_filter($value, $condition) {
		$coupons = is_array($value) ? $value : array($value);
		$coupons_sql = self::prepare_sql_inarray($coupons);
		
		if ('equals' === $condition) {
			$subquery = "SELECT DISTINCT order_coupons.order_id 
						 FROM {$this->get_wpdb_prefix()}woocommerce_order_items as order_coupons
						 WHERE order_coupons.order_item_type = 'coupon' 
						 AND order_coupons.order_item_name IN ({$coupons_sql})";
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} IN ({$subquery})";
		} else {
			// Use NOT EXISTS instead of NOT IN to avoid NULL issues
			$subquery = "NOT EXISTS (
							SELECT 1 
							FROM {$this->get_wpdb_prefix()}woocommerce_order_items as order_coupons
							WHERE order_coupons.order_id = orders.{$this->get_order_id_field()}
							AND order_coupons.order_item_type = 'coupon' 
							AND order_coupons.order_item_name IN ({$coupons_sql})
						)";
			$this->sql_fragments['base_conditions'][] = $subquery;
		}
	}

	/**
	 * Billing Address Filter - Filters by billing country or state
	 */
	private function apply_billing_address_filter($value, $condition) {
		$this->apply_address_filter($value, $condition, 'billing');
	}

	/**
	 * Shipping Address Filter - Filters by shipping country or state
	 */
	private function apply_shipping_address_filter($value, $condition) {
		$this->apply_address_filter($value, $condition, 'shipping');
	}

	/**
	 * Shared method for address filtering (billing/shipping, country/state)
	 * 
	 * @param array $value Address filter value with 'type' and 'value' keys
	 * @param string $condition Filter condition (contains/not_contains/starts_with/ends_with/equals/not_equals)
	 * @param string $address_type Either 'billing' or 'shipping'
	 */
	private function apply_address_filter($value, $condition, $address_type) {
		// Validate input structure
		if (!is_array($value) || empty($value['type']) || empty($value['value'])) {
			return;
		}
		
		$field_type = $value['type'];
		$field_value = $value['value'];
		
		// Validate field type
		if (!in_array($field_type, array('country', 'state'))) {
			return;
		}
	
		// Determine operator and prepare search value
		$is_multiple = is_array($field_value) && count($field_value) > 1;
		
		if ($is_multiple) {
			$operator = ('include' === $condition) ? 'IN' : 'NOT IN';
			$search_value = '(' . self::prepare_sql_inarray($field_value) . ')';
		} else {
			$operator = ( 'include' === $condition ) ? '=' : '!=';
			global $wpdb;
			$single_value = is_array($field_value) ? $field_value[0] : $field_value;
			$search_value = "'" . $wpdb->esc_like($single_value) . "'";
		}
		
		// Build table alias and join
		$table_alias = "order_address_{$address_type}_{$field_type}";
		$join_sql = "LEFT JOIN {$this->get_wpdb_prefix()}wc_order_addresses AS {$table_alias} ON {$table_alias}.order_id = orders.{$this->get_order_id_field()}";
		
		// Build WHERE condition
		if ( $this->is_hpos_enabled ) {
			// HPOS: Use wc_order_addresses table
			$where_condition = "({$table_alias}.address_type = '{$address_type}' AND {$table_alias}.{$field_type} {$operator} {$search_value})";
		} else {
			// Legacy: Use postmeta table
			$meta_key = "_{$address_type}_{$field_type}";
			$table_alias = "ordermeta_{$address_type}_{$field_type}";
			$join_sql = "LEFT JOIN {$this->get_meta_table()} AS {$table_alias} ON {$table_alias}.{$this->get_post_id_field()} = orders.{$this->get_order_id_field()}";
			$where_condition = "({$table_alias}.meta_key = '{$meta_key}' AND {$table_alias}.meta_value {$operator} {$search_value})";
		}
		
		// Add to SQL fragments
		$this->sql_fragments['left_joins'][] = $join_sql;
		$this->sql_fragments['where_conditions'][] = $where_condition;
		
	}

	private function apply_payment_method_filter($value, $condition) {
		$payment_methods = is_array($value) ? $value : array($value);
		$methods_sql = self::prepare_sql_inarray($payment_methods);
		
		if ($this->is_hpos_enabled) {
			if ('include' === $condition) {
				$this->sql_fragments['base_conditions'][] = "orders.payment_method IN ($methods_sql)";
			} elseif ('exclude' === $condition) {
				$this->sql_fragments['base_conditions'][] = "orders.payment_method NOT IN ($methods_sql)";
			}
		} else {
			$this->sql_fragments['left_joins'][] = "LEFT JOIN {$this->get_meta_table()} AS ordermeta_payment_method ON ordermeta_payment_method.{$this->get_post_id_field()} = orders.{$this->get_order_id_field()}";
			
			if ('include' === $condition) {
				$this->sql_fragments['where_conditions'][] = "(ordermeta_payment_method.meta_key = '_payment_method' AND ordermeta_payment_method.meta_value IN ($methods_sql))";
			} elseif ('exclude' === $condition) {
				$this->sql_fragments['where_conditions'][] = "(ordermeta_payment_method.meta_key = '_payment_method' AND ordermeta_payment_method.meta_value NOT IN ($methods_sql))";
			}
		}
	}

	/**
	 * Shipping Method Filter - Filters orders by shipping methods
	 */
	private function apply_shipping_method_filter($value, $condition) {
		// Handle shipping method IDs - $value should be an array of method IDs from multi-select
		$method_ids = is_array($value) ? $value : array($value);
		
		if (empty($method_ids)) {
			return;
		}
		
		// Remove empty values and duplicates
		$method_ids = array_unique(array_filter($method_ids));
		if (empty($method_ids)) {
			return;
		}
		
		$method_ids_sql = self::prepare_sql_inarray($method_ids);
		
		// Single optimized query for simple method IDs
		$subquery = "SELECT DISTINCT oi.order_id 
					 FROM {$this->get_wpdb_prefix()}woocommerce_order_items oi
					 INNER JOIN {$this->get_wpdb_prefix()}woocommerce_order_itemmeta oim 
						 ON oim.order_item_id = oi.order_item_id
					 WHERE oi.order_item_type = 'shipping' 
					 AND oim.meta_key = 'method_id' 
					 AND oim.meta_value IN ({$method_ids_sql})";
		
		// Apply the filter condition
		if ('include' === $condition) {
			// Include orders that use these shipping methods
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} IN ({$subquery})";
		} elseif ('exclude' === $condition) {
			// Exclude orders that use these shipping methods
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} NOT IN ({$subquery})";
		}
	}

	/**
	 * Product Category Filter - Filters orders by product categories
	 */
	private function apply_product_category_filter($value, $condition) {
		// Handle category slugs/IDs - $value should be an array of category slugs from multi-select
		$category_slugs = is_array($value) ? $value : array($value);
		
		// Get all products in the selected categories using WooCommerce function
		$category_products = array();
		if (!empty($category_slugs)) {
			$category_products = wc_get_products(array(
				'category' => $category_slugs, 
				'return' => 'ids', 
				'limit' => -1,
				'status' => array('publish', 'private', 'draft')
			));
		}
		
		// Use the shared product-based filter method
		$this->apply_product_based_filter($category_products, $condition, 'product_category');
	}

	/**
	 * Vendor Filter - Filters orders by vendor (product author)
	 */
	private function apply_vendor_filter($value, $condition) {
		// Handle vendor IDs - $value should be an array of vendor user IDs from wc-customer-search
		$vendor_ids = is_array($value) ? $value : array($value);
		
		// Get all products authored by the selected vendors
		$vendor_products = array();
		foreach ($vendor_ids as $vendor_id) {
			if (!empty($vendor_id)) {
				$products = wc_get_products(array(
					'author' => $vendor_id, 
					'return' => 'ids', 
					'limit' => -1,
					'status' => array('publish', 'private', 'draft')
				));
				$vendor_products = array_merge($vendor_products, $products);
			}
		}
		
		// Remove duplicates and use the shared product-based filter method
		$vendor_products = array_unique($vendor_products);
		$this->apply_product_based_filter($vendor_products, $condition, 'vendor');
	}

	private function apply_completed_date_filter($value, $condition) {
		$this->apply_meta_date_filter('_date_completed', $value, $condition);
	}

	private function apply_paid_date_filter($value, $condition) {
		$this->apply_meta_date_filter('_date_paid', $value, $condition);
	}

	private function apply_modification_date_filter($value, $condition) {
		global $wpdb;
		$date_field = $this->is_hpos_enabled ? 'date_updated_gmt' : 'post_modified';
		
		if (is_array($value) && isset($value['from']) && isset($value['to'])) {
			// Handle date range (Between)
			$from_date = $value['from'] . ' 00:00:00';
			$to_date = $value['to'] . ' 23:59:59.99';
			
			// Use wpdb->prepare for safe SQL
			$prepared_condition = $wpdb->prepare("orders.{$date_field} BETWEEN %s AND %s", $from_date, $to_date);
			$this->sql_fragments['base_conditions'][] = $prepared_condition;
		} else {
			// Handle single date
			$date_value = $value;
			
			switch ($condition) {
				case 'greater_than':
					$prepared_condition = $wpdb->prepare("orders.{$date_field} > %s", $date_value . ' 23:59:59.99');
					$this->sql_fragments['base_conditions'][] = $prepared_condition;
					break;
				case 'less_than':
					$prepared_condition = $wpdb->prepare("orders.{$date_field} < %s", $date_value . ' 00:00:00');
					$this->sql_fragments['base_conditions'][] = $prepared_condition;
					break;
				case 'equals':
					$prepared_condition = $wpdb->prepare("DATE(orders.{$date_field}) = %s", $date_value);
					$this->sql_fragments['base_conditions'][] = $prepared_condition;
					break;
				case 'not_equals':
					$prepared_condition = $wpdb->prepare("DATE(orders.{$date_field}) != %s", $date_value);
					$this->sql_fragments['base_conditions'][] = $prepared_condition;
					break;
			}
		}
	}

	/**
	 * Helper method for meta date filters (completed_date, paid_date)
	 */
	private function apply_meta_date_filter($meta_key, $value, $condition) {
		global $wpdb;
		
		if ($this->is_hpos_enabled) {
			// For HPOS, use wc_order_stats table with correct field names
			$stats_field = str_replace('_date_', 'date_', $meta_key); // _date_completed -> date_completed
			if ('_date_paid' === $meta_key) {
				$stats_field = 'date_paid';
			}
			
			if (is_array($value) && isset($value['from']) && isset($value['to'])) {
				// Handle date range
				$from_date = $value['from'] . ' 00:00:00';
				$to_date = $value['to'] . ' 23:59:59';
				
				$subquery = $wpdb->prepare("SELECT order_id 
							 FROM {$this->get_wpdb_prefix()}wc_order_stats AS order_stats 
							 WHERE order_stats.{$stats_field} IS NOT NULL 
							 AND order_stats.{$stats_field} BETWEEN %s AND %s", $from_date, $to_date);
			} else {
				// Handle single date
				$date_value = $value;
				
				switch ($condition) {
					case 'greater_than':
						$subquery = $wpdb->prepare("SELECT order_id 
									 FROM {$this->get_wpdb_prefix()}wc_order_stats AS order_stats 
									 WHERE order_stats.{$stats_field} IS NOT NULL 
									 AND order_stats.{$stats_field} > %s", $date_value . ' 23:59:59');
						break;
					case 'less_than':
						$subquery = $wpdb->prepare("SELECT order_id 
									 FROM {$this->get_wpdb_prefix()}wc_order_stats AS order_stats 
									 WHERE order_stats.{$stats_field} IS NOT NULL 
									 AND order_stats.{$stats_field} < %s", $date_value . ' 00:00:00');
						break;
					case 'equals':
						$subquery = $wpdb->prepare("SELECT order_id 
									 FROM {$this->get_wpdb_prefix()}wc_order_stats AS order_stats 
									 WHERE order_stats.{$stats_field} IS NOT NULL 
									 AND DATE(order_stats.{$stats_field}) = %s", $date_value);
						break;
					case 'not_equals':
						$subquery = $wpdb->prepare("SELECT order_id 
									 FROM {$this->get_wpdb_prefix()}wc_order_stats AS order_stats 
									 WHERE order_stats.{$stats_field} IS NOT NULL 
									 AND DATE(order_stats.{$stats_field}) != %s", $date_value);
						break;
					default:
						$subquery = $wpdb->prepare("SELECT order_id 
									 FROM {$this->get_wpdb_prefix()}wc_order_stats AS order_stats 
									 WHERE order_stats.{$stats_field} IS NOT NULL 
									 AND DATE(order_stats.{$stats_field}) = %s", $date_value);
						break;
				}
			}
			
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} IN ({$subquery})";
		} else {
			// For legacy, use postmeta table
			if (is_array($value) && isset($value['from']) && isset($value['to'])) {
				$start_timestamp = strtotime($value['from']);
				$end_timestamp = strtotime($value['to']);
				
				$subquery = $wpdb->prepare("SELECT {$this->get_post_id_field()} 
							 FROM {$this->get_meta_table()} AS order_meta_date 
							 WHERE order_meta_date.meta_key = %s 
							 AND order_meta_date.meta_value > 0 
							 AND order_meta_date.meta_value BETWEEN %d AND %d", $meta_key, $start_timestamp, $end_timestamp);
			} else {
				$timestamp = strtotime($value);
				$operator = $this->get_numeric_compare_operator($condition);
				
				$subquery = $wpdb->prepare("SELECT {$this->get_post_id_field()} 
							 FROM {$this->get_meta_table()} AS order_meta_date 
							 WHERE order_meta_date.meta_key = %s 
							 AND order_meta_date.meta_value > 0 
							 AND order_meta_date.meta_value {$operator} %d", $meta_key, $timestamp);
			}
			
			$this->sql_fragments['base_conditions'][] = "orders.{$this->get_order_id_field()} IN ({$subquery})";
		}
	}

	/**
	 * Add exclude already exported filter
	 */
	private function add_exclude_already_exported_filter() {
		global $wpdb;
		
		if ($this->is_hpos_enabled) {
			// HPOS enabled - use wc_orders_meta table directly
			$exclude_condition = "NOT EXISTS (
				SELECT 1 FROM {$wpdb->prefix}wc_orders_meta AS check_exported 
				WHERE check_exported.order_id = orders.id 
				AND check_exported.meta_key = 'wf_order_exported_status' 
				AND check_exported.meta_value = '1'
			)";
		} else {
			// Legacy system - use postmeta table with NOT IN approach
			$exclude_condition = "orders.ID NOT IN (
				SELECT post_id FROM {$wpdb->postmeta} 
				WHERE meta_key = 'wf_order_exported_status' 
				AND meta_value = '1'
			)";
		}
		
		$this->sql_fragments['base_conditions'][] = $exclude_condition;
	
	}

	/**
	 * Build the final optimized SQL query from collected fragments
	 */
	private function build_final_sql_query($export_limit, $current_offset) {
		global $wpdb;
		
		// Get database structure info
		$order_table = $this->get_order_table();
		$order_id_field = $this->get_order_id_field();
		$post_type_field = $this->get_post_type_field();
		
		// Combine all LEFT JOINs (remove duplicates)
		$left_joins = array_unique($this->sql_fragments['left_joins']);
		$left_joins_sql = implode(' ', $left_joins);
		
		// Combine all WHERE conditions
		$base_conditions = implode(' AND ', $this->sql_fragments['base_conditions']);
		$where_conditions = implode(' AND ', $this->sql_fragments['where_conditions']);
		
		// Build final WHERE clause
		$final_where = "WHERE {$post_type_field} = 'shop_order' AND {$base_conditions}";
		if (!empty($where_conditions)) {
			$final_where .= " AND {$where_conditions}";
		}
		
		// Apply filter hook to the final query
		$sql = "SELECT DISTINCT orders.{$order_id_field} AS order_id 
				FROM {$order_table} AS orders
				{$left_joins_sql}
				{$final_where}
				ORDER BY orders.{$order_id_field} ASC 
				LIMIT {$export_limit} OFFSET {$current_offset}";
		
		return apply_filters("wt_ier_order_sql_query", $sql, $base_conditions, $where_conditions, $left_joins_sql);
	}

	/**
	 * Helper functions for database compatibility
	 */
	private function get_order_table() {
		global $wpdb;
		return $this->is_hpos_enabled ? $wpdb->prefix . 'wc_orders' : $wpdb->posts;
	}
	
	private function get_meta_table() {
		global $wpdb;
		return $this->is_hpos_enabled ? $wpdb->prefix . 'wc_orders_meta' : $wpdb->postmeta;
	}
	
	private function get_order_id_field() {
		return $this->is_hpos_enabled ? 'id' : 'ID';
	}
	
	private function get_post_id_field() {
		return $this->is_hpos_enabled ? 'order_id' : 'post_id';
	}
	
	private function get_post_type_field() {
		return $this->is_hpos_enabled ? 'orders.type' : 'orders.post_type';
	}
	
	private function get_status_field() {
		return $this->is_hpos_enabled ? 'status' : 'post_status';
	}
	
	private function get_wpdb_prefix() {
		global $wpdb;
		return $wpdb->prefix;
	}

	/**
	 * Utility functions
	 */
	private function validate_filter_value($element, $value, $condition) {
		// Add validation logic for different filter types
		if (empty($value) && '0' !== $value && 0 !== $value) {
			return false;
		}
		
		// Handle range values
		if (is_array($value) && isset($value['from']) && isset($value['to'])) {
			return !empty($value['from']) || !empty($value['to']);
		}
		
		return true;
	}
	
	private function get_text_compare_operator($condition) {
		switch ($condition) {
			case 'equals':
				return '=';
			case 'not_equals':
				return '!=';
			case 'contains':
				return 'LIKE';
			case 'not_contains':
				return 'NOT LIKE';
			case 'starts_with':
				return 'LIKE';
			case 'ends_with':
				return 'LIKE';
			default:
				return 'LIKE';
		}
	}
	
	private function get_numeric_compare_operator($condition) {
		switch ($condition) {
			case 'equals':
				return '=';
			case 'not_equals':
				return '!=';
			case 'greater_than':
				return '>';
			case 'less_than':
				return '<';
			case 'greater_than_equal':
				return '>=';
			case 'less_than_equal':
				return '<=';
			default:
				return '=';
		}
	}

	private function get_date_compare_operator($condition) {
		switch ($condition) {
			case 'greater_than':
				return '>';
			case 'less_than':
				return '<';
			case 'equals':
				return '=';
			case 'not_equals':
				return '!=';
			default:
				return '=';
		}
	}


	public function generate_row_data($order_id,$form_data) {
 
		$csv_columns = $this->prepare_header();

		$found_meta = array();

		foreach ($csv_columns as $key => $value) {
			if (substr((string) $key, 0, 5) == 'meta:') {
				$found_meta[substr((string) $key, 5)] = $value;
				unset($csv_columns[$key]);
			}
		}
		
		$row = array();
		// Get an instance of the WC_Order object.
		$order = wc_get_order($order_id);
		$line_items = $shipping_items = $fee_items = $tax_items = $coupon_items = $refund_items = array();

		// Get line items.
		foreach ($order->get_items() as $item_id => $item) {
			/* WC_Abstract_Legacy_Order::get_product_from_item() deprecated since version 4.4.0 */
			$product = ( version_compare( WC()->version, '4.4.0', '<' ) ) ? $order->get_product_from_item($item) : $item->get_product();
			if (!is_object($product)) {
				$product = new WC_Product(0);
			}

			$item_meta = self::get_order_line_item_meta($item_id);
			$prod_type = $product->get_type();
			$line_item = array(
				'name' => html_entity_decode(!empty($item['name']) ? $item['name'] : $product->get_title(), ENT_NOQUOTES, 'UTF-8'),
				'product_id' => ($prod_type == 'variable' || $prod_type == 'variation' || $prod_type == 'subscription_variation') ? $product->get_parent_id() : $product->get_id(),
				'sku' => $product->get_sku(),
				'quantity' => $item['qty'],
				'total' => wc_format_decimal($order->get_line_total($item), 2),
				'sub_total' => wc_format_decimal($order->get_line_subtotal($item), 2),
			);

			// Add line item tax.
			$line_tax_data = isset($item['line_tax_data']) ? $item['line_tax_data'] : array();
			$tax_data = maybe_unserialize($line_tax_data);
			$tax_detail = isset($tax_data['total']) ? wc_format_decimal(wc_round_tax_total(array_sum((array) $tax_data['total'])), 2) : '';
			if ($tax_detail != '0.00' && !empty($tax_detail)) {
				$line_item['tax'] = $tax_detail;
				$line_tax_ser = maybe_serialize($line_tax_data);
				$line_item['tax_data'] = $line_tax_ser;
			}

			foreach ($item_meta as $key => $value) {
				switch ($key) {
					case '_qty':
					case '_variation_id':
					case '_product_id':
					case '_line_total':
					case '_line_subtotal':
					case '_tax_class':
					case '_line_tax':
					case '_line_tax_data':
					case '_line_subtotal_tax':
						break;

					default:
						if (is_object($value))
							$value = $value->meta_value;
						if (is_array($value))
							$value = implode(',', $value);
							$line_item['meta:' . $key] = $value;
						break;
				}
			}

			$refunded = wc_format_decimal($order->get_total_refunded_for_item($item_id), 2);
			if ($refunded != '0.00') {
				$line_item['refunded'] = $refunded;
			}

			if ($prod_type === 'variable' || $prod_type === 'variation' || $prod_type === 'subscription_variation') {
				$line_item['_variation_id'] = $product->get_id();
			}
			$line_items[] = $line_item;
		}

		// WooCommerce gift cards plugin.
		if (class_exists('WC_GC_Gift_Card')) {

			$giftcards = $order->get_items('gift_card');
			if ($giftcards) {

				foreach ($giftcards as $id => $giftcard_order_item) {
					$giftcard = new WC_GC_Gift_Card($giftcard_order_item->get_giftcard_id());
					if (!$giftcard) {
						continue;
					}

					if ($giftcard->has_expired()) {
						$gift_card_expires = date_i18n(get_option('date_format'), $giftcard->get_expire_date());
					} else {
						$gift_card_expires = 0 === $giftcard->get_expire_date() ? '' : date_i18n(get_option('date_format', $giftcard->get_expire_date()));
					}
					$line_item = array(
						'giftcard-name' => $giftcard_order_item->get_code(),
						'giftcard-id' => $giftcard_order_item->get_giftcard_id(),
						'used_balance' => $giftcard_order_item->get_amount(),
						'available_balance' => $giftcard->get_balance(),
						'gift_card_expires' => $gift_card_expires
					);

					array_push($line_items, $line_item);
				}
			}
		}

		$line_items = apply_filters('wt_iew_export_order_line_items', $line_items, $order_id, $order);

		// Shipping items is just product x qty under shipping method.
		$line_items_shipping = $order->get_items('shipping');

		foreach ($line_items_shipping as $item_id => $item) {
			$item_meta = self::get_order_line_item_meta($item_id);
			foreach ($item_meta as $key => $value) {
				switch ($key) {
					case 'Items':
					case 'method_id':
					case 'taxes':
						if (is_object($value))
							$value = $value->meta_value;
						if (is_array($value))
							$value = implode(',', $value);
						$meta[$key] = $value;
						break;
				}
			}
			foreach (array('Items', 'method_id', 'taxes') as $value) {
				if (!isset($meta[$value])) {
					$meta[$value] = '';
				}
			}
			$shipping_items[] = trim(implode('|', array('items:' . $meta['Items'], 'method_id:' . $meta['method_id'], 'taxes:' . $meta['taxes'])));
		}

		// Get fee and total.
		$fee_total = 0;
		$fee_tax_total = 0;

		foreach ($order->get_fees() as $fee_id => $fee) {
			$fee_items[] = implode('|', array(
				'name:' . html_entity_decode($fee['name'], ENT_NOQUOTES, 'UTF-8'),
				'total:' . wc_format_decimal($fee['line_total'], 2),
				'tax:' . wc_format_decimal($fee['line_tax'], 2),
				'tax_data:' . maybe_serialize($fee['line_tax_data'])
			));
			$fee_total += (float) $fee['line_total'];
            $fee_tax_total += (float) $fee['line_tax'];
		}

		$order_taxes = $order->get_taxes();
		if (!empty($order_taxes)) {
			foreach ($order_taxes as $tax_id => $tax_item) {
				if (!empty($tax_item->get_shipping_tax_total())) {
					$total = $tax_item->get_tax_total() + $tax_item->get_shipping_tax_total();
				} else {
					$total = $tax_item->get_tax_total();
				}
				$tax_rate_percent = (defined('WC_VERSION') && version_compare(WC_VERSION, '3.5', '<=')) ? ( WC_Tax::get_rate_percent($tax_item->get_rate_id()) ) : $tax_item->get_rate_percent();
				$tax_items[] = implode('|', array(
					'rate_id:' . $tax_item->get_rate_id(),
					'code:' . $tax_item->get_rate_code(),
					'total:' . wc_format_decimal($tax_item->get_tax_total(), 2),
					'label:' . $tax_item->get_label(),
					'tax_rate_compound:' . $tax_item->get_compound(),
					'shipping_tax_amount:' . $tax_item->get_shipping_tax_total(),
					'rate_percent:' . $tax_rate_percent
				));
			}
		}

		// Add coupons.
		foreach ($order->get_items('coupon') as $_ => $coupon_item) {
			$discount_amount = ( version_compare( WC()->version, '4.4.0', '<' ) ) ? $coupon_item['discount_amount'] : $coupon_item->get_discount();
			$discount_amount = !empty($discount_amount) ? $discount_amount : 0;
			$coupon_code = ( version_compare( WC()->version, '4.4.0', '<' ) ) ? $coupon_item['name'] : $coupon_item->get_code();
			$coupon_items[] = implode('|', array(
				'code:' . $coupon_code,
				'amount:' . wc_format_decimal($discount_amount, 2),
			));
		}

		foreach ($order->get_refunds() as $refunded_items) {
				$refund_items[] = implode('|', array(
					'amount:' . $refunded_items->get_amount(),
					'reason:' . $refunded_items->get_reason(),
					'date:' . date('Y-m-d H:i:s', strtotime($refunded_items->get_date_created())),
				));
		}

		$order_data = array(
			'order_id' => $order->get_id(),
			'order_number' => $order->get_order_number(),
			'order_date' => date('Y-m-d H:i:s', strtotime(get_post($order->get_id())->post_date)),
			'paid_date' => $order->get_date_paid(),
			'status' => $order->get_status(),
			'shipping_total' => $order->get_total_shipping(),
			'shipping_tax_total' => wc_format_decimal($order->get_shipping_tax(), 2),
			'fee_total' => wc_format_decimal($fee_total, 2),
			'fee_tax_total' => wc_format_decimal($fee_tax_total, 2),
			'tax_total' => wc_format_decimal($order->get_total_tax(), 2),
			'cart_discount' => (defined('WC_VERSION') && version_compare(WC_VERSION, '2.3', '>=')) ? wc_format_decimal($order->get_total_discount(), 2) : wc_format_decimal($order->get_cart_discount(), 2),
			'order_discount' => (defined('WC_VERSION') && version_compare(WC_VERSION, '2.3', '>=')) ? wc_format_decimal($order->get_total_discount(), 2) : wc_format_decimal($order->get_order_discount(), 2),
			'discount_total' => wc_format_decimal($order->get_total_discount(), 2),
			'order_total' => wc_format_decimal($order->get_total(), 2),
			'order_subtotal' => wc_format_decimal($order->get_subtotal(), 2), // Get order subtotal
			'order_currency' => $order->get_currency(),
			'payment_method' => $order->get_payment_method(),
			'payment_method_title' => $order->get_payment_method_title(),
			'transaction_id' => $order->get_transaction_id(),
			'customer_ip_address' => $order->get_customer_ip_address(),
			'customer_user_agent' => $order->get_customer_user_agent(),
			'shipping_method' => $order->get_shipping_method(),
			'customer_id' => $order->get_user_id(),
			'customer_user' => $order->get_user_id(),
			'customer_email' => ($a = get_userdata($order->get_user_id())) ? $a->user_email : '',
			'billing_first_name' => $order->get_billing_first_name(),
			'billing_last_name' => $order->get_billing_last_name(),
			'billing_company' => $order->get_billing_company(),
			'billing_email' => $order->get_billing_email(),
			'billing_phone' => $order->get_billing_phone(),
			'billing_address_1' => $order->get_billing_address_1(),
			'billing_address_2' => $order->get_billing_address_2(),
			'billing_postcode' => $order->get_billing_postcode(),
			'billing_city' => $order->get_billing_city(),
			'billing_state' => $order->get_billing_state(),
			'billing_country' => $order->get_billing_country(),
			'shipping_first_name' => $order->get_shipping_first_name(),
			'shipping_last_name' => $order->get_shipping_last_name(),
			'shipping_company' => $order->get_shipping_company(),
			'shipping_phone' => (version_compare(WC_VERSION, '5.6', '<')) ? '' : $order->get_shipping_phone(),
			'shipping_address_1' => $order->get_shipping_address_1(),
			'shipping_address_2' => $order->get_shipping_address_2(),
			'shipping_postcode' => $order->get_shipping_postcode(),
			'shipping_city' => $order->get_shipping_city(),
			'shipping_state' => $order->get_shipping_state(),
			'shipping_country' => $order->get_shipping_country(),
			'customer_note' => $order->get_customer_note(),
			'wt_import_key' => $order->get_order_number(),
			'shipping_items' => self::format_data(implode(';', $shipping_items)),
			'fee_items' => implode('||', $fee_items),
			'tax_items' => implode(';', $tax_items),
			'coupon_items' => implode(';', $coupon_items),
			'refund_items' => implode(';', $refund_items),
			'order_notes' => implode('||', (defined('WC_VERSION') && version_compare(WC_VERSION, '3.2', '>=')) ? self::get_order_notes_new($order) : self::get_order_notes($order)),
			'download_permissions' => $order->is_download_permitted() ? $order->is_download_permitted() : 0,
		);

		$order_export_data = array();
		foreach ($csv_columns as $key => $value) {
			if (!$order_data || array_key_exists($key, $order_data)) {
				$order_export_data[$key] = $order_data[$key];
			}
		}

		if ($found_meta) {
			foreach ($found_meta as $key => $value) {
				if($this->is_hpos_enabled){
					$order_export_data[$value] = self::format_data(maybe_serialize($order->get_meta($key)));
				}else{
					$order_export_data[$value] = self::format_data(maybe_serialize(get_post_meta($order_data['order_id'], $key, TRUE)));
				}
			}
		}

		if ($this->exclude_line_items) {
			return apply_filters('hf_alter_csv_order_data', $order_export_data, array('max_line_items' => 0));
		}

		$li = 1;
		foreach ($line_items as $line_item) {
			foreach ($line_item as $name => $value) {
				$line_item[$name] = $name . ':' . $value;
			}
			$line_item = implode(apply_filters('wt_change_item_separator', '|'), $line_item);
			$order_export_data["line_item_{$li}"] = $line_item;
			$li++;
		}

		$max_line_items = $this->line_items_max_count;
		for ($i = 1; $i <= $max_line_items; $i++) {
			$order_export_data["line_item_{$i}"] = !empty($order_export_data["line_item_{$i}"]) ? self::format_data($order_export_data["line_item_{$i}"]) : '';
		}

		if ($this->export_to_separate_columns) {

			for ($i = 1; $i <= $max_line_items; $i++) {

				$order_export_data["line_item_{$i}_name"] = !empty($line_items[$i - 1]['name']) ? $line_items[$i - 1]['name'] : '';
				$order_export_data["line_item_{$i}_product_id"] = !empty($line_items[$i - 1]['product_id']) ? $line_items[$i - 1]['product_id'] : '';
				$order_export_data["line_item_{$i}_sku"] = !empty($line_items[$i - 1]['sku']) ? $line_items[$i - 1]['sku'] : '';
				$order_export_data["line_item_{$i}_quantity"] = !empty($line_items[$i - 1]['quantity']) ? $line_items[$i - 1]['quantity'] : '';
				$order_export_data["line_item_{$i}_total"] = !empty($line_items[$i - 1]['total']) ? $line_items[$i - 1]['total'] : '';
				$order_export_data["line_item_{$i}_subtotal"] = !empty($line_items[$i - 1]['sub_total']) ? $line_items[$i - 1]['sub_total'] : '';
			}
		}

		$order_data_filter_args = array('max_line_items' => $max_line_items);

		if ($this->export_to_separate_rows) {
			$order_export_data = $this->wt_line_item_separate_row_csv_data($order, $order_export_data, $order_data_filter_args);
		}

		return apply_filters('hf_alter_csv_order_data', $order_export_data, $order_data_filter_args,$form_data);
	}

	public static function get_order_line_item_meta($item_id) {
		global $wpdb;
		$filtered_meta = apply_filters('wt_order_export_select_line_item_meta', array());
		$filtered_meta = !empty($filtered_meta) ? implode("','", $filtered_meta) : '';
		$query = "SELECT meta_key,meta_value
            FROM {$wpdb->prefix}woocommerce_order_itemmeta WHERE order_item_id = '$item_id'";
		if (!empty($filtered_meta)) {
			$query .= " AND meta_key IN ('" . $filtered_meta . "')";
		}
		$meta_keys = $wpdb->get_results($query, OBJECT_K);
		return $meta_keys;
	}

	public static function get_order_notes($order) {
		$callback = array('WC_Comments', 'exclude_order_comments');
		$args = array(
			'post_id' => $order->get_id(),
			'approve' => 'approve',
			'type' => 'order_note'
		);
		remove_filter('comments_clauses', $callback);
		$notes = get_comments($args);
		add_filter('comments_clauses', $callback);
		$notes = array_reverse($notes);
		$order_notes = array();
		foreach ($notes as $note) {
			$date = $note->comment_date;
			$customer_note = 0;
			if (get_comment_meta($note->comment_ID, 'is_customer_note', '1')) {
				$customer_note = 1;
			}
			$order_notes[] = implode(apply_filters('wt_change_item_separator', '|'), array(
				'content:' . str_replace(array("\r", "\n"), ' ', $note->comment_content),
				'date:' . (!empty($date) ? $date : current_time('mysql')),
				'customer:' . $customer_note,
				'added_by:' . $note->added_by
			));
		}
		return $order_notes;
	}

	public static function get_order_notes_new($order) {
		$notes = wc_get_order_notes(array('order_id' => $order->get_id(), 'order_by' => 'date_created', 'order' => 'ASC'));
		$order_notes = array();
		foreach ($notes as $note) {
			$order_notes[] = implode(apply_filters('wt_change_item_separator', '|'), array(
				'content:' . str_replace(array("\r", "\n"), ' ', $note->content),
				'date:' . $note->date_created->date('Y-m-d H:i:s'),
				'customer:' . $note->customer_note,
				'added_by:' . $note->added_by
			));
		}
		return $order_notes;
	}

	/**
	 * Format the data if required
	 * @param  string $meta_value
	 * @param  string $meta name of meta key
	 * @return string
	 */
	public static function format_export_meta($meta_value, $meta) {
		switch ($meta) {
			case '_sale_price_dates_from' :
			case '_sale_price_dates_to' :
				return $meta_value ? date('Y-m-d', $meta_value) : '';
				break;
			case '_upsell_ids' :
			case '_crosssell_ids' :
				return implode('|', array_filter((array) json_decode($meta_value)));
				break;
			default :
				return $meta_value;
				break;
		}
	}
	
	/**
	 * Format data for writing to the output file.
	 * 
	 * @param type $data
	 * @return type
	 */
	public static function format_data($data) {
		if (!is_array($data))
		$data = (string) urldecode($data);
       
		$use_mb = function_exists('mb_detect_encoding');
		$enc = '';
		if ($use_mb) {
			$enc = mb_detect_encoding($data, 'UTF-8, ISO-8859-1', true);
		}
		$data = ( $enc == 'UTF-8' ) ? $data : utf8_encode($data);

		return $data;
	}

	/**
	 * Wrap a column in quotes for the CSV
	 * @param  string data to wrap
	 * @return string wrapped data
	 */
	public static function wrap_column($data) {
		return '"' . str_replace('"', '""', $data) . '"';
	}

	/**
	 * Get maximum number of line item for an order in the database.
	 * @return int Max line item number.
	 */
	public static function get_max_line_items() {
		global $wpdb;
		$query_line_items = "select COUNT(p.order_id) AS ttal from {$wpdb->prefix}woocommerce_order_items as p where order_item_type ='line_item' GROUP BY p.order_id ORDER BY ttal DESC LIMIT 1";
		$line_item_keys = $wpdb->get_col($query_line_items);
		$max_line_items = $line_item_keys[0];
		return $max_line_items;
	}
	
	
	/**
	 * Billing/Shipping Country/State pairs - convert in to SQL pairs.
	 * @param  array $pairs Country/State pairs.
	 * @return array Wrapped data.
	 */
	public static function country_state_pairs( $pairs ) {
		$valid_types = array( 'country', 'state' );
		$pair_types = array();
		$delimiters = array(
			'<>' => 'NOT IN',
			'=' => 'IN'
		);
		$sql_operators = array( 'NOT SET', 'IS SET' );

		foreach ( $pairs as $pair ) {
			$pair = trim( $pair );
			$op = '';
			$single_op = false;
			foreach ($delimiters as $delim => $op_seek) {
				$t = explode( $delim, $pair );
				$single_op = in_array($delim, $sql_operators);
				if (count($t) == 2) {
					$op = $op_seek;
					break;
				}
			}
			if (!$op) {
				continue;
			}
			if ($single_op) {
				$t[1] = '';
			}

			list( $filter_type, $filter_value ) = array_map("trim", $t);
			$empty = __('empty');
			if ($empty == $filter_value) {
				$filter_value = '';
			}

			$filter_type = strtolower($filter_type);

			if ($valid_types AND!in_array($filter_type, $valid_types)) {
				continue;
			}

			$filter_type = addslashes($filter_type);
			if (!isset($pair_types[$op])) {
				$pair_types[$op] = array();
			}
			if (!isset($pair_types[$op] [$filter_type])) {
				$pair_types[$op] [$filter_type] = array();
			}
			$pair_types[$op][$filter_type][] = addslashes($filter_value);
		}
		return $pair_types;
	}
	
	/**
	 * Array to single quoted array.
	 * @param  array $arr_values Values to be single quoted.
	 * @return string Wrapped data.
	 */
	public static function prepare_sql_inarray($arr_values) {
		global $wpdb;
		$values = array();
		foreach ($arr_values as $s) {
			$values[] = $wpdb->prepare('%s', $s);
		}

		return implode(",", $values);
	}

	
}
