<?php

/**
 * Order section of the plugin
 *
 * @link            
 *
 * @package  Wt_Import_Export_For_Woo 
 */
if (!defined('ABSPATH')) {
    exit;
}

use Automattic\WooCommerce\Utilities\OrderUtil;
class Wt_Import_Export_For_Woo_Order {

    public $module_id = '';
    public static $module_id_static = '';
    public $module_base = 'order';
    public $module_name = 'Order Import Export for WooCommerce';
    public $min_base_version= '1.0.0'; /* Minimum `Import export plugin` required to run this add on plugin */

    private $importer = null;
    private $exporter = null;
    private $all_meta_keys = array();
    private $exclude_hidden_meta_columns = array();
    private $found_meta = array();
    private $found_hidden_meta = array();
    private $selected_column_names = null;

    public function __construct()
    {      
        /**
        *   Checking the minimum required version of `Import export plugin` plugin available
        */
        if(!Wt_Import_Export_For_Woo_Common_Helper::check_base_version($this->module_base, $this->module_name, $this->min_base_version))
        {
            return;
        }
        if(!function_exists('is_plugin_active'))
        {
            include_once(ABSPATH.'wp-admin/includes/plugin.php');
        }
        if(!is_plugin_active('woocommerce/woocommerce.php'))
        {
            return;
        }


       
        $this->module_id = Wt_Import_Export_For_Woo::get_module_id($this->module_base);
        self::$module_id_static = $this->module_id;
                
        add_filter('wt_iew_exporter_post_types', array($this, 'wt_iew_exporter_post_types'), 10, 1);
        add_filter('wt_iew_importer_post_types', array($this, 'wt_iew_exporter_post_types'), 10, 1);

        add_filter('wt_iew_exporter_alter_filter_fields', array($this, 'exporter_alter_filter_fields'), 10, 3);
        
        add_filter('wt_iew_exporter_alter_mapping_fields', array($this, 'exporter_alter_mapping_fields'), 10, 3);        
        add_filter('wt_iew_importer_alter_mapping_fields', array($this, 'get_importer_post_columns'), 10, 3);  
        
        add_filter('wt_iew_exporter_alter_advanced_fields', array($this, 'exporter_alter_advanced_fields'), 10, 3);
        add_filter('wt_iew_importer_alter_advanced_fields', array($this, 'importer_alter_advanced_fields'), 10, 3);
        
        add_filter('wt_iew_exporter_alter_meta_mapping_fields', array($this, 'exporter_alter_meta_mapping_fields'), 10, 3);
        add_filter('wt_iew_importer_alter_meta_mapping_fields', array($this, 'importer_alter_meta_mapping_fields'), 10, 3);

        add_filter('wt_iew_exporter_alter_mapping_enabled_fields', array($this, 'exporter_alter_mapping_enabled_fields'), 10, 3);
        add_filter('wt_iew_importer_alter_mapping_enabled_fields', array($this, 'exporter_alter_mapping_enabled_fields'), 10, 3);

        add_filter('wt_iew_exporter_do_export', array($this, 'exporter_do_export'), 10, 7);
        add_filter('wt_iew_importer_do_import', array($this, 'importer_do_import'), 10, 8);

        add_filter('wt_iew_importer_steps', array($this, 'importer_steps'), 10, 2);

        add_filter( 'woocommerce_customer_search_customers', array($this, 'alter_customer_search_args_for_vendor_field'), 10, 1);

        add_action('admin_footer-edit.php', array($this, 'wt_add_order_bulk_actions'));
        add_action('load-edit.php', array($this, 'wt_process_order_bulk_actions'));
		
		add_filter('wt_add_woocommerce_debug_tools', array($this, 'wt_order_debug_tools'));
    }

    public function alter_customer_search_args_for_vendor_field($args) {

         $exclude = isset( $_GET['exclude'] ) ? wp_unslash( $_GET['exclude'] ) : '';

         if ('__VENDOR_FIELD__' !== $exclude) {
             return $args;
         }
        /**
         * vendor - WC Vendors Marketplace.
        * seller - Dokan plugin.
        */
        $roles = apply_filters('wt_iew_allowed_roles_for_vendor_field',  array( 'author','vendor','seller' ));
        $search_columns = apply_filters('wt_iew_search_columns_for_vendor_field',  array( 'user_login', 'user_email', 'display_name' ));
        $args['role__in']       = $roles;
        $args['search_columns'] = $search_columns;
        $args['orderby']        = 'display_name';
        $args['order']          = 'ASC';
        return $args;
    }

	/**
	 * Add more tools options under WC status > tools
	 * 
	 * @param array $tools WC Tools items
	 */
	public function wt_order_debug_tools($wc_tools) {

		$wc_tools['wt_delete_trashed_orders'] = array(
			'name' => __('Remove all trashed orders', 'wt-import-export-for-woo'),
			'button' => __('Delete all trashed orders', 'wt-import-export-for-woo'),
			'desc' => __('This tool will delete all trashed orders.', 'wt-import-export-for-woo'),
			'callback' => array($this, 'wt_remove_all_trashed_orders')
		);

		$wc_tools['wt_delete_all_orders'] = array(
			'name' => __('Remove all orders', 'wt-import-export-for-woo'),
			'button' => __('Delete all orders', 'wt-import-export-for-woo'),
			'desc' => __('This tool will delete all orders allowing you to start fresh.', 'wt-import-export-for-woo'),
			'callback' => array($this, 'wt_remove_all_orders')
		);

		return $wc_tools;
	}

	public function wt_remove_all_trashed_orders() {
		global $wpdb;
		$result = absint($wpdb->delete($wpdb->posts, array('post_type' => 'shop_order', 'post_status' => 'trash')));

		$wpdb->query("DELETE pm
			FROM {$wpdb->postmeta} pm
			LEFT JOIN {$wpdb->posts} wp ON wp.ID = pm.post_id
			WHERE wp.ID IS NULL");

		// Delete order items with no post
		$wpdb->query("DELETE oi
                        FROM {$wpdb->prefix}woocommerce_order_items oi
                        LEFT JOIN {$wpdb->posts} wp ON wp.ID = oi.order_id
                        WHERE wp.ID IS NULL");

		// Delete order item meta with no post
		$wpdb->query("DELETE om
                        FROM {$wpdb->prefix}woocommerce_order_itemmeta om
                        LEFT JOIN {$wpdb->prefix}woocommerce_order_items oi ON oi.order_item_id = om.order_item_id
                        WHERE oi.order_item_id IS NULL");
		echo '<div class="updated"><p>' . sprintf(__('%d Orders Deleted', 'wt-import-export-for-woo'), ( $result)) . '</p></div>';
	}

	public function wt_remove_all_orders() {
		global $wpdb;

		$result = absint($wpdb->delete($wpdb->posts, array('post_type' => 'shop_order')));

		$wpdb->query("DELETE pm
			FROM {$wpdb->postmeta} pm
			LEFT JOIN {$wpdb->posts} wp ON wp.ID = pm.post_id
			WHERE wp.ID IS NULL");

		// Delete order items with no post
		$wpdb->query("DELETE oi
                        FROM {$wpdb->prefix}woocommerce_order_items oi
                        LEFT JOIN {$wpdb->posts} wp ON wp.ID = oi.order_id
                        WHERE wp.ID IS NULL");

		// Delete order item meta with no post
		$wpdb->query("DELETE om
                        FROM {$wpdb->prefix}woocommerce_order_itemmeta om
                        LEFT JOIN {$wpdb->prefix}woocommerce_order_items oi ON oi.order_item_id = om.order_item_id
                        WHERE oi.order_item_id IS NULL");
		echo '<div class="updated"><p>' . sprintf(__('%d Orders Deleted', 'wt-import-export-for-woo'), $result) . '</p></div>';
	}

	public function wt_add_order_bulk_actions() {
        global $post_type, $post_status;

        if ($post_type == 'shop_order' && $post_status != 'trash') {
            ?>
            <script type="text/javascript">
                jQuery(document).ready(function ($) {
                    var $downloadOrders = $('<option>').val('wt_iew_download_orders').text('<?php _e('Export as CSV', 'wt-import-export-for-woo') ?>');
                    $('select[name^="action"]').append($downloadOrders);
                });
            </script>
            <?php
        }
    }
    
        /**
     * Order page bulk export action
     * 
     */
    public function wt_process_order_bulk_actions() {
        global $typenow;
        if ($typenow == 'shop_order') {
            // get the action list
            $wp_list_table = _get_list_table('WP_Posts_List_Table');
            $action = $wp_list_table->current_action();
            if (!in_array($action, array('wt_iew_download_orders'))) {
                return;
            }
            // security check
            check_admin_referer('bulk-posts');

            if (isset($_REQUEST['post'])) {
                $order_ids = array_map('absint', $_REQUEST['post']);
            }
            if (empty($order_ids)) {
                return;
            }
            // give an unlimited timeout if possible
            @set_time_limit(0);

            if ($action == 'wt_iew_download_orders') {
                
                
                include_once( 'export/class-wt-orderimpexpcsv-exporter.php' );
                Wt_Import_Export_For_Woo_Order_Bulk_Export::do_export('shop_order', $order_ids);
            }
        }
    }
    
    /**
    *   Altering advanced step description
    */
    public function importer_steps($steps, $base)
    {
        if($this->module_base==$base)
        {
            $steps['advanced']['description']=__('Use options from below to decide updates to existing orders, batch import count or schedule an import. You can also save the template file for future imports.', 'wt-import-export-for-woo');
        }
        return $steps;
    }
    
    public function importer_do_import($import_data, $base, $step, $form_data, $selected_template_data, $method_import, $batch_offset, $is_last_batch) {        
        if ($this->module_base != $base) {
            return $import_data;
        }             
        
        if(0 == $batch_offset){                        
            $memory = size_format(wt_let_to_num(ini_get('memory_limit')));
            $wp_memory = size_format(wt_let_to_num(WP_MEMORY_LIMIT));                      
            Wt_Import_Export_For_Woo_Logwriter::write_log($this->module_base, 'import', '---[ New import started at '.date('Y-m-d H:i:s').' ] PHP Memory: ' . $memory . ', WP Memory: ' . $wp_memory);
        }
                
        include plugin_dir_path(__FILE__) . 'import/import.php';
        $import = new Wt_Import_Export_For_Woo_Order_Import($this);
        
        $response = $import->prepare_data_to_import($import_data,$form_data,$batch_offset,$is_last_batch);
        
        if($is_last_batch){
            Wt_Import_Export_For_Woo_Logwriter::write_log($this->module_base, 'import', '---[ Import ended at '.date('Y-m-d H:i:s').']---');
        }
        
        return $response;
    }

    public function exporter_do_export($export_data, $base, $step, $form_data, $selected_template_data, $method_export, $batch_offset) {
        if ($this->module_base != $base) {
            return $export_data;
        }
        
        switch ($method_export) {
            case 'quick':
                $this->set_export_columns_for_quick_export($form_data);
                break;

            case 'template':
            case 'new':
                $this->set_selected_column_names($form_data);
                break;
            
            default:
                break;
        }

        include plugin_dir_path(__FILE__) . 'export/export.php';
        $export = new Wt_Import_Export_For_Woo_Order_Export($this);
                      
        $data_row = $export->prepare_data_to_export($form_data, $batch_offset);
        
        $header_row = $export->prepare_header(); 

        $export_data = array(
            'head_data' => $header_row,
            'body_data' => isset($data_row['data']) ? $data_row['data'] : array(),
            'total' => isset($data_row['total']) ? $data_row['total'] : array(),
        ); 
        if(isset($data_row['no_post'])){
            $export_data['no_post'] = $data_row['no_post'];
        }
        return $export_data;
    }

    /**
     * Adding current post type to export list
     *
     */
    public function wt_iew_exporter_post_types($arr) {
        $arr['order'] = __('Order', 'wt-import-export-for-woo');
        return $arr;
    }

    /*
     * Setting default export columns for quick export
     */

    public function set_export_columns_for_quick_export($form_data) {

        $post_columns = self::get_order_post_columns();

        $this->selected_column_names = array_combine(array_keys($post_columns), array_keys($post_columns));

        if (isset($form_data['method_export_form_data']['mapping_enabled_fields']) && !empty($form_data['method_export_form_data']['mapping_enabled_fields'])) {
            foreach ($form_data['method_export_form_data']['mapping_enabled_fields'] as $value) {
                $additional_quick_export_fields[$value] = array('fields' => array());
            }

            $export_additional_columns = $this->exporter_alter_meta_mapping_fields($additional_quick_export_fields, $this->module_base, array());
            foreach ($export_additional_columns as $value) {
                $this->selected_column_names = array_merge($this->selected_column_names, $value['fields']);
            }
        }
    }

    public static function get_order_statuses() {
        $statuses = wc_get_order_statuses();
        return apply_filters('wt_iew_allowed_order_statuses',  $statuses);
    }

    public static function get_order_sort_columns() {
        $sort_columns = array('post_parent', 'ID', 'post_author', 'post_date', 'post_title', 'post_name', 'post_modified', 'menu_order', 'post_modified_gmt', 'rand', 'comment_count');
        return apply_filters('wt_iew_allowed_order_sort_columns', array_combine($sort_columns, $sort_columns));
    }

    public static function get_order_post_columns() {
        return include plugin_dir_path(__FILE__) . 'data/data-order-post-columns.php';
    }
    
    public function get_importer_post_columns($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }
        $colunm = include plugin_dir_path(__FILE__) . 'data/data/data-wf-reserved-fields-pair.php';
//        $colunm = array_map(function($vl){ return array('title'=>$vl, 'description'=>$vl); }, $arr); 
        return $colunm;
    }

    public function exporter_alter_mapping_enabled_fields($mapping_enabled_fields, $base, $form_data_mapping_enabled_fields) {
        if ($base == $this->module_base) {
            $mapping_enabled_fields = array();
            $mapping_enabled_fields['meta'] = array(__('Custom meta', 'wt-import-export-for-woo'), 1);
            $mapping_enabled_fields['hidden_meta'] = array(__('Hidden meta', 'wt-import-export-for-woo'), 0);
        }
        return $mapping_enabled_fields;
    }

    public function exporter_alter_meta_mapping_fields($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }

        foreach ($fields as $key => $value) {
            switch ($key) {
                
                case 'meta':
                    $meta_attributes = array();
                    $found_meta = $this->wt_get_found_meta();
                    foreach ($found_meta as $meta) {
                        $fields[$key]['fields']['meta:' . $meta] = 'meta:' . $meta;
                    }

                    break;               

                case 'hidden_meta':
                    $found_hidden_meta = $this->wt_get_found_hidden_meta();
                    foreach ($found_hidden_meta as $hidden_meta) {
                        $fields[$key]['fields']['meta:' . $hidden_meta] = 'meta:' . $hidden_meta;
                    }
                    break;
                default:
                    break;
            }
        }

        return $fields;
    }
    
    
    public function importer_alter_meta_mapping_fields($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }
        $fields=$this->exporter_alter_meta_mapping_fields($fields, $base, $step_page_form_data);
        $out=array();
        foreach ($fields as $key => $value) 
        {
            $value['fields']=array_map(function($vl){ return array('title'=>$vl, 'description'=>$vl); }, $value['fields']);
            $out[$key]=$value;
        }
        return $out;
    }
    
    public function wt_get_found_meta() {   

        if (!empty($this->found_meta)) {
            return $this->found_meta;
        }

        // Loop products and load meta data
        $found_meta = array();
        // Some of the values may not be usable (e.g. arrays of arrays) but the worse
        // that can happen is we get an empty column.

        $all_meta_keys = $this->wt_get_all_meta_keys();

        $csv_columns = self::get_order_post_columns();

        $exclude_hidden_meta_columns = $this->wt_get_exclude_hidden_meta_columns();

        foreach ($all_meta_keys as $meta) {

            if (!$meta || (substr((string) $meta, 0, 1) == '_') || in_array($meta, $exclude_hidden_meta_columns) || in_array($meta, array_keys($csv_columns)) || in_array('meta:' . $meta, array_keys($csv_columns)))
                continue;

            $found_meta[] = $meta;
        }
        
        $found_meta = array_diff($found_meta, array_keys($csv_columns));
        $this->found_meta = $found_meta;
        return $this->found_meta;
    }

    public function wt_get_found_hidden_meta() {

        if (!empty($this->found_hidden_meta)) {
            return $this->found_hidden_meta;
        }

        // Loop products and load meta data
        $found_hidden_meta = array();
        // Some of the values may not be usable (e.g. arrays of arrays) but the worse
        // that can happen is we get an empty column.

        $all_meta_keys = $this->wt_get_all_meta_keys();
        $csv_columns = self::get_order_post_columns();
        $exclude_hidden_meta_columns = $this->wt_get_exclude_hidden_meta_columns();
        foreach ($all_meta_keys as $meta) {

            if (!$meta || (substr((string) $meta, 0, 1) != '_') || ((substr((string) $meta, 0, 1) == '_') && in_array(substr((string) $meta,1), array_keys($csv_columns)) ) || in_array($meta, $exclude_hidden_meta_columns) || in_array($meta, array_keys($csv_columns)) || in_array('meta:' . $meta, array_keys($csv_columns)))
                continue;

            $found_hidden_meta[] = $meta;
        }

        $found_hidden_meta = array_diff($found_hidden_meta, array_keys($csv_columns));

        $this->found_hidden_meta = $found_hidden_meta;
        return $this->found_hidden_meta;
    }

    public function wt_get_exclude_hidden_meta_columns() {

        if (!empty($this->exclude_hidden_meta_columns)) {
            return $this->exclude_hidden_meta_columns;
        }

        $exclude_hidden_meta_columns = include( plugin_dir_path(__FILE__) . 'data/data-wf-exclude-hidden-meta-columns.php' );

        $this->exclude_hidden_meta_columns = $exclude_hidden_meta_columns;
        return $this->exclude_hidden_meta_columns;
    }

    public function wt_get_all_meta_keys() {

        if (!empty($this->all_meta_keys)) {
            return $this->all_meta_keys;
        }

        $all_meta_keys = self::get_all_metakeys();

        $this->all_meta_keys = $all_meta_keys;
        return $this->all_meta_keys;
    }

    /**
     * Get a list of all the meta keys for a post type. This includes all public, private,
     * used, no-longer used etc. They will be sorted once fetched.
     */
    public static function get_all_metakeys() {
        global $wpdb;
        $limit =  apply_filters( 'wt_ier_limit_order_meta', 2000);
        if(self::wt_get_order_table_name()){
            $meta = $wpdb->get_col( $wpdb->prepare( "SELECT DISTINCT pm.meta_key
                                    FROM {$wpdb->prefix}wc_orders_meta AS pm
                                    LEFT JOIN {$wpdb->prefix}wc_orders AS p ON p.id = pm.order_id
                                    WHERE p.type = 'shop_order'
                                    ORDER BY pm.meta_key LIMIT %d", $limit)         
                                    );
        }else{
            $meta = $wpdb->get_col( $wpdb->prepare( "SELECT DISTINCT pm.meta_key
                                    FROM {$wpdb->postmeta} AS pm
                                    LEFT JOIN {$wpdb->posts} AS p ON p.ID = pm.post_id
                                    WHERE p.post_type = 'shop_order'
                                    ORDER BY pm.meta_key LIMIT %d", $limit)         
                                    );
        }
        //sort($meta);
        return $meta;
    }

    public function set_selected_column_names($full_form_data) {
        if (is_null($this->selected_column_names)) {
            if (isset($full_form_data['mapping_form_data']['mapping_selected_fields']) && !empty($full_form_data['mapping_form_data']['mapping_selected_fields'])) {
                $this->selected_column_names = $full_form_data['mapping_form_data']['mapping_selected_fields'];
            }
            if (isset($full_form_data['meta_step_form_data']['mapping_selected_fields']) && !empty($full_form_data['meta_step_form_data']['mapping_selected_fields'])) {
                $export_additional_columns = $full_form_data['meta_step_form_data']['mapping_selected_fields'];
                foreach ($export_additional_columns as $value) {
                    $this->selected_column_names = array_merge($this->selected_column_names, $value);
                }
            }
        }

        return $full_form_data;
    }

    public function get_selected_column_names() {

        return $this->selected_column_names;
    }

    public function exporter_alter_mapping_fields($fields, $base, $mapping_form_data) {
        if ($base == $this->module_base) {
            $fields = self::get_order_post_columns();
        }
        return $fields;
    }

    public function exporter_alter_advanced_fields($fields, $base, $advanced_form_data) {
        if ($this->module_base != $base) {
            return $fields;
        }

        unset($fields['export_shortcode_tohtml']);
        
        $out = array();
        $out['exclude_already_exported'] = array(
            'label' => __("Exclude already exported", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                'Yes' => __('Yes', 'wt-import-export-for-woo'),
                'No' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => 'No',
            'field_name' => 'exclude_already_exported',
            'help_text' => __("Option 'Yes' excludes the previously exported orders.", 'wt-import-export-for-woo'),
        );
        $out['exclude_line_items'] = array(
            'label' => __("Exclude line items", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                'Yes' => __('Yes', 'wt-import-export-for-woo'),
                'No' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => 'No',
            'field_name' => 'exclude_line_items',
            'help_text' => __("Option 'Yes' excludes the line items", 'wt-import-export-for-woo'),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'wt_iew_exclude_line_items',
            )
        );
        $out['export_to_separate'] = array(
            'label' => __("Export line items in", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                'default' => __('Migration mode', 'wt-import-export-for-woo'),
                'column' => __('Separate columns', 'wt-import-export-for-woo'),
                'row' => __('Separate rows', 'wt-import-export-for-woo')                
            ),
            'value' => 'default',
            'field_name' => 'export_to_separate',
            //'help_text' => __("Option 'Yes' exports the line items within the order into separate columns in the exported file.", 'wt-import-export-for-woo'),
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('This option will export each line item details into a single column. This option is mainly used for the order migration purpose.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_export_to_separate', 'value'=>'default')
                    )
                ),
                array(
                    'help_text'=> __('This option will export each line item details into a separate column.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_export_to_separate', 'value'=>'column')
                    )
                ),
                array(
                    'help_text'=> __('This option will export each line item details into a separate row.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_export_to_separate', 'value'=>'row')
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'wt_iew_exclude_line_items',
                'val'=>'No',
                'depth'=>1, /* indicates the left margin of fields */                
            )
        );

        foreach ($fields as $fieldk => $fieldv) {
            $out[$fieldk] = $fieldv;
        }
        return $out;
    }
    
    public function importer_alter_advanced_fields($fields, $base, $advanced_form_data) {

        if ($this->module_base != $base) {
            return $fields;
        }
        $out = array();
        
        $out['skip_new'] = array(
            'label' => __("Update Only", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'skip_new',
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('The store is updated with the data from the input file only for matching/existing records from the file. If the post ID of the order being imported exists already(for any of the other post types like coupon, product, user, pages, media etc) skip the order from being inserted into the store.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_skip_new', 'value'=>1)
                    )
                ),
                array(
                    'help_text'=> __('The entire data from the input file is processed for an update or insert as the case maybe.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_skip_new', 'value'=>0)
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'wt_iew_skip_new',
            )
        ); 

        $out['found_action_merge'] = array(
            'label' => __("If order exists in the store", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
//                'import' => __('Import as new item'),
                'skip' => __('Skip', 'wt-import-export-for-woo'),
                'update' => __('Update', 'wt-import-export-for-woo'),                
            ),
            'value' => 'skip',
            'field_name' => 'found_action',
            'help_text' => __('Orders are matched by their order IDs.', 'wt-import-export-for-woo'),
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('Retains the order in the store as is and skips the matching order from the input file.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_found_action', 'value'=>'skip')
                    )
                ),
                array(
                    'help_text'=> __('Update order as per data from the input file', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_found_action', 'value'=>'update')
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'wt_iew_found_action'
            )
        );       
        
        $out['merge_empty_cells'] = array(
            'label' => __("Update even if empty values", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'merge_empty_cells',
            'help_text' => __('Updates the order data respectively even if some of the columns in the input file contains empty value. <p>Note: This is not applicable for line_items, tax_items, fee _items and shipping_items</p>', 'wt-import-export-for-woo'),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'wt_iew_found_action',
                'val'=>'update',
            )
        );
        
        $out['conflict_with_existing_post'] = array(
            'label' => __("If conflict with an existing Post ID", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(                
                'skip' => __('Skip item', 'wt-import-export-for-woo'),
                'import' => __('Import as new item', 'wt-import-export-for-woo'),
            ),
            'value' => 'skip',
            'field_name' => 'id_conflict',
            'help_text' => __('All the items within woocommerce/wordpress are treated as posts and assigned a unique ID as and when they are created in the store. The post ID uniquely identifies an item irrespective of the post type be it order/product/pages/attachments/revisions etc.', 'wt-import-export-for-woo'),
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('If the post ID of the order being imported exists already(for any of the other post types like coupon, product, user, pages, media etc) skip the order from being inserted into the store.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_id_conflict', 'value'=>'skip')
                    )
                ),
                array(
                    'help_text'=> __('Insert the order into the store with a new order ID(next available post ID) different from the value in the input file.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_id_conflict', 'value'=>'import')
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'wt_iew_skip_new',
                'val'=>'0',
                'depth'=>0,
            )
        );

        
//        $out['merge'] = array(
//            'label' => __("Update order if exists"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                '1' => __('Yes'),
//                '0' => __('No')
//            ),
//            'value' => '0',
//            'field_name' => 'merge',
//            'help_text' => __('Existing orders are identified by their IDs.'),
//            'form_toggler'=>array(
//                'type'=>'parent',
//                'target'=>'wt_iew_merge'
//            )
//        );  
//        
//        $out['skip_new'] = array(
//            'label' => __("Skip New Order"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                '1' => __('Yes'),
//                '0' => __('No')
//            ),
//            'value' => '0',
//            'field_name' => 'skip_new',
//            'help_text' => __('While updating existing order, enable this to skip order which are not already present in the store.'),
//            'form_toggler'=>array(
//                'type'=>'child',
//                'id'=>'wt_iew_merge',
//                'val'=>'1',
//            )
//        );
//        
//        $out['found_action_merge'] = array(
//            'label' => __("Skip or import if Order not found"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                'import' => __('Import as new item'),
//                'skip' => __('Skip item'),
//                
//            ),
//            'value' => 'import',
//            'field_name' => 'found_action',
//            'help_text' => __('Skip or import if Order not found.'),
//            'form_toggler'=>array(
//                'type'=>'',
//                'id'=>'wt_iew_merge',
//                'val'=>'1',
//                'target'=>'wt_iew_use_same_id'
//            )
//        );
//        
//        $out['found_action_import'] = array(
//            'label' => __("Skip or import"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                'import' => __('Import as new item'),
//                'skip' => __('Skip item'),
//                
//            ),
//            'value' => 'import',
//            'field_name' => 'found_action',
//            'help_text' => __('Skip or import if found a non Order post type in given ID.'),
//            'form_toggler'=>array(
//                'type'=>'',
//                'id'=>'wt_iew_merge',
//                'val'=>'0',
//                'target'=>'wt_iew_use_same_id'
//            )
//        );
//        
//        $out['use_same_id'] = array(
//            'label' => __("Use the same ID for Order on import"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                '1' => __('Yes'),
//                '0' => __('No')
//            ),
//            'value' => '0',
//            'field_name' => 'use_same_id',
//            'help_text' => __('Use the same ID for Order on import.'),
//            'form_toggler'=>array(
//                'type'=>'',
//                'id'=>'wt_iew_use_same_id',
//                'val'=>'import',
//            )
//        );
//        
//        $out['merge_empty_cells'] = array(
//            'label' => __("Merge empty cells"),
//            'type' => 'radio',
//            'radio_fields' => array(
//                '1' => __('Yes'),
//                '0' => __('No')
//            ),
//            'value' => '0',
//            'field_name' => 'merge_empty_cells',
//            'help_text' => __('Check to merge the empty cells in CSV, otherwise empty cells will be ignored.'),
//            'form_toggler'=>array(
//                'type'=>'child',
//                'id'=>'wt_iew_merge',
//                'val'=>'1',
//            )
//        );
//        

        
      /* Temparay commented */
        $out['status_mail'] = array(
            'label' => __("Email customer on order status change", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'status_mail',
            'help_text' => __('Select ‘Yes’ if an email is to be sent to the customer of the corresponding order when the status is changed during import.', 'wt-import-export-for-woo'),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'wt_iew_merge',
                'val'=>'1',
            )
        );
        
        $out['create_user'] = array(
            'label' => __("Create user", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'create_user',
            'help_text' => __('Select ‘Yes’ if you want the application to create the customer associated with the order being imported if the customer does not exist in the store. Only the username(extracted from the email id), email id and address( shipping address) is added to the profile.', 'wt-import-export-for-woo'),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'notify_customer',
            )
        );
        
        $out['notify_customer'] = array(
            'label' => __("Notify the customer", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'notify_customer',
            'help_text' => __('Notify the customer by email when created successfully. Customer will have to use the forgot password link to access the account.', 'wt-import-export-for-woo'),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'notify_customer',
                'val'=>'1',
            )
        );
 
        $out['ord_link_using_sku'] = array(
            'label' => __("Link products using SKU instead of Product ID", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'ord_link_using_sku',
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('Link the products associated with the imported orders by their SKU.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_ord_link_using_sku', 'value'=>1)
                    )
                ),
                array(
                    'help_text'=> sprintf(__('Link the products associated with the imported orders by their Product ID. In case of a conflict with %sIDs of other existing post types%s the link cannot be established.', 'wt-import-export-for-woo'), '<b>', '</b>'),
                    'condition'=>array(
                        array('field'=>'wt_iew_ord_link_using_sku', 'value'=>0)
                    )
                )
            ),
        );
        
        
        $out['delete_existing'] = array(
            'label' => __( 'Delete non-matching orders from store', 'wt-import-export-for-woo' ),
            'type' => 'checkbox',
            'checkbox_fields' => array(
                1 => __( 'Enable', 'wt-import-export-for-woo' ),
            ),
            'value' => 0,
            'field_name' => 'delete_existing',
            'tip_description' => __( 'For e.g: if you have an order #123 in your store and your import file has orders #234, #345; the order #123 is deleted from the store prior to importing orders #234, #345.', 'wt-import-export-for-woo' ),
        );
		
	$out['update_stock_details'] = array(
            'label' => __( 'Update stock details', 'wt-import-export-for-woo' ),
            'type' => 'checkbox',
            'checkbox_fields' => array( 1 => __( 'Enable' ) ),
            'value' => 0,
            'field_name' => 'update_stock_details',
            'help_text' => __( 'Updates the sale count and stock quantity of a product associated with the order.', 'wt-import-export-for-woo' ),
            'tip_description' => __( 'Note: Ensure the manage stock option is enabled. This feature is not meant to work for the refunded, cancelled or failed order statuses.', 'wt-import-export-for-woo' ),
        );                                
        
        
        foreach ($fields as $fieldk => $fieldv) {
            $out[$fieldk] = $fieldv;
        }
        return $out;
    }
    
    /**
     *  Customize the items in filter export page
     * 
     *  1.3.1 - updated filter fields
     */
     public function exporter_alter_filter_fields($fields, $base, $filter_form_data) {
        if ( $this->module_base !== $base ) {
            return $fields;
        }

         // Initialize filter structure
         $fields = $this->initialize_filter_structure($fields);

        // Get saved template data and prepare data
        $saved_template_data = $this->get_saved_template_data($filter_form_data);
        $all_filter_options = $this->get_all_filter_options();
        $selected_elements = $this->extract_selected_elements($saved_template_data);     
        
        if ( isset( $filter_form_data['old_template_data'] ) && is_array($filter_form_data['old_template_data']) ) {
            $fields = $this->create_old_template_rows($fields, $base, $all_filter_options, $selected_elements, $filter_form_data);
        } else {
            // Create filter rows based on data type
            if ( ! empty( $saved_template_data ) ) {
                $fields = $this->create_saved_template_rows($fields, $saved_template_data, $all_filter_options, $selected_elements);
                // Apply saved template values
                $fields = $this->alter_filter_fields_with_saved_template($fields, $saved_template_data);
            } else {
                $fields = $this->create_default_rows($fields, $all_filter_options);
            }
        }
        
        // Add extra fields if needed
        $fields = $this->add_extra_filter_fields($fields, $base, $filter_form_data);
        
        // Add available filter options for dropdown
        $fields['multi_field_row']['available_filter_options'] = $all_filter_options;
		
        return $fields;
    }

    /**
     * Initialize the basic filter structure
     * 
     * @since 1.3.1
     */
    private function initialize_filter_structure($fields) {
        if ( isset( $fields['limit'] ) ) {
             unset( $fields['limit'] );
        }

        if ( isset( $fields['offset'] ) ) {
             unset( $fields['offset'] );
        }

        $fields['multi_field_row']['type'] = 'multi_field_row';
        $fields['multi_field_row']['fields'] = array();
        $fields['multi_field_row']['fields'][] = $this->get_filter_column_headers();
            return $fields;
        }
    
    /**
     * Get saved template data from filter form data
     * 
     * @since 1.3.1
     */
    private function get_saved_template_data($filter_form_data) {
        return ( isset( $filter_form_data['saved_template_data'] ) && is_array( $filter_form_data['saved_template_data'] ) ) ? $filter_form_data['saved_template_data'] : array();
    }
    
    /**
     * Get all available filter options
     * 
     * @since 1.3.1
     */
    private function get_all_filter_options() {
        $default_filter_options = $this->get_default_filter_options();
        $extra_filter_options = $this->get_extra_filter_options();
        
        // Ensure both are arrays before merging
        $default_filter_options = is_array( $default_filter_options ) ? $default_filter_options : array();
        $extra_filter_options = is_array( $extra_filter_options ) ? $extra_filter_options : array();
        
        return array_merge( $default_filter_options, $extra_filter_options );
    }
    
    /**
     * Extract selected elements from saved template data
     * 
     * @since 1.3.1
     */
    private function extract_selected_elements($saved_template_data) {
        $selected_elements = array();
        if ( ! empty( $saved_template_data ) ) {
            foreach ( $saved_template_data as $template_data ) {
                if ( ! empty( $template_data['element'] ) ) {
                    $selected_elements[] = $template_data['element'];
                }
            }
        }
        return $selected_elements;
    }
    
    /**
     * Create saved template rows
     * 
     * @since 1.3.1
     */
    private function create_saved_template_rows($fields, $saved_template_data, $all_filter_options, $selected_elements) {
        $valid_template_count = $this->count_valid_template_entries( $saved_template_data );
        $index = 0;
        $current_row = 0;
        
        foreach ( $saved_template_data as $template_key => $template_data ) {
            $element = isset($template_data['element']) ? $template_data['element'] : '';
            
            if ( empty( $element ) ) {
                continue;
            }
            
            $row_fields = $this->create_filter_row_structure($index, $element, $all_filter_options, $current_row === $valid_template_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $element, $all_filter_options, $selected_elements);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
            $current_row++;
        }

        return $fields;
    }

    /**
     * Create old template rows
     * 
     * @since 1.3.1
     */
    private function create_old_template_rows($fields, $base, $all_filter_options, $selected_elements, $filter_form_data) {
        if ( $this->module_base != $base ) {
            return $fields;
        }

        // Get the old template data from filter_form_data
        $old_template_data = isset($filter_form_data['old_template_data']) ? $filter_form_data['old_template_data'] : array();
        
        if ( empty($old_template_data) || ! is_array( $old_template_data ) ) {
            return $fields;
        }

        // Convert old template data to new structure
        $converted_template_data = $this->convert_old_template_to_new_structure($old_template_data);

        if ( empty($converted_template_data) ) {
            $fields = $this->create_default_rows($fields, $all_filter_options);
            return $fields;
        }
        
        $valid_template_count = count($converted_template_data);
        $index = 0;
        $current_row = 0;
        
        foreach ( $converted_template_data as $template_data ) {
            $element = isset($template_data['element']) ? $template_data['element'] : '';
            
            if ( empty( $element ) ) {
                continue;
            }
            
            $row_fields = $this->create_filter_row_structure($index, $element, $all_filter_options, $current_row === $valid_template_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $element, $all_filter_options, $selected_elements);
            
            // Set the values from old template data
            $row_fields = $this->set_old_template_values($row_fields, $template_data, $index);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
            $current_row++;
        }

        return $fields;
    }

    /**
     * Convert old template data to new structure
     * 
     * @since 1.3.1
     */
    private function convert_old_template_to_new_structure($old_template_data) {
        static $field_mapping = null;
        
        if ( null === $field_mapping ) {
            $field_mapping = array(
                'wt_iew_orders' => array('new_field' => 'orders', 'condition' => 'include'),
                'wt_iew_order_status' => array('new_field' => 'order_status', 'condition' => 'include'),
                'wt_iew_products' => array('new_field' => 'item_ordered', 'condition' => 'include'),
                'wt_iew_order_productscat' => array('new_field' => 'product_category', 'condition' => 'equals'),
                'wt_iew_email' => array('new_field' => 'customer', 'condition' => 'include'),
                'wt_iew_vendor' => array('new_field' => 'vendor', 'condition' => 'equals'),
                'wt_iew_coupons' => array('new_field' => 'coupons', 'condition' => 'equals'),
                'wt_iew_order_payment_method' => array('new_field' => 'payment_method', 'condition' => 'include'),
                'wt_iew_shipping_method' => array('new_field' => 'shipping_method', 'condition' => 'include'),
            );
        }
        
        $converted_data = array();
        
        // Handle simple field mappings
        foreach ( $field_mapping as $old_field => $field_config ) {
            if ( isset( $old_template_data[$old_field] ) && ! empty( $old_template_data[$old_field] ) ) {
                $value = $old_template_data[$old_field];
                $new_field = $field_config['new_field'];
                $condition = $field_config['condition'];

                $converted_data[] = array(
                    'element' => $new_field,
                    'value' => $value,
                    'condition' => $condition,
                    'value_for' => $new_field
                );
            }
        }
        
        //Handle date range specially (WT_IEW_DATE_FROM / WT_IEW_DATE_TO)
        if ( ! empty($old_template_data['wt_iew_date_from']) ||
            ! empty($old_template_data['wt_iew_date_to'])) {
            $converted_data[] = array(
                'element' => 'order_date',
                'condition' => 'in_between',
                'value' => array(
                    'from' => ! empty($old_template_data['wt_iew_date_from']) ? $old_template_data['wt_iew_date_from'] : '',
                    'to' => ! empty($old_template_data['wt_iew_date_to']) ? $old_template_data['wt_iew_date_to'] : ''
                ),
                'value_for' => 'order_date'
            );
   
        }       
        
        return $converted_data;
    }

       /**
     * Set old template values in the row fields
     * 
     * @since 1.3.1
     */
    private function set_old_template_values($row_fields, $template_data, $index) {
        if ( ! isset( $row_fields['rows'] ) || ! is_array( $row_fields['rows'] ) ) {
            return $row_fields;
        }
        
        $element = $template_data['element'];
        $value = $template_data['value'];
        $condition = $template_data['condition'];
        $element_field_name = 'wt_iew_filter_element[' . $index . ']';
        
        foreach ( $row_fields['rows'] as $row_index => $row ) {
            $field_name = $row['field_name'] ?? '';
            
            // Set element selector value
            if ( $field_name === $element_field_name ) {
                $row_fields['rows'][$row_index]['value'] = $element;
            }
            
            // Set condition field value
            if ( isset( $row['condition_field'] ) && $row['condition_field'] === true ) {
                $condition_type = $row['data_value_for'] ?? '';
                
                if ( 'greater-than-less-than' === $condition_type ) {
                    $row_fields['rows'][$row_index]['value'] = $condition;
                }
            }
            
            // Set value field
            if ( isset( $row['data_value_for'] ) && $row['data_value_for'] === $element && ! empty( $value ) ) {
                $row_fields['rows'][$row_index]['value'] = $value;
            }
        }
        
        return $row_fields;
    }

    /**
     * Create default filter rows
     * 
     * @since 1.3.1
     */
    private function create_default_rows($fields, $all_filter_options) {
        $default_filter_options = $this->get_default_filter_options();
        $default_filter_count = count($default_filter_options);
        $index = 0;
        
        foreach ( $default_filter_options as $filter_key => $filter_value ) {
            $row_fields = $this->create_filter_row_structure($index, $filter_key, $all_filter_options, $index === $default_filter_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $filter_key, $all_filter_options, array(), $default_filter_options);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
        }
        
        return $fields;
    }
    
    /**
     * Count valid template entries
     * 
     * @since 1.3.1
     */
    private function count_valid_template_entries($saved_template_data) {
        $count = 0;
        foreach ( $saved_template_data as $template_data ) {
            $element = isset( $template_data['element'] ) ? $template_data['element'] : '';
            if ( ! empty( $element ) ) {
                $count++;
            }
        }
        return $count;
    }
    
    /**
     * Create the basic structure for a filter row
     * 
     * @since 1.3.1
     */
    private function create_filter_row_structure($index, $element, $all_filter_options, $is_last_row) {
        return array(
            'identifier' => 'multi_field_row_' . ($index + 1),
            'add_new_btn' => $is_last_row,
            'row_delete_btn' => true,
            'default_value' => $element,
            'default_condition' => isset($all_filter_options[$element]) ? $all_filter_options[$element]['condition'] : 'equals-not-equals',
            'row_container_attr' => array('style' => 'display:flex;flex-direction:row;gap:40px;'),
            'row_main_attr' => array('style' => 'display:flex;flex-direction:row;gap:4px;align-items:center;padding: 10px 0px;margin-left: 24px;'),
            'rows' => array()
        );
    }
    
    /**
     * Build the fields for a filter row
     * 
     * @since 1.3.1
     */
    private function build_filter_row_fields($index, $element, $all_filter_options, $selected_elements = array(), $default_filter_options = array() ) {
        $rows = array();
        
        // Element selector field
        $available_options = $this->get_available_options( $element, $all_filter_options, $selected_elements, $index, $default_filter_options );
        $rows[] = array(
            'field_name' => 'wt_iew_filter_element[' . $index . ']',
            'type' => 'select',
            'value' => $element,
            'sele_vals' => $available_options,
            'css_class' => '',
            'attr' => $this->get_common_style_for_element_selector(),
        );
        
        // Add condition fields
        $rows = array_merge( $rows, $this->get_condition_fields( $index ) );
        
        // Add value fields
        $value_fields = $this->get_value_fields( $index );
        $rows = array_merge( $rows, array_values( $value_fields ) );
        
        return $rows;
    }
    
    /**
     * Get available options for element selector
     * 
     * @since 1.3.1
     */
    private function get_available_options($current_element, $all_filter_options, $selected_elements, $index = 0, $default_filter_options = array() ) {
        $available_options = array();
        
        if ( ! empty( $selected_elements ) ) {
            // For saved template data - exclude already selected elements
            foreach ( $all_filter_options as $option_key => $option_value ) {
                if ( ! in_array( $option_key, $selected_elements ) || $option_key === $current_element ) {
                    $available_options[$option_key] = $option_value;
                }
            }
        } else {
            // For default rows - all rows show the same available options
            // Exclude only the default elements that are used in other rows
            $default_elements = array_keys( $default_filter_options );
            
            foreach ( $all_filter_options as $option_key => $option_value ) {
                // Include all options except the ones used as defaults in other rows
                if ( ! in_array( $option_key, $default_elements ) || $option_key === $current_element ) {
                    $available_options[$option_key] = $option_value;
                }
            }
        }
        
        return $available_options;
    }

    /**
     * Apply saved template values to fields
     * 
     * @since 1.3.1
     */
    private function alter_filter_fields_with_saved_template($fields, $saved_template_data) {
        // Early validation
        if ( empty($fields) || ! is_array( $fields ) || 
            empty( $saved_template_data ) || ! is_array( $saved_template_data ) ) {
            return $fields;
        }
        
        // Cache filter options to avoid repeated array_merge calls
        $all_filter_options = array_merge( $this->get_default_filter_options(), $this->get_extra_filter_options() );

        foreach ( $saved_template_data as $template_key => $template_data ) {
            // Extract data with null coalescing for cleaner code
            $element = $template_data['element'] ?? '';
            $condition = $template_data['condition'] ?? '';
            $value = $template_data['value'] ?? '';

            // Skip empty elements early
            if ( empty( $element ) ) {
                continue;
            }

            // Get condition type and skip if not found
            $element_condition_type = $all_filter_options[$element]['condition'] ?? '';
            if ( empty( $element_condition_type ) ) {
                continue;
            }

            // Extract row index with validation
            if ( ! preg_match( '/multi_field_row_(\d+)/', $template_key, $matches ) ) {
                continue;
            }
            $row_index = max(0, intval($matches[1]) - 1);

            // Validate field exists and has rows
            if ( ! isset( $fields['multi_field_row']['fields'][$row_index]['rows'] ) || 
                ! is_array( $fields['multi_field_row']['fields'][$row_index]['rows'] ) ) {
                continue;
            }

            $rows = &$fields['multi_field_row']['fields'][$row_index]['rows'];

            // Process each row field
            foreach ( $rows as $row_idx => &$row ) {
                $this->set_field_value_if_matches($row, $row_idx, $element, $condition, $value, $element_condition_type, $row_index);
            }
        }

        return $fields;
    }

    /**
     * Helper method to set field values if they match the criteria
     * 
     * @since 1.3.1
     */
    private function set_field_value_if_matches(&$row, $row_idx, $element, $condition, $value, $element_condition_type, $row_index) {
        // Set element selector value
        if ( isset($row['field_name'] ) && 
            preg_match('/wt_iew_filter_element\[(\d+)\]/', $row['field_name'], $matches) &&
            intval($matches[1]) === $row_index && 
            isset($row['sele_vals'][$element])) {
            $row['value'] = $element;
        }

        // Set condition field value
        if ( isset( $row['condition_field']) && true === $row['condition_field'] &&
            isset( $row['data_value_for'] ) && $row['data_value_for'] === $element_condition_type && 
            ! empty( $condition ) ) {
            $row['value'] = $condition;
        }

        // Set value field
        if ( isset( $row['data_value_for'] ) && $row['data_value_for'] === $element && ! empty( $value ) ) {
            $row['value'] = $value;
        }
    }

    /**
     * Get filter column headers
     * 
     * @since 1.3.1
     */
    private function get_filter_column_headers() {
        return array(
            'row_container_attr' => array('style' => 'display:flex;flex-direction:row;gap:40px;'),
            'row_main_attr' => array('style' => 'display:flex;flex-direction:row;gap:4px;align-items:center;padding: 10px 0px;margin-left: 24px;'),
            'type' => 'header_row',
            'rows' => array(
                array(
                    'type' => 'label',
                    'label' => __('Element', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 250px;'),
                ),
                array(
                    'type' => 'label',
                    'label' => __('Condition', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 132px;'),
                ),
                array(
                    'type' => 'label',
                    'label' => __('Value', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 350px;'),
                ),
            ),
        );
    }

    /**
     * Get default filter options
     * 
     * @since 1.3.1
     */
    private function get_default_filter_options() {
        return array(
            'orders' => array('label' => __('Order', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Order', 'wt-import-export-for-woo')),
            'order_date' => array('label' => __('Order Date', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Order Date', 'wt-import-export-for-woo')),
            'order_status' => array('label' => __('Order Status', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Order Status', 'wt-import-export-for-woo')),
            'customer' => array('label' => __('Customer', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Customer', 'wt-import-export-for-woo')),
            'item_ordered' => array('label' => __('Item ordered', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Item ordered', 'wt-import-export-for-woo')),
        );
    }
    
    /**
     * Get extra filter options
     * 
     * @since 1.3.1
     */
    private function get_extra_filter_options() {
        return array(
            'user_role' => array('label' => __('User Role', 'wt-import-export-for-woo'), 'condition' => 'equals-not-equals', 'value' => __('User Role', 'wt-import-export-for-woo')),
            'billing_address' => array('label' => __('Billing Address', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Billing Address', 'wt-import-export-for-woo')),
            'shipping_address' => array('label' => __('Shipping Address', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Shipping Address', 'wt-import-export-for-woo')),
            'order_total' => array('label' => __('Order Total', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Order Total', 'wt-import-export-for-woo')),
            'payment_method' => array('label' => __('Payment Method', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Payment Method', 'wt-import-export-for-woo')),
            'shipping_method' => array('label' => __('Shipping Method', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Shipping Method', 'wt-import-export-for-woo')),
            'product_category' => array('label' => __('Product Category', 'wt-import-export-for-woo'), 'condition' => 'equals-not-equals', 'value' => __('Product Category', 'wt-import-export-for-woo')),
            'vendor' => array('label' => __('Vendor', 'wt-import-export-for-woo'), 'condition' => 'equals-not-equals', 'value' => __('Vendor', 'wt-import-export-for-woo')),
            'coupons' => array('label' => __('Coupons', 'wt-import-export-for-woo'), 'condition' => 'equals-not-equals', 'value' => __('Coupons', 'wt-import-export-for-woo')),
            'completed_date' => array('label' => __('Completed date', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Completed date', 'wt-import-export-for-woo')),
            'paid_date' => array('label' => __('Paid date', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Paid date', 'wt-import-export-for-woo')),
            'modification_date' => array('label' => __('Modification date', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Modification date', 'wt-import-export-for-woo')),
        );
    }

    /**
     * Get common style for element selector
     * 
     * @since 1.3.1
     */
    private function get_common_style_for_element_selector() {
        return array('style' => 'width: 250px; height: 40px; border-radius: 6px;');
    }

    /**
     * Get common style for value fields
     * 
     * @since 1.3.1
     */
    private function get_common_style_for_value_fields() {
        return array('style' => 'width: 350px; height: 40px; border-radius: 6px;');
    }

    /**
     * Get condition fields
     * 
     * @since 1.3.1
     */
    private function get_condition_fields($index) {
        $style = array('style' => 'width: 132px; height: 40px; border-radius: 6px;');
        return array(
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'include' => __('Include', 'wt-import-export-for-woo'),
                    'exclude' => __('Exclude', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'exc-inc',
                'visible_if' => 'exc-inc',
                'condition_field' => true,
            ),
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'equals' => __('Equals', 'wt-import-export-for-woo'),
                    'not_equals' => __('Not equals', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'equals-not-equals',
                'visible_if' => 'equals-not-equals',
                'condition_field' => true,
            ),
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'contains' => __('Contains', 'wt-import-export-for-woo'),
                    'not_contains' => __('Not contains', 'wt-import-export-for-woo'),
                    'starts_with' => __('Starts with', 'wt-import-export-for-woo'),
                    'ends_with' => __('Ends with', 'wt-import-export-for-woo'),
                    'equals' => __('Equals', 'wt-import-export-for-woo'),
                    'not_equals' => __('Not equals', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'contains-not-contains',
                'visible_if' => 'contains-not-contains',
                'condition_field' => true,
            ),
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'greater_than' => __('Greater than', 'wt-import-export-for-woo'),
                    'less_than' => __('Less than', 'wt-import-export-for-woo'),
                    'equals' => __('Equals', 'wt-import-export-for-woo'),
                    'not_equals' => __('Not equals', 'wt-import-export-for-woo'),
                    'in_between' => __('In between', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'greater-than-less-than',
                'visible_if' => 'greater-than-less-than',
                'condition_field' => true,
            ),
        );
    }

    /**
     * Get value fields
     * 
     * @since 1.3.1
     */
    private function get_value_fields($index) {
        $style = $this->get_common_style_for_value_fields();
    
        return array(
            'orders' => array(
                'field_name' => "orders[$index]",
                'data_value_for' => 'orders',
                'type' => 'text',
                'placeholder' => __('Enter order IDs/numbers', 'wt-import-export-for-woo'),
                'attr' => $style,
                'visible_if' => 'orders',
            ),
            'order_date' => array(
                'field_name' => "order_date[$index]",
                'data_value_for' => 'order_date',
                'type' => 'custom_date_range',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'order_date',
                'index' => $index,
            ),
            'order_status' => array(
                'field_name' => "order_status[$index]",
                'data_value_for' => 'order_status',
                'type' => 'multi_select',
                'placeholder' => __('All orders', 'wt-import-export-for-woo'),
                'sele_vals' => self::get_order_statuses(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'order_status',
            ),
            'customer' => array(
                'field_name' => "customer[$index]",
                'data_value_for' => 'customer',
                'type' => 'multi_select',
                'placeholder' => __('Search for customers', 'wt-import-export-for-woo'),
                'sele_vals' => array(),
                'css_class' => 'wc-customer-search',
                'attr' => $style,
                'visible_if' => 'customer',
            ),
            'user_role' => array(
                'field_name' => "user_role[$index]",
                'data_value_for' => 'user_role',
                'type' => 'multi_select',
                'placeholder' => __('Select user roles', 'wt-import-export-for-woo'),
                'sele_vals' => $this->get_user_roles(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'user_role',
            ),
            'billing_address' => array(
                'field_name' => "billing_address[$index]",
                'data_value_for' => 'billing_address',
                'type' => 'custom_address_filter',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'billing_address',
                'index' => $index,
                'address_type' => 'billing',
            ),
            'shipping_address' => array(
                'field_name' => "shipping_address[$index]",
                'data_value_for' => 'shipping_address',
                'type' => 'custom_address_filter',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'shipping_address',
                'index' => $index,
                'address_type' => 'shipping',
            ),
            'order_total' => array(
                'field_name' => "order_total[$index]",
                'data_value_for' => 'order_total',
                'type' => 'numeric_range',
                'placeholder' => __('Enter amount', 'wt-import-export-for-woo'),
                'from_placeholder' => __('From amount', 'wt-import-export-for-woo'),
                'to_placeholder' => __('To amount', 'wt-import-export-for-woo'),
                'step' => '0.01',
                'css_class' => 'price',
                'min' => '0',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'order_total',
                'index' => $index,
            ),
            'payment_method' => array(
                'field_name' => "payment_method[$index]",
                'data_value_for' => 'payment_method',
                'type' => 'multi_select',
                'placeholder' => __('All payment methods', 'wt-import-export-for-woo'),
                'sele_vals' => $this->wt_get_payment_methods(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'payment_method',
            ),
            'shipping_method' => array(
                'field_name' => "shipping_method[$index]",
                'data_value_for' => 'shipping_method',
                'type' => 'multi_select',
                'placeholder' => __('All shipping methods', 'wt-import-export-for-woo'),
                'sele_vals' => $this->wt_get_shipping_methods(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'shipping_method',
            ),
            'item_ordered' => array(
                'field_name' => "item_ordered[$index]",
                'data_value_for' => 'item_ordered',
                'type' => 'multi_select',
                'placeholder' => __('Search for products', 'wt-import-export-for-woo'),
                'sele_vals' => array(),
                'css_class' => 'wc-product-search',
                'attr' => $style,
                'visible_if' => 'item_ordered',
            ),
            'product_category' => array(
                'field_name' => "product_category[$index]",
                'data_value_for' => 'product_category',
                'type' => 'multi_select',
                'placeholder' => __('Any category', 'wt-import-export-for-woo'),
                'sele_vals' => $this->get_product_categories(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'product_category',
            ),
            'vendor' => array(
                'field_name' => "vendor[$index]",
                'data_value_for' => 'vendor',
                'type' => 'multi_select',
                'placeholder' => __('Search for vendors', 'wt-import-export-for-woo'),
                'sele_vals' => array(),
                'css_class' => 'wc-customer-search',
               // 'attr' => $style + array('data-action' => 'wt_json_search_vendors'),
               'attr' => $style + array('data-exclude' => '__VENDOR_FIELD__'),
                'visible_if' => 'vendor',
            ),
            'coupons' => array(
                'field_name' => "coupons[$index]",
                'data_value_for' => 'coupons',
                'type' => 'multi_select',
                'placeholder' => __('Search for coupons', 'wt-import-export-for-woo'),
                'sele_vals' => array(),
                'css_class' => 'wt-coupon-search',
                'attr' => $style,
                'visible_if' => 'coupons',
            ),
            'completed_date' => array(
                'field_name' => "completed_date[$index]",
                'data_value_for' => 'completed_date',
                'type' => 'custom_date_range',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'completed_date',
                'index' => $index,
            ),
            'paid_date' => array(
                'field_name' => "paid_date[$index]",
                'data_value_for' => 'paid_date',
                'type' => 'custom_date_range',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'paid_date',
                'index' => $index,
            ),
            'modification_date' => array(
                'field_name' => "modification_date[$index]",
                'data_value_for' => 'modification_date',
                'type' => 'custom_date_range',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'modification_date',
                'index' => $index,
            ),
        );
    }

    public function exporter_alter_filter_fields_old($fields, $base, $filter_form_data) {

        if ( $this->module_base !== $base ) {
            return $fields;
        }
    
            /* altering help text of default fields */
            $fields['limit']['label']=__('Total number of orders to export', 'wt-import-export-for-woo'); 
            $fields['limit']['help_text']=__('Exports specified number of orders. e.g. Entering 500 with a skip count of 10 will export orders from 11th to 510th position.', 'wt-import-export-for-woo');
            $fields['offset']['label']=__('Skip first <i>n</i> orders', 'wt-import-export-for-woo');
            $fields['offset']['help_text']=__('Skips specified number of orders from the beginning. e.g. Enter 10 to skip first 10 orders from export.', 'wt-import-export-for-woo');
    
            
            
            $fields['orderfrom_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3"> '.__( 'Filter by date', 'wt-import-export-for-woo' ).' <span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );
            
            $fields['order_date_options'] = array(
                'tr_html' => '<tr class="orderfilter orderfilter-first">
                <td colspan="3">
                    <select name="wt_iew_date_option" id="wt_iew_date_option" style="width:150px;float:left;margin-right:10px;">
                            <option value="post_date">'.__('Order date', 'wt-import-export-for-woo').'</option>
                            <option value="post_modified">'.__('Modified date', 'wt-import-export-for-woo').'</option>	
                            <option value="date_paid">'.__('Paid date', 'wt-import-export-for-woo').'</option>
                            <option value="date_completed">'.__('Completed date', 'wt-import-export-for-woo').'</option>
                        </select>
                </td>
                <td colspan="3" style="margin-top: 10px;">
                <label for="date_type" style="padding: 7px;background-color: #cccc;margin-right: 7px;"><input type="radio" id="wt_iew_date_type" name="wt_iew_date_type" value="custom" checked> '.__('Custom', 'wt-import-export-for-woo').'  </label>
                <label for="date_type" style="padding: 7px;background-color: #cccc;margin-right: 7px;"><input type="radio" id="wt_iew_date_type" name="wt_iew_date_type" value="last_week" > '.__('Last week', 'wt-import-export-for-woo').'  </label>
                <label for="date_type" style="padding: 7px;background-color: #cccc;margin-right: 7px;"><input type="radio" id="wt_iew_date_type" name="wt_iew_date_type" value="last_month" > '.__('Last month', 'wt-import-export-for-woo').'  </label>
                <label for="date_type" style="padding: 7px;background-color: #cccc;margin-right: 7px;"><input type="radio" id="wt_iew_date_type" name="wt_iew_date_type" value="last_year" > '.__('Last year', 'wt-import-export-for-woo').'  </label><br/><br/>
                <input placeholder="'.__('Date from', 'wt-import-export-for-woo').'" type="text" class="wt_iew_datepicker" name="wt_iew_date_from" value=""> <br/>
                <input placeholder="'.__('Date to', 'wt-import-export-for-woo').'" type="text" class="wt_iew_datepicker" name="wt_iew_date_to" value="">
                </td>
                </tr>'
                
            );
            
            $fields['orderids_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by order ID', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );
            
            $fields['orders'] = array(
                'label' => __('Order IDs', 'wt-import-export-for-woo'),
                'placeholder' => __('Enter order IDs separated by comma', 'wt-import-export-for-woo'),
                'field_name' => 'orders',
                'sele_vals' => '',
                'help_text' => __('Enter order IDs separated by comma to export specific orders.', 'wt-import-export-for-woo'),
                'type' => 'text',
                'css_class' => '',
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );
            $fields['orderstatus_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by order status', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );            
            $fields['order_status'] = array(
                'label' => __('Order Statuses', 'wt-import-export-for-woo'),
                'placeholder' => __('All Order', 'wt-import-export-for-woo'),
                'field_name' => 'order_status',
                'sele_vals' => self::get_order_statuses(),
                'help_text' => __('Filter orders by their status type. You can specify more than one type for export.', 'wt-import-export-for-woo'),
                'type' => 'multi_select',
                'css_class' => 'wc-enhanced-select',
                'validation_rule' => array('type'=>'text_arr'),
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );
            $fields['orderproducts_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by product', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );			
            $fields['products'] = array(
                'label' => __('Product', 'wt-import-export-for-woo'),
                'placeholder' => __('Search for a product&hellip;', 'wt-import-export-for-woo'),
                'field_name' => 'products',
                'sele_vals' => array(),
                'help_text' => __('Export orders containing specific products. Enter the product name or SKU or ID to export orders containing specified products.', 'wt-import-export-for-woo'),
                'type' => 'multi_select',
                'css_class' => 'wc-product-search',
                'validation_rule' => array('type'=>'text_arr'),
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );
            
            $fields['orderproductscat_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by product category', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );			
            $fields['order_productscat'] = array(
                'label' => __('Product category', 'wt-import-export-for-woo'),
                'placeholder' => __('Search for a category&hellip;', 'wt-import-export-for-woo'),
                'field_name' => 'order_productscat',
                'sele_vals' => $this->get_product_categories(),
                'help_text' => __('Export orders that belong to specific product categories. Search and select one or multiple categories.', 'wt-import-export-for-woo'),
                'type' => 'multi_select',
                'css_class' => 'wc-enhanced-select',
                'validation_rule' => array('type'=>'text_arr'),
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );	
            
            $fields['orderemail_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by customer', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );		
            $fields['email'] = array(
                'label' => __('Email'),
                'placeholder' => __('Search for a customer&hellip;', 'wt-import-export-for-woo'),
                'field_name' => 'email',
                'sele_vals' => array(),
                'help_text' => __('Input the customer email to export orders pertaining to only these customers.', 'wt-import-export-for-woo'),
                'type' => 'multi_select',
                'css_class' => 'wc-customer-search',
                'validation_rule' => array('type'=>'text_arr'),
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );
                        
            $fields['ordervendor_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by vendor', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );		
            $fields['vendor'] = array(
                'label' => __('Vendor'),
                'placeholder' => __('Search for a vendor&hellip;', 'wt-import-export-for-woo'),
                'field_name' => 'vendor',
                'sele_vals' => array(),
                'help_text' => __('Input the vendor email to export orders pertaining to only these vendors.', 'wt-import-export-for-woo'),
                'type' => 'select',
                'css_class' => 'wc-customer-search',
                'validation_rule' => array('type'=>'text_arr'),
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );
            
            $fields['ordercoupons_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by coupons', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );	
            
            $fields['coupons'] = array(
                'label' => __('Coupons', 'wt-import-export-for-woo'),
                'placeholder' => __('Search for a coupon&hellip;', 'wt-import-export-for-woo'),
                'field_name' => 'coupons',
                'sele_vals' => array(),
                'help_text' => __('Exports orders redeemed with specific coupon codes. Multiple coupon codes can be selected.', 'wt-import-export-for-woo'),
                'type' => 'multi_select',
                'css_class' => 'wt-coupon-search',
                'validation_rule' => array('type'=>'text_arr'),
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );
            
            
            $fields['orderpaymentmethods_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by payment methods', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );
            
            $payment_method_list = array();
            $payment_methods = WC()->payment_gateways->payment_gateways();
            foreach ($payment_methods as $payment_method) {
    
                //1.2.9 - For  Payments methods in WooPayments plugin, get the title using get_title() instead of title.
                if ( 0 === strpos($payment_method->id, 'woocommerce_payments') && method_exists( $payment_method, 'get_title' ) ) {
                    // For WooCommerce Payments methods, get the title from the payment method object
                    $payment_method_list[$payment_method->id] = $payment_method->get_title();
                } else {
                    // For other payment methods, use the standard title
                    $payment_method_list[$payment_method->id] = $payment_method->title;
                }
            }
    
            $fields['order_payment_method'] = array(
                'label' => __('Payment methods', 'wt-import-export-for-woo'),
                'placeholder' => __('Search for a payment method&hellip;', 'wt-import-export-for-woo'),
                'field_name' => 'order_payment_method',
                'sele_vals' => $payment_method_list,
                'help_text' => __('Exports orders placed with specific payment methods.', 'wt-import-export-for-woo'),
                'type' => 'multi_select',
                'css_class' => 'wc-enhanced-select',
                'validation_rule' => array('type'=>'text_arr'),
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );
            
            $fields['ordershippingmethods_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by shipping methods', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );	
            
            $shipping_method_list = array();
            $shipping_methods = WC()->shipping->load_shipping_methods();
            foreach ($shipping_methods as $shipping_method) {
                $shipping_method_list[$shipping_method->id] = $shipping_method->method_title;
            }
            $fields['shipping_method'] = array(
                'label' => __('Shipping methods', 'wt-import-export-for-woo'),
                'placeholder' => __('Search for a shipping method&hellip;', 'wt-import-export-for-woo'),
                'field_name' => 'shipping_method',
                'sele_vals' => $shipping_method_list,
                'help_text' => __('Exports orders placed with specific shipping method.', 'wt-import-export-for-woo'),
                'type' => 'multi_select',
                'css_class' => 'wc-enhanced-select',
                'validation_rule' => array('type'=>'text_arr'),
                'tr_class' => 'orderfilter',
                'merge_right' => true				
            );
            
            $fields['orderbillingaddr_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by billing address', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );	
            
    
            $fields['billing_addr_options'] = array(
                
                'tr_html' => '<tr class="orderfilter">
                <th class="">
                    <label>'.__('Billing address', 'wt-import-export-for-woo').'</label>
                </th>
                <td colspan="3">
                    <select name="wt_billing_locations" id="wt_billing_locations" style="width:150px;float:left;margin-right:10px;">
                            <option>'.__('--Select--', 'wt-import-export-for-woo').'</option>
                            <option>'.__('Country', 'wt-import-export-for-woo').'</option>
                            <option>'.__('State', 'wt-import-export-for-woo').'</option>
                        </select>
                        <select name="wt_billing_compare" id="wt_billing_compare" style="width:130px;float:left;margin-right:10px;">
                            <option value="=">is equal to</option>
                            <option value="&lt;&gt;">not equal to</option>
                        </select>
                        <button id="add_billing_locations" class="button-secondary">'.__('Select', 'wt-import-export-for-woo').'</button>
                                <select id="wt_iew_billing_locations_check" class="wc-enhanced-select" multiple
                        name="wt_iew_billing_locations_check"
                        style="width: 100%; max-width: 25%;">
                </select>
                </td></tr>'
                
            );
            
            
            $fields['ordershippingaddr_empty_row'] = array(
                'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__( 'Filter by shipping address', 'wt-import-export-for-woo' ).'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
            );	
            
            $fields['shipping_addr_options'] = array(
                
                'tr_html' => '<tr class="orderfilter">
                <th class="">
                    <label>'.__('Shipping address', 'wt-import-export-for-woo').'</label>
                </th>
                <td colspan="3">
                    <select name="wt_shipping_locations" id="wt_shipping_locations" style="width:150px;float:left;margin-right:10px;">
                            <option>'.__('--Select--', 'wt-import-export-for-woo').'</option>
                            <option>'.__('Country', 'wt-import-export-for-woo').'</option>
                            <option>'.__('State', 'wt-import-export-for-woo').'</option>
                        </select>
                        <select name="wt_shipping_compare" id="wt_shipping_compare" style="width:130px;float:left;margin-right:10px;">
                            <option value="=">is equal to</option>
                            <option value="&lt;&gt;">not equal to</option>
                        </select>
                        <button id="add_shipping_locations" class="button-secondary">'.__('Select', 'wt-import-export-for-woo').'</button>
                                <select id="wt_iew_shipping_locations_check" class="wc-enhanced-select" multiple
                        name="wt_iew_shipping_locations_check"
                        style="width: 100%; max-width: 25%;">
                </select>
                </td></tr>'
                
            );
            
            
    
        // Move the limit and offset filters to the bottom ( less priority ).
        $limit_options = $fields['limit'];
        $offset_options = $fields['offset'];
        unset( $fields['limit'], $fields['offset'] );
        $fields['orderlimit_empty_row'] = array(
            'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__('Total number of orders to export', 'wt-import-export-for-woo').'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
        );
        $fields['limit'] = $limit_options;
        $fields['limit']['tr_class'] = 'orderfilter';
        $fields['limit']['merge_right'] = true;
        $fields['orderoffset_empty_row'] = array(
            'tr_html' => '<tr class="orderfilterby"><td colspan="3">'.__('Number of orders to skip from the beginning', 'wt-import-export-for-woo').'<span style="float:right;" class="dashicons dashicons-arrow-down-alt2"></span></td></tr>'
        );
        $fields['offset'] = $offset_options;
        $fields['offset']['tr_class'] = 'orderfilter';
        $fields['offset']['merge_right'] = true;
        
        return $fields;
    }

    /**
     * Add extra filter fields (limit, offset, sort, etc.)
     * 
     * @since 1.3.1
     */
    private function add_extra_filter_fields($fields, $base, $filter_form_data) {
        if ( ! apply_filters( 'wt_iew_exporter_add_extra_filter_fields', false, $base, $filter_form_data ) ) {
            return $fields;
        }
        
        $fields['limit'] = array(
            'label' => __('Total number of orders to export', 'wt-import-export-for-woo'),
            'value' => '',
            'type' => 'number',
            'attr' => array('step' => 1, 'min' => 0),
            'field_name' => 'limit',
            'placeholder' => __('Unlimited', 'wt-import-export-for-woo'),
            'help_text' => __('Exports specified number of orders. e.g. Entering 500 with a skip count of 10 will export orders from 11th to 510th position.', 'wt-import-export-for-woo'),
        );
        
        $fields['offset'] = array(
            'label' => __('Skip first <i>n</i> orders', 'wt-import-export-for-woo'),
            'value' => '',
            'field_name' => 'offset',
            'placeholder' => '0',
            'help_text' => __('Skips specified number of orders from the beginning. e.g. Enter 10 to skip first 10 orders from export.', 'wt-import-export-for-woo'),
            'type' => 'number',
            'attr' => array('step' => 1, 'min' => 0),
            'validation_rule' => array('type' => 'absint')
        );
        
        $sort_columns = self::get_order_sort_columns();
        $fields['sort_columns'] = array(
            'label' => __('Sort Columns', 'wt-import-export-for-woo'),
            'placeholder' => __('ID'),
            'field_name' => 'sort_columns',
            'sele_vals' => $sort_columns,
            'help_text' => __('Sort the exported data based on the selected columns in order specified. Defaulted to ID.', 'wt-import-export-for-woo'),
            'type' => 'select',
            'validation_rule' => array('type' => 'text_arr')
        );
        
        $fields['order_by'] = array(
            'label' => __('Sort By', 'wt-import-export-for-woo'),
            'placeholder' => __('ASC'),
            'field_name' => 'order_by',
            'sele_vals' => array('ASC' => __('Ascending', 'wt-import-export-for-woo'), 'DESC' => __('Descending', 'wt-import-export-for-woo')),
            'help_text' => __('Defaulted to Ascending. Applicable to above selected columns in the order specified.', 'wt-import-export-for-woo'),
            'type' => 'select',
        );
        
        return $fields;
    }

    public function wt_get_shipping_methods() {
        $shipping_method_list = array();
			$shipping_methods = WC()->shipping->load_shipping_methods();
			foreach ($shipping_methods as $shipping_method) {
                // Right key for local pick up is "pickup_location", so we need to skip the "local_pickup" key to avoid duplicatation
                if ( "local_pickup" === $shipping_method->id ) {
                    continue;
                }
				$shipping_method_list[$shipping_method->id] = $shipping_method->method_title;
			}
			return $shipping_method_list;
    }

    public function wt_get_payment_methods() {
        $payment_method_list = array();
			$payment_methods = WC()->payment_gateways->payment_gateways();
			foreach ($payment_methods as $payment_method) {

                //1.2.9 - For  Payments methods in WooPayments plugin, get the title using get_title() instead of title.
				if ( 0 === strpos($payment_method->id, 'woocommerce_payments') && method_exists( $payment_method, 'get_title' ) ) {
					// For WooCommerce Payments methods, get the title from the payment method object
					$payment_method_list[$payment_method->id] = $payment_method->get_title();
				} else {
					// For other payment methods, use the standard title
					$payment_method_list[$payment_method->id] = $payment_method->title;
				}
			}
			return $payment_method_list;
    }

    public function get_user_roles() {
        if (!function_exists('get_editable_roles')) {
            require_once(ABSPATH . 'wp-admin/includes/user.php');
        }
        
        $user_roles = array();
        $roles = get_editable_roles();
        
        foreach ($roles as $role_key => $role_info) {
            $user_roles[$role_key] = $role_info['name'];
        }
        
        return $user_roles;
    }

    public static function wt_get_product_id_by_sku($sku) {
        global $wpdb;
        $post_exists_sku = $wpdb->get_var($wpdb->prepare("
	    		SELECT $wpdb->posts.ID
	    		FROM $wpdb->posts
	    		LEFT JOIN $wpdb->postmeta ON ( $wpdb->posts.ID = $wpdb->postmeta.post_id )
	    		WHERE $wpdb->posts.post_status IN ( 'publish', 'private', 'draft', 'pending', 'future' )
	    		AND $wpdb->postmeta.meta_key = '_sku' AND $wpdb->postmeta.meta_value = '%s'
	    		", $sku));
        if ($post_exists_sku) {
            return $post_exists_sku;
        }
        return false;
    }
    
    public function get_item_by_id($id) {
        $post['edit_url']=get_edit_post_link($id);
        $post['title'] = $id;
        return $post; 
    }
	public static function get_item_link_by_id($id) {
		if (self::wt_get_order_table_name()) {
			$post['edit_url'] = admin_url( "admin.php?page=wc-orders&action=edit&id={$id}" );
		} else {
			$post['edit_url'] = get_edit_post_link( $id );
		}
		$post['title']    = $id;
		return $post;
    }
		
	/**
	 * Gets the product categories.
	 * 
	 * @return array
	 */
	public function get_product_categories() {


        if(class_exists("WP_Term_Query")){
            $term_query = new \WP_Term_Query( [
                'taxonomy'	 => 'product_cat',
                'hide_empty' => true,
            ] );
    
            $product_categories_arr = array();
    
            $product_categories = $term_query->get_terms();
        }else{
            $sql = "SELECT t.name , t.slug 
            FROM wp_terms AS t
            INNER JOIN wp_term_taxonomy AS tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy = 'product_cat'
            AND tt.count > 0;
            ";
             global $wpdb;
            $product_categories = $wpdb->get_results($sql);
        }
        foreach ($product_categories as $key => $product_category ) {
            $product_categories_arr[$product_category->slug] = $product_category->name;
        }
		return $product_categories_arr;
	}
	
	/**
	 * 
	 * @global type $wpdb
	 * @param string $type
	 * @param string $key
	 * @return array
	 */
	public static function get_order_meta_values( $type, $key ) {
        global $wpdb;
    
        // Check if HPOS table is enabled
        $is_hpos_enabled = self::wt_get_order_table_name();
    
        // Determine the correct query to retrieve order IDs based on HPOS status
        if ($is_hpos_enabled) {
            // HPOS is enabled
            $order_ids = $wpdb->get_col( "SELECT id FROM {$wpdb->prefix}wc_orders WHERE type = 'shop_order' ORDER BY id DESC LIMIT 1000" );
            
        } else {
            // Legacy system
            $order_ids = $wpdb->get_col( "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'shop_order' ORDER BY ID DESC LIMIT 1000" );
        }
    
        // Check if we have order IDs
        if (empty($order_ids)) {
            return array();
        }
    
        // Format the order IDs as a comma-separated string
        $order_ids_placeholder = implode(',',$order_ids);
    
        // Prepare the query based on the table name
        if ($is_hpos_enabled) {
            $type = trim($type , '_');
            $key = strtolower($key);
            // HPOS table query
            $query = $wpdb->prepare(
                "SELECT DISTINCT $key FROM {$wpdb->prefix}wc_order_addresses  WHERE address_type = %s AND order_id IN ($order_ids_placeholder)",
                $type 
            );
        } else {
            // Legacy table query
            $query = $wpdb->prepare(
                "SELECT DISTINCT meta_value FROM  $wpdb->postmeta  WHERE meta_key = %s AND post_id IN ($order_ids_placeholder)",
                $type . strtolower($key)
            );
        }
    
        // Execute the query and process results
        $results = $wpdb->get_col($query);
        $data = array_filter($results);
        sort($data);
    
        // Output the data as JSON
        echo json_encode($data);
    }
    /**
	  * Is HPOS table enabled.
	  *
	  * @return bool
	  */
    public static function wt_get_order_table_name() {
	    if(get_option('woocommerce_custom_orders_table_enabled') === 'yes'){
            return true;
		}else{
            return false;
        }
	}
}

new Wt_Import_Export_For_Woo_Order();

// Add category/tag import export addon
include_once( __DIR__ . "/../coupon/coupon.php" );
include_once( __DIR__ . "/../subscription/subscription.php" );
