<?php

if (!defined('WPINC')) {
    exit;
}
use Automattic\WooCommerce\Utilities\OrderUtil;

class Wt_Import_Export_For_Woo_Subscription_Export {

    public $parent_module = null;
    private $line_items_max_count = 0;
    private $export_to_separate_columns = false;
    private $export_to_separate_rows = false;    
    private $line_item_meta;
    private $exclude_line_items = false;
    private $sql_fragments = array();
    private $header_keys = null; // Header columns. 

    public function __construct($parent_object) {

        $this->parent_module = $parent_object;
    }

    public function prepare_header() {

        if ( ! is_null( $this->header_keys ) ){
            return $this->header_keys;
        }


        $export_columns = $this->parent_module->get_selected_column_names();
        
        if($this->exclude_line_items){
            return apply_filters('hf_alter_coupon_csv_header', $export_columns);
        }
        
        $this->line_item_meta = self::get_all_line_item_metakeys();
        
        $max_line_items = $this->line_items_max_count;
      
        for ($i = 1; $i <= $max_line_items; $i++) {
            $export_columns["line_item_{$i}"] = "line_item_{$i}";
        }        
        if ($this->export_to_separate_columns) {
            for ($i = 1; $i <= $max_line_items; $i++) {
                foreach ($this->line_item_meta as $meta_value) {
                    $new_val = str_replace("_", " ", $meta_value);
                    $export_columns["line_item_{$i}_name"] = "Product Item {$i} Name";
                    $export_columns["line_item_{$i}_product_id"] = "Product Item {$i} id";
                    $export_columns["line_item_{$i}_sku"] = "Product Item {$i} SKU";
                    $export_columns["line_item_{$i}_quantity"] = "Product Item {$i} Quantity";
                    $export_columns["line_item_{$i}_total"] = "Product Item {$i} Total";
                    $export_columns["line_item_{$i}_subtotal"] = "Product Item {$i} Subtotal";
                    if (in_array($meta_value, array("_product_id", "_qty", "_variation_id", "_line_total", "_line_subtotal", "_tax_class", "_line_tax", "_line_tax_data", "_line_subtotal_tax"))) {
                        continue;
                    } else {
                        $export_columns["line_item_{$i}_$meta_value"] = "Product Item {$i} $new_val";
                    }
                }
            }
        }
        
        if ($this->export_to_separate_rows) {
            $export_columns = $this->wt_line_item_separate_row_csv_header($export_columns);
        }
                        
        $this->header_keys = apply_filters('hf_alter_coupon_csv_header', $export_columns, $max_line_items);

        return $this->header_keys;

    }
    
    public function wt_line_item_separate_row_csv_header($export_columns) {


        foreach ($export_columns as $s_key => $value) {
            if (strstr($s_key, 'line_item_')) {
                unset($export_columns[$s_key]);
            }
        }

        $export_columns["line_item_product_id"] = "item_product_id";
        $export_columns["line_item_name"] = "item_name";
        $export_columns["line_item_sku"] = "item_sku";
        $export_columns["line_item_quantity"] = "item_quantity";
        $export_columns["line_item_subtotal"] = "item_subtotal";
        $export_columns["line_item_subtotal_tax"] = "item_subtotal_tax";
        $export_columns["line_item_total"] = "item_total";
        $export_columns["line_item_total_tax"] = "item_total_tax";
        $export_columns["item_refunded"] = "item_refunded";
        $export_columns["item_refunded_qty"] = "item_refunded_qty";
        $export_columns["item_meta"] = "item_meta";
        return $export_columns;
    }

    public function wt_line_item_separate_row_csv_data($order, $order_export_data, $order_data_filter_args) {        
        if ($order) {
            /* to fix the data sort order issue  */ 
            $uset_data = array('line_item_product_id','line_item_name','line_item_sku','line_item_quantity','line_item_subtotal','line_item_subtotal_tax','line_item_total','line_item_total_tax','item_refunded','item_refunded_qty','item_meta');
            foreach ($uset_data as $value) {
                unset($order_export_data[$value]);
            }            
            foreach ($order->get_items() as $item_key => $item) {
                foreach ($order_export_data as $key => $value) {
                    if (strpos($key, 'line_item_') !== false) {
                        continue;
                    } else {
                        $data1[$key] = $value;
                    }
                }
                
                
                $item_data = $item->get_data();
                $product = $item->get_product();
                $data1["line_item_product_id"] = !empty($item_data['product_id']) ? $item_data['product_id'] : '';
                $data1["line_item_name"] = !empty($item_data['name']) ? $item_data['name'] : '';
                $data1["line_item_sku"] = !empty($product) ? $product->get_sku() : '';
                $data1["line_item_quantity"] = !empty($item_data['quantity']) ? $item_data['quantity'] : '';
                $data1["line_item_subtotal"] = !empty($item_data['subtotal']) ? $item_data['subtotal'] : 0;
                $data1["line_item_subtotal_tax"] = !empty($item_data['subtotal_tax']) ? $item_data['subtotal_tax'] : 0;
                $data1["line_item_total"] = !empty($item_data['total']) ? $item_data['total'] : 0;
                $data1["line_item_total_tax"] = !empty($item_data['total_tax']) ? $item_data['total_tax'] : 0;

                $data1["item_refunded"] = !empty($order->get_total_refunded_for_item($item_key)) ? $order->get_total_refunded_for_item($item_key) : '';
                $data1["item_refunded_qty"] = !empty($order->get_qty_refunded_for_item($item_key)) ? absint($order->get_qty_refunded_for_item($item_key)) : '';
                $data1["item_meta"] = !empty($item_data['meta_data']) ? json_encode($item_data['meta_data']) : '';

                
                $row[] = $data1;

            }
           return $row;
        }
   
    }

    public function wt_ier_alter_order_data_befor_export_for_separate_row($data_array) {
        $new_data_array = array();
        foreach ($data_array as $key => $avalue) {
            if (is_array($avalue)) {
                if (count($avalue) == 1) {
                    $new_data_array[] = $avalue[0];
                } elseif (count($avalue) > 1) {
                    foreach ($avalue as $arrkey => $arrvalue) {
                        $new_data_array[] = $arrvalue;
                    }
                }
            }
        }
        return $new_data_array;
    }

     /**
     * Prepare data that will be exported.
     */
    public function prepare_data_to_export($form_data, $batch_offset) {
 
        // Initialize SQL fragment collectors
        $this->sql_fragments = array(
            'left_joins' => array(),
            'where_conditions' => array(),
            'subqueries' => array(),
            'base_conditions' => array('1') // Always true base condition
        );
                
        // Extract basic parameters
        $export_limit = !empty($form_data['filter_form_data']['wt_iew_limit']) ? intval($form_data['filter_form_data']['wt_iew_limit']) : 999999999;
        $current_offset = !empty($form_data['filter_form_data']['wt_iew_offset']) ? intval($form_data['filter_form_data']['wt_iew_offset']) : 0;
        $batch_count = !empty($form_data['advanced_form_data']['wt_iew_batch_count']) ? $form_data['advanced_form_data']['wt_iew_batch_count'] : Wt_Import_Export_For_Woo_Common_Helper::get_advanced_settings('default_export_batch');
        $exclude_already_exported = (!empty($form_data['advanced_form_data']['wt_iew_exclude_already_exported']) && $form_data['advanced_form_data']['wt_iew_exclude_already_exported'] == 'Yes') ? true : false;
        
        // Handle exclude already exported subscriptions
        if ($exclude_already_exported) {
            $this->add_exclude_already_exported_filter();
        }
        
        // Setup export options
        $this->export_to_separate_columns = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate_columns']) && $form_data['advanced_form_data']['wt_iew_export_to_separate_columns'] == 'Yes') ? true : false;
        if (!$this->export_to_separate_columns) {
            $this->export_to_separate_columns = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate']) && $form_data['advanced_form_data']['wt_iew_export_to_separate'] == 'column') ? true : false;
        }

        $this->export_to_separate_rows = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate_rows']) && $form_data['advanced_form_data']['wt_iew_export_to_separate_rows'] == 'Yes') ? true : false;
        if (!$this->export_to_separate_rows) {
            $this->export_to_separate_rows = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate']) && $form_data['advanced_form_data']['wt_iew_export_to_separate'] == 'row') ? true : false;
        }
        $this->exclude_line_items = (!empty($form_data['advanced_form_data']['wt_iew_exclude_line_items']) && $form_data['advanced_form_data']['wt_iew_exclude_line_items'] == 'Yes') ? true : false;
        
        /**
         * Process filters using the new filter system.
         * Build the final optimized SQL query.
         * @since 1.3.1
         */
        $this->filter_subscriptions_to_export($form_data);
        $subscription_sql_query = $this->build_final_sql_query($export_limit, $current_offset);

        $filter_form_data = isset($form_data['filter_form_data']) && is_array($form_data['filter_form_data']) ? $form_data['filter_form_data'] : array();
        $advanced_form_data = is_array($form_data['advanced_form_data']) ? $form_data['advanced_form_data'] : array();
        $transient_key = 'wt_iew_subscriptions_export_hybrid_' . md5(json_encode(array_merge($filter_form_data, $advanced_form_data)));
        
        $total_queried_subscriptions = get_transient($transient_key);
        if (false === $total_queried_subscriptions) {
            // Execute single query and cache results
            global $wpdb;
            $total_queried_subscriptions = $wpdb->get_col($subscription_sql_query);
            set_transient($transient_key, $total_queried_subscriptions, 60); // Cache for 60 seconds
        }
        
        // Use cached subscription IDs with batch processing
        $subscription_ids = apply_filters('wt_subscriptionimpexpcsv_alter_subscription_ids', $total_queried_subscriptions);
        $total_records = count($subscription_ids);
        
        // Calculate batch limits
        if ($batch_count <= $export_limit) {
            if (($batch_offset + $batch_count) > $export_limit) { //last offset
                $limit = $export_limit - $batch_offset;
            } else {
                $limit = $batch_count;
            }
        } else {
            $limit = $export_limit - $batch_offset;
        }
        
        $data_array = array();
        if ($batch_offset < $export_limit) {
            
            // Set up line items max count for first batch
            if ($batch_offset == 0) {
                $this->line_items_max_count = $this->get_max_line_items();
                update_option('wt_subscription_line_items_max_count', $this->line_items_max_count);
            }

            if (empty($this->line_items_max_count)) {
                $this->line_items_max_count = get_option('wt_subscription_line_items_max_count');
            }
            
            // Process batch of subscription IDs
            $batch_subscription_ids = array_slice($subscription_ids, $batch_offset, $limit);
            
            foreach ($batch_subscription_ids as $subscription_id) {
                if (self::hf_get_subscription($subscription_id)) {
                    $data_array[] = $this->generate_subscription_row_data($subscription_id, $form_data);
                    
                    // Update exported status
                    if (self::is_hpos_enabled()) {
                        global $wpdb;
                        $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->prefix}wc_orders_meta WHERE meta_key = 'wt_subscription_exported_status' AND order_id = %d;", $subscription_id));
                        $wpdb->query($wpdb->prepare("INSERT INTO {$wpdb->prefix}wc_orders_meta (order_id, meta_key, meta_value) VALUES (%d, %s, %s)", $subscription_id, 'wt_subscription_exported_status', TRUE));
                    } else {
                        update_post_meta($subscription_id, 'wt_subscription_exported_status', TRUE);
                    }
                }
            }

            if($this->export_to_separate_rows){
                $data_array = $this->wt_ier_alter_order_data_befor_export_for_separate_row($data_array);
            }
            
            $data_array = apply_filters('wt_ier_alter_subscriptions_data_befor_export', $data_array);  
            
            $return['total'] = $total_records; 
            $return['data'] = $data_array;
            if ( 0 == $batch_offset && 0 == $total_records ){
                $return['no_post'] = __( 'Nothing to export under the selected criteria. Please check and try adjusting the filters.' );
            }
            return $return;
        }
        
        return array('total' => 0, 'data' => array());
    }  
    
    /**
     * Process filters for the new filter system
     * 
     * @since 1.3.1
     */
    public function filter_subscriptions_to_export($form_data) {
        // Check if multi-field filter data exists
        if ( empty( $form_data['filter_form_data']['wt_iew_multi_field_row_array'] ) ) {
            return;
        }
        
        // Decode the JSON string to get the filter array
        $filter_array = json_decode($form_data['filter_form_data']['wt_iew_multi_field_row_array'], true);
        
        if (empty($filter_array)) {
            return;
        }
      
        // Process each filter row and collect SQL fragments
        foreach ($filter_array as $key => $filter_row) {
            $element = isset($filter_row['element']) ? $filter_row['element'] : '';
            $value = isset($filter_row['value']) ? $filter_row['value'] : '';
            $condition = isset($filter_row['condition']) ? $filter_row['condition'] : '';
            
            // Skip empty filters
            if (empty($element)) {
                continue;
            }
     
            // Validate values based on filter type
            if (!$this->validate_filter_value($element, $value, $condition)) {
                continue;
            }
            
            // Apply filter and collect SQL fragments
            $this->apply_filter_by_element($element, $value, $condition);
        }
        
        // Log the final SQL fragments
    }

    /**
     * Element mapping system - returns SQL fragments instead of executing queries
     */
    private function apply_filter_by_element($element, $value, $condition) {
        $filter_methods = array(
            'subscriptions' => 'apply_subscription_filter',
            'subscription_status' => 'apply_subscription_status_filter',
            'order_date' => 'apply_order_date_filter',
            'next_payment_date' => 'apply_next_payment_date_filter',
            'payment_method' => 'apply_payment_methods_filter',
            'customer' => 'apply_customer_filter',
            'item_ordered' => 'apply_product_filter',
            'coupons' => 'apply_coupons_filter',
            'billing_address' => 'apply_billing_address_filter',
            'shipping_address' => 'apply_shipping_address_filter'
        );
        
        if (isset($filter_methods[$element])) {
            $method_name = $filter_methods[$element];
            $this->$method_name($value, $condition);
        } else {
        }
    }

    /**
     * Subscription Filter (Include/Exclude by ID or order number)
     */
    private function apply_subscription_filter($value, $condition) {
        // Handle different input formats
        if (is_array($value)) {
            $subscription_ids = $value;
        } elseif (is_string($value)) {
            // Split comma-separated string into array
            $subscription_ids = array_filter(array_map('trim', explode(',', $value)));
        } else {
            $subscription_ids = array($value);
        }
        
        // Remove empty values and ensure integers
        $subscription_ids = array_filter(array_map('intval', $subscription_ids));
        
        if (!empty($subscription_ids)) {
            $subscriptions_sql = self::prepare_sql_inarray($subscription_ids);
                    
            if ('include' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.{$this->get_subscription_id_field()} IN ($subscriptions_sql)";
            } elseif ('exclude' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.{$this->get_subscription_id_field()} NOT IN ($subscriptions_sql)";
            }
        }
    }

    /**
     * Subscription Status Filter
     */
    private function apply_subscription_status_filter($value, $condition) {
        if (empty($value) || !is_array($value)) {
            return;
        }
        
        $statuses_sql = self::prepare_sql_inarray($value);
        
        if (self::is_hpos_enabled()) {
            if ('include' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.status IN ($statuses_sql)";
            } elseif ('exclude' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.status NOT IN ($statuses_sql)";
            }
        } else {
            if ('include' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.post_status IN ($statuses_sql)";
            } elseif ('exclude' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.post_status NOT IN ($statuses_sql)";
            }
        }
    }

    /**
     * Order Date Filter (Default)
     */
    private function apply_order_date_filter($value, $condition) {
        if (empty($value)) {
            return;
        }
        
        global $wpdb;
        $date_field = self::is_hpos_enabled() ? 'date_created_gmt' : 'post_date';
        
        if (is_array($value) && isset($value['from']) && isset($value['to'])) {
            // Handle date range (Between)
            $from_date = apply_filters('wt_iew_alter_subscription_start_date', $value['from']);
            $to_date = apply_filters('wt_iew_alter_subscription_end_date', $value['to']);
            
            $from_date = $from_date . ' 00:00:00';
            $to_date = $to_date . ' 23:59:59.99';
            
            $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                "subscriptions.{$date_field} BETWEEN %s AND %s",
                $from_date,
                $to_date
            );
        } else {
            // Handle single date
            $date_value = $value;
            
            switch ($condition) {
                case 'greater_than':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "subscriptions.{$date_field} > %s",
                        $date_value . ' 23:59:59.99'
                    );
                    break;
                case 'less_than':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "subscriptions.{$date_field} < %s",
                        $date_value . ' 00:00:00'
                    );
                    break;
                case 'equals':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "DATE(subscriptions.{$date_field}) = %s",
                        $date_value
                    );
                    break;
                case 'not_equals':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "DATE(subscriptions.{$date_field}) != %s",
                        $date_value
                    );
                    break;
                case 'in_between':
                    // This is handled above for array values
                    break;
            }
        }
    }

    /**
     * Next Payment Date Filter
     */
    private function apply_next_payment_date_filter($value, $condition) {
        if (empty($value)) {
            return;
        }
        
        global $wpdb;
        $meta_key = '_schedule_next_payment';
        
        if (self::is_hpos_enabled()) {
            $this->sql_fragments['left_joins'][] = "LEFT JOIN {$wpdb->prefix}wc_orders_meta AS next_payment_meta ON subscriptions.id = next_payment_meta.order_id";
            
            switch ($condition) {
                case 'greater_than':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "next_payment_meta.meta_key = %s AND next_payment_meta.meta_value > %s",
                        $meta_key,
                        $value . ' 23:59:59.99'
                    );
                    break;
                case 'less_than':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "next_payment_meta.meta_key = %s AND next_payment_meta.meta_value < %s",
                        $meta_key,
                        $value . ' 00:00:00'
                    );
                    break;
                case 'equals':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "next_payment_meta.meta_key = %s AND DATE(next_payment_meta.meta_value) = %s",
                        $meta_key,
                        $value
                    );
                    break;
                case 'not_equals':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "next_payment_meta.meta_key = %s AND DATE(next_payment_meta.meta_value) != %s",
                        $meta_key,
                        $value
                    );
                    break;
                case 'in_between':
                    // Handle date range if needed
                    if (is_array($value) && isset($value['from']) && isset($value['to'])) {
                        $from_date = $value['from'] . ' 00:00:00';
                        $to_date = $value['to'] . ' 23:59:59.99';
                        $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                            "next_payment_meta.meta_key = %s AND next_payment_meta.meta_value BETWEEN %s AND %s",
                            $meta_key,
                            $from_date,
                            $to_date
                        );
                    }
                    break;
            }
        } else {
            switch ($condition) {
                case 'greater_than':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "subscriptions.ID IN (
                            SELECT post_id FROM {$wpdb->postmeta} 
                            WHERE meta_key = %s 
                            AND meta_value > %s
                        )",
                        $meta_key,
                        $value . ' 23:59:59.99'
                    );
                    break;
                case 'less_than':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "subscriptions.ID IN (
                            SELECT post_id FROM {$wpdb->postmeta} 
                            WHERE meta_key = %s 
                            AND meta_value < %s
                        )",
                        $meta_key,
                        $value . ' 00:00:00'
                    );
                    break;
                case 'equals':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "subscriptions.ID IN (
                            SELECT post_id FROM {$wpdb->postmeta} 
                            WHERE meta_key = %s 
                            AND meta_value = %s
                        )",
                        $meta_key,
                        $value
                    );
                    break;
                case 'not_equals':
                    $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                        "subscriptions.ID IN (
                            SELECT post_id FROM {$wpdb->postmeta} 
                            WHERE meta_key = %s 
                            AND meta_value != %s
                        )",
                        $meta_key,
                        $value
                    );
                    break;
                case 'in_between':
                    // Handle date range if needed
                    if (is_array($value) && isset($value['from']) && isset($value['to'])) {
                        $from_date = $value['from'] . ' 00:00:00';
                        $to_date = $value['to'] . ' 23:59:59.99';
                        $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                            "subscriptions.ID IN (
                                SELECT post_id FROM {$wpdb->postmeta} 
                                WHERE meta_key = %s 
                                AND meta_value BETWEEN %s AND %s
                            )",
                            $meta_key,
                            $from_date,
                            $to_date
                        );
                    }
                    break;
            }
        }
    }

    /**
     * Payment Methods Filter
     */
    private function apply_payment_methods_filter($value, $condition) {
        if (empty($value) || !is_array($value)) {
            return;
        }
        
        global $wpdb;
        $payment_methods_sql = self::prepare_sql_inarray($value);
        
        if (self::is_hpos_enabled()) {
            if ('include' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.payment_method IN ($payment_methods_sql)";
            } elseif ('exclude' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.payment_method NOT IN ($payment_methods_sql)";
            }
        } else {
            $meta_key = '_payment_method';
            if ('include' === $condition) {
                $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                    "subscriptions.ID IN (
                        SELECT post_id FROM {$wpdb->postmeta} 
                        WHERE meta_key = %s 
                        AND meta_value IN ($payment_methods_sql)
                    )",
                    $meta_key
                );
            } elseif ('exclude' === $condition) {
                $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                    "subscriptions.ID NOT IN (
                        SELECT post_id FROM {$wpdb->postmeta} 
                        WHERE meta_key = %s 
                        AND meta_value IN ($payment_methods_sql)
                    )",
                    $meta_key
                );
            }
        }
    }

    /**
     * Customer Filter
     */
    private function apply_customer_filter($value, $condition) {
        if (empty($value) || !is_array($value)) {
            return;
        }
        
        global $wpdb;
        $customer_ids = array_map('intval', $value);
        $customer_ids_sql = self::prepare_sql_inarray($customer_ids);
        
        if (self::is_hpos_enabled()) {
            if ('equals' === $condition) {
                $this->sql_fragments['base_conditions'][] = "subscriptions.customer_id IN ($customer_ids_sql)";
            } elseif ('not_equals' === $condition) {
                // Use NOT EXISTS to avoid NULL issues with NOT IN
                $this->sql_fragments['base_conditions'][] = "subscriptions.customer_id IS NOT NULL AND subscriptions.customer_id NOT IN ($customer_ids_sql)";
            }
        } else {
            $meta_key = '_customer_user';
            if ('equals' === $condition) {
                $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                    "subscriptions.ID IN (
                        SELECT post_id FROM {$wpdb->postmeta} 
                        WHERE meta_key = %s 
                        AND meta_value IN ($customer_ids_sql)
                    )",
                    $meta_key
                );
            } elseif ('not_equals' === $condition) {
                // Use NOT EXISTS to avoid NULL issues with NOT IN
                $this->sql_fragments['base_conditions'][] = $wpdb->prepare(
                    "NOT EXISTS (
                        SELECT 1 FROM {$wpdb->postmeta} 
                        WHERE post_id = subscriptions.ID 
                        AND meta_key = %s 
                        AND meta_value IN ($customer_ids_sql)
                    )",
                    $meta_key
                );
            }
        }
    }

    /**
     * Product Filter
     */
    private function apply_product_filter($value, $condition) {
        // Handle both array and string inputs
        if (is_string($value)) {
            $value = array_filter(array_map('trim', explode(',', $value)));
        }
        
        if (empty($value) || !is_array($value)) {
            return;
        }
                
        global $wpdb;
        
        // Get subscription IDs for the specified products using WooCommerce functions
        $prod_subscription_ids = array();
        
        if (function_exists('wcs_get_subscriptions_for_product')) {
            $prod_subscription_ids = wcs_get_subscriptions_for_product($value);
        } elseif (function_exists('hf_get_subscriptions_for_product')) {
            $prod_subscription_ids = hf_get_subscriptions_for_product($value);
        } else {
        }
        
        // Convert WooCommerce subscription objects to IDs if needed
        if (!empty($prod_subscription_ids)) {
            $subscription_ids = array();
            foreach ($prod_subscription_ids as $subscription) {
                if (is_object($subscription) && method_exists($subscription, 'get_id')) {
                    $subscription_ids[] = $subscription->get_id();
                } elseif (is_numeric($subscription)) {
                    $subscription_ids[] = $subscription;
                }
            }
            
            if (!empty($subscription_ids)) {
                $subscription_ids_sql = self::prepare_sql_inarray($subscription_ids);
                
                if ('include' === $condition) {
                    $this->sql_fragments['base_conditions'][] = "subscriptions.{$this->get_subscription_id_field()} IN ($subscription_ids_sql)";
                } elseif ('exclude' === $condition) {
                    $this->sql_fragments['base_conditions'][] = "subscriptions.{$this->get_subscription_id_field()} NOT IN ($subscription_ids_sql)";
                }
                
            }
        } 
    }

    /**
     * Coupons Filter
     */
    private function apply_coupons_filter($value, $condition) {
        if (empty($value) || !is_array($value)) {
            return;
        }
        
        global $wpdb;
        $coupons_sql = self::prepare_sql_inarray($value);
        
        if ('equals' === $condition) {
            $subquery = "SELECT DISTINCT order_coupons.order_id 
                         FROM {$wpdb->prefix}woocommerce_order_items as order_coupons
                         WHERE order_coupons.order_item_type = 'coupon' 
                         AND order_coupons.order_item_name IN ({$coupons_sql})";
            $this->sql_fragments['base_conditions'][] = "subscriptions.{$this->get_subscription_id_field()} IN ({$subquery})";
        } else {
            // Use NOT EXISTS instead of NOT IN to avoid NULL issues
            $subquery = "NOT EXISTS (
                            SELECT 1 
                            FROM {$wpdb->prefix}woocommerce_order_items as order_coupons
                            WHERE order_coupons.order_id = subscriptions.{$this->get_subscription_id_field()}
                            AND order_coupons.order_item_type = 'coupon' 
                            AND order_coupons.order_item_name IN ({$coupons_sql})
                        )";
            $this->sql_fragments['base_conditions'][] = $subquery;
        }
    }

    /**
     * Shared method for address filtering (billing/shipping, country/state)
     */
    private function apply_address_filter($value, $condition, $address_type) {
        global $wpdb;
        
        // Validate input structure
        if (!is_array($value) || empty($value['type']) || empty($value['value'])) {
            return;
        }
        
        $field_type = $value['type'];
        $field_value = $value['value'];
        
        // Validate field type
        if (!in_array($field_type, array('country', 'state'))) {
            return;
        }
        
        // Determine operator and prepare search value
        $is_multiple = is_array($field_value) && count($field_value) > 1;
        
        if ($is_multiple) {
            $operator = ('include' === $condition) ? 'IN' : 'NOT IN';
            $search_value = '(' . self::prepare_sql_inarray($field_value) . ')';
        } else {
            $operator = ( 'include' === $condition ) ? '=' : '!=';
            $single_value = is_array($field_value) ? $field_value[0] : $field_value;
            $search_value = "'" . $wpdb->esc_like($single_value) . "'";
        }
        
        // Build table alias and join
        $table_alias = "order_address_{$address_type}_{$field_type}";
        $join_sql = "LEFT JOIN {$wpdb->prefix}wc_order_addresses AS {$table_alias} ON {$table_alias}.order_id = subscriptions.id";
        
        // Build WHERE condition
        if ( self::is_hpos_enabled() ) {
            // HPOS: Use wc_order_addresses table
            $where_condition = $wpdb->prepare(
                "({$table_alias}.address_type = %s AND {$table_alias}.{$field_type} {$operator} {$search_value})",
                $address_type
            );
        } else {
            // Legacy: Use postmeta table
            $meta_key = "_{$address_type}_{$field_type}";
            $table_alias = "ordermeta_{$address_type}_{$field_type}";
            $join_sql = "LEFT JOIN {$wpdb->postmeta} AS {$table_alias} ON {$table_alias}.post_id = subscriptions.ID";
            $where_condition = $wpdb->prepare(
                "({$table_alias}.meta_key = %s AND {$table_alias}.meta_value {$operator} {$search_value})",
                $meta_key
            );
        }
        
        // Add to SQL fragments - use base_conditions instead of where_conditions
        $this->sql_fragments['left_joins'][] = $join_sql;
        $this->sql_fragments['base_conditions'][] = $where_condition;
    }

    /**
     * Billing Address Filter - Filters by billing country or state
     */
    private function apply_billing_address_filter($value, $condition) {
        $this->apply_address_filter($value, $condition, 'billing');
    }

    /**
     * Shipping Address Filter - Filters by shipping country or state
     */
    private function apply_shipping_address_filter($value, $condition) {
        $this->apply_address_filter($value, $condition, 'shipping');
    }

    /**
     * Add exclude already exported filter
     */
    private function add_exclude_already_exported_filter() {
        global $wpdb;
        
        if (self::is_hpos_enabled()) {
            // For HPOS, we need to check if the subscription doesn't have the exported status meta
            // We'll use a NOT EXISTS subquery to be more precise
            $exclude_condition = "NOT EXISTS (
                SELECT 1 FROM {$wpdb->prefix}wc_orders_meta AS check_exported 
                WHERE check_exported.order_id = subscriptions.id 
                AND check_exported.meta_key = 'wt_subscription_exported_status' 
                AND check_exported.meta_value = '1'
            )";
            $this->sql_fragments['base_conditions'][] = $exclude_condition;
        } else {
            $exclude_condition = "subscriptions.ID NOT IN (
                SELECT post_id FROM {$wpdb->postmeta} 
                WHERE meta_key = 'wt_subscription_exported_status' 
                AND meta_value = '1'
            )";
            $this->sql_fragments['base_conditions'][] = $exclude_condition;
        }
    }

    /**
     * Validate filter value based on element type
     */
    private function validate_filter_value($element, $value, $condition) {
        if (empty($value)) {
            return false;
        }
        
        switch ($element) {
            case 'subscriptions':
            case 'subscription_status':
            case 'payment_method':
            case 'customer':
            case 'coupons':
            case 'item_ordered':
            case 'order_date':
            case 'next_payment_date':
                // Allow both array (date range) and string (single date) values
                return is_array($value) || is_string($value);
            case 'billing_address':
            case 'shipping_address':
                return is_string($value) || (is_array($value) && isset($value['type']));
            default:
                return false;
        }
    }

    /**
     * Build the final SQL query from collected fragments
     */
    private function build_final_sql_query($export_limit, $current_offset) {
        global $wpdb;
        
        $subscription_plugin = 'WC';
        if(class_exists('HF_Subscription')){
            $subscription_plugin = 'HF';
        }
        
        if (self::is_hpos_enabled()) {
            // HPOS enabled - use wc_orders table
            $post_type = ($subscription_plugin == 'WC') ? 'shop_subscription' : 'hf_shop_subscription';
            
            $query = "SELECT DISTINCT subscriptions.id FROM {$wpdb->prefix}wc_orders AS subscriptions";
            
            // Add left joins
            if (!empty($this->sql_fragments['left_joins'])) {
                $query .= ' ' . implode(' ', array_unique($this->sql_fragments['left_joins']));
            }
            
            $query .= " WHERE subscriptions.type = '{$post_type}'";
            
            // Add base conditions
            if (!empty($this->sql_fragments['base_conditions'])) {
                $query .= ' AND ' . implode(' AND ', $this->sql_fragments['base_conditions']);
            }
            
            $query .= " ORDER BY subscriptions.id DESC";
            
        } else {
            // Legacy system - use posts table
            $post_type = ($subscription_plugin == 'WC') ? 'shop_subscription' : 'hf_shop_subscription';
            
            $query = "SELECT DISTINCT subscriptions.ID FROM {$wpdb->posts} AS subscriptions";
            
            // Add left joins
            if (!empty($this->sql_fragments['left_joins'])) {
                $query .= ' ' . implode(' ', array_unique($this->sql_fragments['left_joins']));
            }
            
            $query .= " WHERE subscriptions.post_type = '{$post_type}'";
            
            // Add base conditions
            if (!empty($this->sql_fragments['base_conditions'])) {
                $query .= ' AND ' . implode(' AND ', $this->sql_fragments['base_conditions']);
            }
            
            $query .= " ORDER BY subscriptions.ID DESC";
        }
   
        return $query;
    }

    /**
     * Get subscription meta value using proper getters for internal meta keys
     * 
     * Filters available:
     * - wt_iew_subscription_use_direct_meta: Use direct meta access for all keys
     * - wt_iew_subscription_use_direct_meta_{$meta_key}: Use direct meta access for specific key
     * 
     * Example usage:
     * add_filter('wt_iew_subscription_use_direct_meta', '__return_true'); // Use direct meta for all
     * add_filter('wt_iew_subscription_use_direct_meta__cancelled_email_sent', '__return_true'); // Use direct meta for specific key
     * 
     * @since 1.3.1
     */
    private function get_subscription_meta_value($subscription, $meta_key) {
        // Allow filtering to use direct meta access instead of getter methods
        // This bypasses getter methods and uses direct meta access (may trigger warnings)
        $use_direct_meta = apply_filters('wt_iew_subscription_use_direct_meta', false, $meta_key, $subscription);
        $use_direct_meta_for_key = apply_filters('wt_iew_subscription_use_direct_meta_' . $meta_key, false, $subscription);
        
        if ($use_direct_meta || $use_direct_meta_for_key) {
            // Use direct meta access (may trigger warnings but gets all data)
            return maybe_serialize($subscription->get_meta($meta_key, true));
        }
        
        // Handle internal meta keys with proper getters
        switch ($meta_key) {
            case '_requires_manual_renewal':
                return $subscription->is_manual() ? 'yes' : 'no';
            case '_customer_ip_address':
                return $subscription->get_customer_ip_address();
            case '_customer_user_agent':
                return $subscription->get_customer_user_agent();
            case '_order_key':
                return $subscription->get_order_key();
            case '_prices_include_tax':
                return $subscription->get_prices_include_tax() ? 'yes' : 'no';
            case '_suspension_count':
                $suspension_count = '';
                if ( method_exists($subscription, 'get_suspension_count') ) {
                // Use the proper WooCommerce Subscriptions method
                    $suspension_count = $subscription->get_suspension_count();
                } else {
                    $suspension_count = get_post_meta($subscription->get_id(), '_suspension_count', true);
                }
                return $suspension_count;
            case '_trial_period':
                $trial_period = '';
                if ( method_exists($subscription, 'get_trial_period') ) {
                    // Use the proper WooCommerce Subscriptions method
                    $trial_period = $subscription->get_trial_period();
                } else {
                    $trial_period = get_post_meta($subscription->get_id(), '_trial_period', true);
                }
                return $trial_period;
            case '_cancelled_email_sent':
                $cancelled_email_sent = '';
                if ( method_exists($subscription, 'get_cancelled_email_sent') ) {
                // Use the proper WooCommerce Subscriptions method
                    $cancelled_email_sent = $subscription->get_cancelled_email_sent();
                } else {
                    $cancelled_email_sent = get_post_meta($subscription->get_id(), '_cancelled_email_sent', true);
                }
                return $cancelled_email_sent;

                default:
                // For other meta keys, use the generic method
                return maybe_serialize($subscription->get_meta($meta_key, true));
        }
    }

    /**
     * Get subscription ID field name based on HPOS status
     */
    private function get_subscription_id_field() {
        return self::is_hpos_enabled() ? 'id' : 'ID';
    }

    /**
     * Get subscription status field name based on HPOS status
     */
    private function get_subscription_status_field() {
        return self::is_hpos_enabled() ? 'status' : 'post_status';
    }

    /**
     * Generate subscription row data
     */
    private function generate_subscription_row_data($subscription_id, $form_data) {
        $subscription = self::hf_get_subscription($subscription_id);
        
        if (!$subscription) {
            return array();
        }
        
        $subscription_plugin = 'WC';
        if(class_exists('HF_Subscription')){
            $subscription_plugin = 'HF';
        }
        
        if($subscription_plugin == 'WC'){
            return $this->get_subscriptions_csv_row($subscription);
        } else {
            return $this->get_wt_subscriptions_csv_row($subscription);
        }
    }

    /**
     * Prepare SQL IN array for safe query building
     */
    private static function prepare_sql_inarray($array) {
        if (empty($array)) {
            return "''";
        }
        
        $escaped_values = array();
        foreach ($array as $value) {
            if (is_numeric($value)) {
                $escaped_values[] = intval($value);
            } else {
                $escaped_values[] = "'" . esc_sql($value) . "'";
            }
        }
        
        return implode(',', $escaped_values);
    }


    /**
     * Prepare data that will be exported.
     */
    public function prepare_data_to_export_old($form_data, $batch_offset) {
                
        $export_statuses = !empty($form_data['filter_form_data']['wt_iew_statuses']) ? $form_data['filter_form_data']['wt_iew_statuses'] : 'any';        
        $start_date = !empty($form_data['filter_form_data']['wt_iew_start_date']) ? $form_data['filter_form_data']['wt_iew_start_date'] : '';//date('Y-m-d 00:00', 0);
        $end_date = !empty($form_data['filter_form_data']['wt_iew_end_date']) ? $form_data['filter_form_data']['wt_iew_end_date']. ' 23:59:59.99' : '';//date('Y-m-d 23:59', current_time('timestamp'));
        $next_pay_date = !empty($form_data['filter_form_data']['wt_iew_next_pay_date']) ? $form_data['filter_form_data']['wt_iew_next_pay_date'] : '';
        
        $payment_methods = !empty($form_data['filter_form_data']['wt_iew_payment_methods']) ? wc_clean($form_data['filter_form_data']['wt_iew_payment_methods']) : array();        
        $email = !empty($form_data['filter_form_data']['wt_iew_email']) ? wc_clean($form_data['filter_form_data']['wt_iew_email']) : array();
        $products = !empty($form_data['filter_form_data']['wt_iew_products']) ? wc_clean($form_data['filter_form_data']['wt_iew_products']): array();

        $coupons = !empty($form_data['filter_form_data']['wt_iew_coupons']) ?  array_filter(explode(',', strtolower($form_data['filter_form_data']['wt_iew_coupons'])),'trim'): array();
                               
        $export_sortby = !empty($form_data['filter_form_data']['wt_iew_sort_columns']) ? implode(' ', $form_data['filter_form_data']['wt_iew_sort_columns']) : 'ID'; // get_post accept spaced string
        $export_sort_order = !empty($form_data['filter_form_data']['wt_iew_order_by']) ? $form_data['filter_form_data']['wt_iew_order_by'] : 'ASC';
        
        $export_limit = !empty($form_data['filter_form_data']['wt_iew_limit']) ? intval($form_data['filter_form_data']['wt_iew_limit']) : 999999999; //user limit
        $current_offset = !empty($form_data['filter_form_data']['wt_iew_offset']) ? intval($form_data['filter_form_data']['wt_iew_offset']) : 0; //user offset
        $batch_count = !empty($form_data['advanced_form_data']['wt_iew_batch_count']) ? $form_data['advanced_form_data']['wt_iew_batch_count'] : Wt_Import_Export_For_Woo_Common_Helper::get_advanced_settings('default_export_batch');          

        $exclude_already_exported = (!empty($form_data['advanced_form_data']['wt_iew_exclude_already_exported']) && $form_data['advanced_form_data']['wt_iew_exclude_already_exported'] == 'Yes') ? true : false;
        
        $this->export_to_separate_columns = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate_columns']) && $form_data['advanced_form_data']['wt_iew_export_to_separate_columns'] == 'Yes') ? true : false;                       
        if(!$this->export_to_separate_columns){
            $this->export_to_separate_columns = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate']) && $form_data['advanced_form_data']['wt_iew_export_to_separate'] == 'column') ? true : false;               
        }

        $this->export_to_separate_rows = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate_rows']) && $form_data['advanced_form_data']['wt_iew_export_to_separate_rows'] == 'Yes') ? true : false;               
        if(!$this->export_to_separate_rows){
            $this->export_to_separate_rows = (!empty($form_data['advanced_form_data']['wt_iew_export_to_separate']) && $form_data['advanced_form_data']['wt_iew_export_to_separate'] == 'row') ? true : false;               
        }
        $this->exclude_line_items = (!empty($form_data['advanced_form_data']['wt_iew_exclude_line_items']) && $form_data['advanced_form_data']['wt_iew_exclude_line_items'] == 'Yes') ? true : false;               
        
        
        
        $subscription_plugin = 'WC';
        if(class_exists('HF_Subscription')){
            $subscription_plugin = 'HF';
        }

        $real_offset = ($current_offset + $batch_offset);

        if($batch_count<=$export_limit)
        {
            if(($batch_offset+$batch_count)>$export_limit) //last offset
            {
                $limit=$export_limit-$batch_offset;
            }else
            {
                $limit=$batch_count;
            }
        }else
        {
            $limit=$export_limit;
        }
      


        $data_array = array();
        if ($batch_offset < $export_limit)
        {
            if(self::is_hpos_enabled()){
                global $wpdb;
                $format = array();
                $post_type = ($subscription_plugin == 'WC') ? 'shop_subscription' : 'hf_shop_subscription';
                $query_args="SELECT DISTINCT po.id FROM {$wpdb->prefix}wc_orders as po LEFT JOIN {$wpdb->prefix}wc_orders_meta AS pm ON po.id = pm.order_id WHERE type = %s"; 
                $format[]=$post_type;
                if( $end_date ){
                    $query_args.= "AND date_created_gmt<=%s";
                    $format[] = $end_date;
                }
                if( $start_date ){
                    $query_args.= "AND  date_created_gmt>=%s ";
                    $format[] = $start_date;
                }
                if (!empty($export_statuses)) {
                    $statuses = $export_statuses;
                    if (!empty($statuses) && is_array($statuses)) {
                        $value=array();
                        foreach($statuses as $status){
                           
                            $values[] = "'$status'";
                        }
                        $statuses = implode(",", $values);
                        if (!in_array($statuses, array('any', 'trash'))) {
                            $query_args .= " AND po.status IN ($statuses)";
                        }
                    }
                }  
                
                if (!empty($payment_methods)) {
                    $query_args .= " AND (";
                    $first_payment_method = true;
                    foreach ($payment_methods as $key => $value) {
                        $value = strtolower($value);
                        if (!$first_payment_method) {
                            $query_args .= " OR";
                        }
                        $query_args .= " po.payment_method = %s";
                        $format[] = $value;
                        $first_payment_method = false;
                    }
                    $query_args .= ")";
                }
                if (!empty($next_pay_date)) {
                    $query_args .= " AND (pm.meta_key = '_schedule_next_payment' AND pm.meta_value LIKE %s)";
                    $format[] = '%' . $next_pay_date . '%';
                }
                $query_args .= " ORDER BY id DESC";
                
            }else{
                $query_args = array(
                    'fields' => 'ids',
                    'post_type' =>  ($subscription_plugin == 'WC') ? 'shop_subscription' : 'hf_shop_subscription',
                    'post_status' => 'any',                
                    'orderby' => $export_sortby,
                    'order' => $export_sort_order,                
                );
                
                if( $end_date || $start_date ){
                    $query_args['date_query'] =  array(
                        array(
                            'before' => $end_date,
                            'after' => $start_date,
                            'inclusive' => true,
                        ),
                    );
                }
                
                if (!empty($export_statuses)) {
                    $statuses = $export_statuses;
                    if (!empty($statuses) && is_array($statuses)) {
                        $query_args['post_status'] = implode(',', $statuses);
                        if (!in_array($query_args['post_status'], array('any', 'trash'))) {
                            $query_args['post_status'] = self::hf_sanitize_subscription_status_keys($query_args['post_status']);
                        }
                    }
                }            
                
                if (!empty($payment_methods)) {
                    $meta_query = array('relation' => 'OR');
                    foreach ($payment_methods as $key => $value) {
                        $value = strtolower($value);
                        $meta_query[] = array(
                            'key' => '_payment_method',
                            'value' => $value,
                        );
                    }
                    $query_args['meta_query'][] =$meta_query;
                }
                
                if (!empty($next_pay_date)) {                
                   $query_args['meta_query'][]  = array(
                            'key' => '_schedule_next_payment',
                            'value' => $next_pay_date,
                            'compare' => 'LIKE'
                        );
                      
                }
            }
         
                                    
            $query_args = apply_filters('woocommerce_get_subscriptions_query_args', $query_args);


            $subscription_order_ids = $prod_subscription_ids = $coupon_subscription_ids = 0;
            
            if (!empty($email)) {

                global $wpdb;				
				$subscription_type = 'shop_subscription';
                if( class_exists( 'HF_Subscription')){              
					$subscription_type = 'hf_shop_subscription';
				}
                $email[] = $subscription_type;

				if ( self::is_hpos_enabled() ) {
					$subscription_order_ids = $wpdb->get_col( $wpdb->prepare( "SELECT id FROM {$wpdb->prefix}wc_orders WHERE customer_id IN (" . implode( ',', array_fill( 0, count( $email ) - 1, '%d' ) ) . ') AND type = %s ', $email ) );
				} else {
					$subscription_order_ids = $wpdb->get_col( $wpdb->prepare( "SELECT p.ID FROM {$wpdb->prefix}posts AS p INNER JOIN {$wpdb->prefix}postmeta AS pm ON p.ID = pm.post_id  WHERE pm.meta_key = '_customer_user' AND pm.meta_value IN (" . implode( ',', array_fill( 0, count( $email ) - 1, '%d' ) ) . ') AND p.post_type = %s ', $email ) );
				}           
            } 
             
            if (!empty($products) ) {

				$prod_subscription_ids = array();
				if( function_exists('wcs_get_subscriptions_for_product')){
					$prod_subscription_ids = wcs_get_subscriptions_for_product($products);
				}else{
					if( function_exists('hf_get_subscriptions_for_product')){
						$prod_subscription_ids = hf_get_subscriptions_for_product($products);
					}
				}
            }

            if (!empty($coupons)) {
                $coupon_subscription_ids = array();
				$subscription_type = 'shop_subscription';
                if( class_exists( 'HF_Subscription')){              
					$subscription_type = 'hf_shop_subscription';;
				}
				$coupon_subscription_ids = self::wt_get_subscription_of_coupons($coupons, $subscription_type);
            }         
            
            /**
            *   taking total records
            */
            $total_records=0;
            if($batch_offset==0) //first batch
            {
                $total_item_args = $query_args;
                if(self::is_hpos_enabled()){
                    $total_item_args.=" LIMIT %d OFFSET %d";
                    $format[] = $export_limit;
                    $format[] = $current_offset;
                    $subscription_post_ids = $wpdb->get_col($wpdb->prepare( $total_item_args, $format));
                }else{
                    $total_item_args['posts_per_page'] = $export_limit; //user given limit
                    $total_item_args['offset'] = $current_offset; //user given offset                
                    $subscription_post_ids = get_posts($total_item_args);    
                }
                
                if (!empty($email)) {
                    if(!empty($subscription_order_ids)){
						$subscription_post_ids = array_intersect($subscription_order_ids, $subscription_post_ids);  
					}
                }
                if (!empty($products) ) { 
					if(!empty($prod_subscription_ids)){
						$subscription_post_ids = array_intersect($prod_subscription_ids, $subscription_post_ids);
					}
                }
                if (!empty($coupons)) {
					if(!empty($coupon_subscription_ids)){
						$subscription_post_ids = array_intersect($coupon_subscription_ids, $subscription_post_ids);
					}
                }
              
                                                
                foreach ($subscription_post_ids as $key => $subscription_id) {
                    if (!$subscription_id )
                        unset($subscription_post_ids[$key]);

                    if ($exclude_already_exported) {
                        if(get_post_meta($subscription_id, 'wt_subscription_exported_status', 1))
                            unset($subscription_post_ids[$key]);
                    }                    
                }
                                
                
                $total_records = count($subscription_post_ids);   
                
                $this->line_items_max_count = $this->get_max_line_items($subscription_post_ids);
                                                
                add_option('wt_subscription_order_line_items_max_count',$this->line_items_max_count);  
                                
            }
            
            if(empty($this->line_items_max_count)){
                $this->line_items_max_count = get_option('wt_subscription_order_line_items_max_count');
            }
            if(self::is_hpos_enabled()){
                $query_args.=" LIMIT %d OFFSET %d";
                if($batch_offset==0){
                    $format= array_slice($format,0,count($format)-2);
                }
				$format[]=  $limit;
				$format[]=  $real_offset;
                $subscription_post_ids = $wpdb->get_col($wpdb->prepare( $query_args, $format));
            }else{
                $query_args['offset'] = $real_offset;
                $query_args['posts_per_page'] = $limit;
                    
                $subscription_post_ids = get_posts($query_args);  
            }
            
         


            if (!empty($email)) {
                if(!empty($subscription_order_ids)){
                    $subscription_post_ids = array_intersect($subscription_order_ids, $subscription_post_ids);  
                }
            }
            if (!empty($products) ) { 
                if(!empty($prod_subscription_ids)){
                    $subscription_post_ids = array_intersect($prod_subscription_ids, $subscription_post_ids);
                }
            }
            if (!empty($coupons)) {
                if(!empty($coupon_subscription_ids)){
                    $subscription_post_ids = array_intersect($coupon_subscription_ids, $subscription_post_ids);
                }
            }
            
          
            $subscriptions = array();
            foreach ($subscription_post_ids as $subscription_id) {
                if (!$subscription_id )
                    break; 
                
                if ($exclude_already_exported) {
                    if(self::is_hpos_enabled()){
                        $sql = "SELECT meta_value from {$wpdb->prefix}wc_orders_meta WHERE order_id = $subscription_id AND meta_key = 'wt_subscription_exported_status'";
                        $result=$wpdb->get_col( $sql);  
                        if(empty($result)){
                            $result=false;
                        }

                    }else{
                        $result = get_post_meta($subscription_id, 'wt_subscription_exported_status', 1);
                    }
                    if($result)
                        break;  
                }

                $subscriptions[]  = self::hf_get_subscription($subscription_id);
            }
          
            $subscriptions = apply_filters('hf_retrieved_subscriptions', $subscriptions);

            if($subscription_plugin == 'WC'){
            // Loop orders
                foreach ($subscriptions as $subscription) {
                   
                    $data_array[] = $this->get_subscriptions_csv_row($subscription);
                    
                    // updating records with expoted status 
                    update_post_meta($subscription->get_id(), 'wt_subscription_exported_status', TRUE);
                }
            }else{
                // Loop orders
                foreach ($subscriptions as $subscription) {

                    $data_array[] = $this->get_wt_subscriptions_csv_row($subscription);
                    
                    // updating records with expoted status 
                    update_post_meta($subscription->get_id(), 'wt_subscription_exported_status', TRUE);
                }
            }

            if($this->export_to_separate_rows){
                $data_array = $this->wt_ier_alter_order_data_befor_export_for_separate_row($data_array);
            }
            
            $data_array = apply_filters('wt_ier_alter_subscriptions_data_befor_export', $data_array);  
            
            
            $return['total'] = $total_records; 
            $return['data'] = $data_array;
			if( 0 == $batch_offset && 0 == $total_records ){
            	$return['no_post'] = __( 'Nothing to export under the selected criteria. Please check and try adjusting the filters.' );
            }
            return $return;
        }
    }  
    
   

    public static function hf_get_subscription($subscription) {
        if (is_object($subscription) && self::hf_is_subscription($subscription)) {
            $subscription = $subscription->id;
        }
        $subscription_plugin = 'WC';
        if(class_exists('HF_Subscription')){
            $subscription_plugin = 'HF';
        }
        if($subscription_plugin == 'WC'){
        if (!class_exists('WC_Subscription')):
            require WP_PLUGIN_DIR.'/woocommerce-subscriptions/wcs-functions.php';
            require WP_PLUGIN_DIR.'/woocommerce-subscriptions/includes/class-wc-subscription.php';
        endif;
        $subscription = new WC_Subscription($subscription);
        }else{
        if (!class_exists('HF_Subscription')):
            require WP_PLUGIN_DIR.'/xa-woocommerce-subscriptions/includes/subscription-common-functions.php';
            require WP_PLUGIN_DIR.'/xa-woocommerce-subscriptions/includes/components/class-subscription.php';
        endif;
        $subscription = new HF_Subscription($subscription);
        }
        if (!self::hf_is_subscription($subscription)) {
            $subscription = false;
        }
        return apply_filters('hf_get_subscription', $subscription);
    }
    
    
    public static function hf_is_subscription($subscription) {
        if (is_object($subscription) && (is_a($subscription, 'WC_Subscription') || is_a($subscription, 'HF_Subscription'))) {
            $is_subscription = true;
        } elseif (is_numeric($subscription) && ('shop_subscription' == get_post_type($subscription) || 'hf_shop_subscription' == get_post_type($subscription))) {
            $is_subscription = true;
        } else {
            $is_subscription = false;
        }
        return apply_filters('hf_is_subscription', $is_subscription, $subscription);
    }
    
    
    public function get_subscriptions_csv_row($subscription) {
                        
        //$csv_columns = $this->parent_module->get_selected_column_names();          
        $csv_columns = $this->prepare_header();
        
        
//        if (empty($export_columns)) {
//            $export_columns = $csv_columns;
//        }
        $fee_total = $fee_tax_total = 0;
        $fee_items = $shipping_items = array();
        if (0 != sizeof(array_intersect(array_keys($csv_columns), array('fee_total', 'fee_tax_total', 'fee_items')))) {
            foreach ($subscription->get_fees() as $fee_id => $fee) {
                $fee_items[] = implode('|', array(
                    'name:' . html_entity_decode($fee['name'], ENT_NOQUOTES, 'UTF-8'),
                    'total:' . wc_format_decimal($fee['line_total'], 2),
                    'tax:' . wc_format_decimal($fee['line_tax'], 2),
                    'tax_data:' . $fee['line_tax_data'],
                    'tax_class:' . $fee['tax_class'],
                ));
                $fee_total += (float) $fee['line_total'];
                $fee_tax_total += (float)$fee['line_tax'];
            }
        }
        
        $line_items_shipping = $subscription->get_items('shipping');
        foreach ($line_items_shipping as $item_id => $item) {
            if (is_object($item)) {
                if ($meta_data = $item->get_formatted_meta_data('')) :
                    foreach ($meta_data as $meta_id => $meta) :
                        if (in_array($meta->key, $line_items_shipping)) {
                            continue;
                        }
                        // html entity decode is not working preoperly
                        $shipping_items[] = implode('|', array('item:' . wp_kses_post($meta->display_key), 'value:' . str_replace('&times;', 'X', strip_tags($meta->display_value))));
                    endforeach;
                endif;
            }
        }

        if (!function_exists('get_user_by')) {
            require ABSPATH . 'wp-includes/pluggable.php';
        }

        $version = WC()->version;

        $user_values = get_user_by('ID',( version_compare( $version, '2.7', '<' ) ) ? $subscription->customer_user : $subscription->get_customer_id());
        
        // Preparing data for export
        foreach ($csv_columns as $header_key => $_) {
            switch ($header_key) {
                case 'subscription_id':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->id : $subscription->get_id();
                    break;
                case 'subscription_status':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->post_status : $subscription->get_status();
                    break;
                case 'customer_id':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->customer_user : $subscription->get_customer_id();
                    break;
                case 'customer_username':
                    $value = is_object($user_values) ? $user_values->user_login : '';
                    break;
                case 'customer_email':
                    $value = is_object($user_values) ? $user_values->user_email : '';
                    break;
                case 'fee_total':
                case 'fee_tax_total':
                    $value = ${$header_key};
                    break;
                case 'order_shipping_tax':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? ( empty( $subscription->{$header_key} ) ? 0 : $subscription->{$header_key} ) : $subscription->get_shipping_tax();
                    break;
                case 'order_total':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? ( empty( $subscription->{$header_key} ) ? 0 : $subscription->{$header_key} ) : $subscription->get_total();
                    break;
                case 'order_tax':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? ( empty( $subscription->{$header_key} ) ? 0 : $subscription->{$header_key} ) : $subscription->get_total_tax();
                    break;
                case 'order_shipping':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? ( empty( $subscription->{$header_key} ) ? 0 : $subscription->{$header_key} ) : $subscription->get_total_shipping();
                    break;
                case 'cart_discount_tax':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? ( empty( $subscription->{$header_key} ) ? 0 : $subscription->{$header_key} ) : $subscription->get_discount_tax();
                    break;
                case 'cart_discount':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? ( empty( $subscription->{$header_key} ) ? 0 : $subscription->{$header_key} ) : $subscription->get_total_discount();
                    break;
                case 'date_created':
                case 'trial_end_date':
                case 'next_payment_date':
                case 'last_order_date_created':
                case 'end_date':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_date($header_key);
                    break;
                case 'billing_period':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_period();
                    break;
                case 'billing_interval':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_interval();
                    break;
                case 'payment_method':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_payment_method();
                    break;
                case 'payment_method_title':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_payment_method_title();
                    break;
                case 'billing_first_name':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_first_name();
                    break;
                case 'billing_last_name':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_last_name();
                    break;
                case 'billing_email':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_email();
                    break;
                case 'billing_phone':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_phone();
                    break;
                case 'billing_address_1':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_address_1();
                    break;
                case 'billing_address_2':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_address_2();
                    break;
                case 'billing_postcode':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_postcode();
                    break;
                case 'billing_city':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_city();
                    break;
                case 'billing_state':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_state();
                    break;
                case 'billing_country':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_country();
                    break;
                case 'billing_company':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_billing_company();
                    break;
                case 'shipping_first_name':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_first_name();
                    break;
                case 'shipping_last_name':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_last_name();
                    break;
                case 'shipping_address_1':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_address_1();
                    break;
                case 'shipping_address_2':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_address_2();
                    break;
                case 'shipping_postcode':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_postcode();
                    break;
                case 'shipping_city':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_city();
                    break;
                case 'shipping_state':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_state();
                    break;
                case 'shipping_country':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_country();
                    break;
                case 'shipping_company':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_shipping_company();
                    break;
                case 'shipping_phone':
                    $value = (version_compare(WC_VERSION, '5.6', '<')) ? '' : $subscription->get_shipping_phone();
                    break;                
                case 'customer_note':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_customer_note();
                    break;
                case 'order_currency':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_currency();
                    break;
                case 'post_parent':
                    if (!empty($subscription->get_parent() ))
                        $value = $subscription->get_parent_id();
                    else
                        $value = 0;
                    break;
                case 'order_notes':
                    $order_notes = implode('||', (defined('WC_VERSION') && (WC_VERSION >= 3.2)) ? self::get_order_notes_new($subscription) : self::get_order_notes($subscription));
                    
//                    remove_filter('comments_clauses', array('WC_Comments', 'exclude_order_comments'));
//                    $notes = get_comments(array('post_id' => (version_compare( $version, '2.7', '<' ) ) ? $subscription->id : $subscription->get_id(), 'approve' => 'approve', 'type' => 'order_note'));
//                    add_filter('comments_clauses', array('WC_Comments', 'exclude_order_comments'));
//                    $order_notes = array();
//                    foreach ($notes as $note) {
//                        $order_notes[] = str_replace(array("\r", "\n"), ' ', $note->comment_content);
//                    }
//                    if (!empty($order_notes)) {
//                        $value = implode(';', $order_notes);
//                    } else {
//                        $value = '';
//                    }
                    if (!empty($order_notes)) {
                        $value = $order_notes;
                    } else {
                        $value = '';
                    }
                                        
                    break;
                case 'renewal_orders':
                    $renewal_orders = $subscription->get_related_orders('ids', 'renewal');
                    if (!empty($renewal_orders)) {
                        $value = implode('|', $renewal_orders);
                    } else {
                        $value = '';
                    }
                    break; 
                    
                    case 'order_items':    
                    $line_items = array();
                    $order_items = array();
                    foreach ($subscription->get_items() as $item_id => $item) {
                        $product = $item->get_product();
                        if (!is_object($product)) {
                            $product = new WC_Product(0);
                        }

            //                        $product_id = self::hf_get_canonical_product_id($item);
                        $item_meta = self::get_order_line_item_meta($item_id);
                        $prod_type = ( version_compare( $version, '3.0.0', '<' ) ) ? $product->product_type : $product->get_type();
                        $line_item = array(
                            'product_id' => ( version_compare( $version, '2.7.0', '<' ) ) ? $product->id : (($prod_type == 'variable' || $prod_type == 'variation' || $prod_type == 'subscription_variation') ? $product->get_parent_id() : $product->get_id()),
                            'name' => html_entity_decode($item['name'], ENT_NOQUOTES, 'UTF-8'),
                            'sku' => $product->get_sku(),
                            'quantity' => $item['qty'],
                            'total' => wc_format_decimal($subscription->get_line_total($item), 2),
                            'sub_total' => wc_format_decimal($subscription->get_line_subtotal($item), 2),
                        );

                        // add line item tax
                        $line_tax_data = isset($item['line_tax_data']) ? $item['line_tax_data'] : array();
                        $tax_data = maybe_unserialize($line_tax_data);
                        $tax_detail = isset($tax_data['total']) ? wc_format_decimal(wc_round_tax_total(array_sum((array) $tax_data['total'])), 2) : '';
                        if ($tax_detail != '0.00' && !empty($tax_detail)) {
                            $line_item['tax'] = $tax_detail;
                        }
                        $line_tax_ser = maybe_serialize($line_tax_data);
                        $line_item['tax_data'] = $line_tax_ser;

                        foreach ($item_meta as $key => $value) {
                            switch ($key) {
                                case '_qty':
                                case '_variation_id':
                                case '_product_id':
                                case '_line_total':
                                case '_line_subtotal':
                                case '_tax_class':
                                case '_line_tax':
                                case '_line_tax_data':
                                case '_line_subtotal_tax':
                                    break;

                                default:
                                    if(is_object($value))
                                    $value = $value->meta_value;
                                    if (is_array($value))
                                        $value = implode(',', $value);
                                    $line_item[$key] = $value;
                                    break;
                            }
                        }

                        if ($prod_type === 'variable' || $prod_type === 'variation' || $prod_type === 'subscription_variation') {
                            $line_item['_variation_id'] = ( version_compare( $version, '2.7', '>' ) ) ? $product->get_id() : $product->variation_id;
                        }
                        
						$order_item_data_arr = array();
                        foreach ($line_item as $name => $value) {
                            $order_item_data_arr[$name] = $name . ':' . $value;
                        }
                        
                        $order_item_data = implode('|', $order_item_data_arr);

                        if ($line_item) {
                            $order_items[] = $order_item_data;
                            $line_items[] = $line_item;
                        }
                    }
                    if (!empty($order_items)) {
                        $value = implode('||', $order_items);
                    }                                        
                    break; 
                case 'coupon_items':
                    $coupon_items = array();
                    foreach ($subscription->get_items('coupon') as $_ => $coupon_item) {
                        $coupon_name = $coupon_item->get_name();
                        $coupon_discount = $coupon_item->get_discount();
                        $coupon = new WC_Coupon($coupon_name);
                        $coupon_post = get_post($coupon->get_id());
                        $coupon_items[] = implode('|', array(
                            'code:' . $coupon_name,
                            'description:' . ( is_object($coupon_post) ? $coupon_post->post_excerpt : '' ),
                            'amount:' . wc_format_decimal($coupon_discount, 2),
                        ));
                    }
                    if (!empty($coupon_items)) {
                        $value = implode(';', $coupon_items);
                    } else {
                        $value = '';
                    }
                    break;

                case 'payment_token_data';
                    $token_id = $subscription->get_payment_tokens();
                    if($token_id){
                        $t_id = maybe_unserialize($token_id);
                        if(isset($t_id[0])){
                            $data_store    = WC_Data_Store::load( 'payment-token' );
                            $token_data = $data_store->get_tokens( array('token_id'=>$t_id[0]) );
                            $token_meta_data = $data_store->get_metadata( $t_id[0] );
                            $token = array(
                                'token_data' => ( ! empty( $token_data ) ) ? $token_data : '',
                                'token_meta_data' => ( ! empty ( $token_meta_data ) ) ? $token_meta_data : '',
                                );
                        }
                    }
                    if(isset($token) && !empty ($token)){
                        $value =serialize($token);
                    }else{
                        $value = '';
                    }
                    break;
                case 'download_permissions':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? ($subscription->download_permissions_granted ? $subscription->download_permissions_granted : 0) :($subscription->is_download_permitted());
                    break;
                case 'shipping_method':
                    $shipping_lines = array();
                    foreach ($subscription->get_shipping_methods() as $shipping_item_id => $shipping_item) {
                        $shipping_lines[] = implode('|', array(
                            'method_id:' . $shipping_item['method_id'],
                            'method_title:' . $shipping_item['name'],
                            'total:' . wc_format_decimal($shipping_item['cost'], 2),
                            'total_tax:' . wc_format_decimal( $shipping_item['total_tax'], 2 ),
							'taxes:' . maybe_serialize($shipping_item['taxes']),
                            )
                        );
                    }
                    if (!empty($shipping_lines)) {
                        $value = implode(';', $shipping_lines);
                    } else {
                        $value = '';
                    }
                    break;
                case 'fee_items':
                    $value = implode(';', $fee_items);
                    break;
                case 'shipping_items':
                    $value = implode(';', $shipping_items);
                    break;
                case 'tax_items':
                    $tax_items = array();
                    foreach ( $subscription->get_taxes() as $tax_code => $tax ) {
						$tax_items[] = implode('|', array(
							'rate_id:' . $tax->get_rate_id(),
							'code:' . $tax->get_rate_code(),
							'total:' . wc_format_decimal($tax->get_tax_total(), 2),
							'label:' . $tax->get_label(),
							'tax_rate_compound:' . $tax->get_compound(),
							'shipping_tax_amount:' . $tax->get_shipping_tax_total(),
							'rate_percent:' . $tax->get_rate_percent(),
						));
                    }
                    if (!empty($tax_items)) {
                        $value = implode(';', $tax_items);
                    } else {
                        $value = '';
                    }
                    break;
                default :
                    if(strstr($header_key, 'meta:')){
                        if($header_key == 'meta:_payment_tokens'){
                            $value = $subscription->get_payment_tokens();
                        }else{
        
                            $meta_key = str_replace('meta:', '', $header_key);
                            //1.3.1 - use proper getters for internal meta keys
                            $value = $this->get_subscription_meta_value($subscription, $meta_key);
                        }
                    } else {
                        $value = '';
                    }
            }
            $csv_row[$header_key] = $value;
        }

//        $subscription_data = array();
//        foreach ($csv_columns as $header_key => $_) {
//            // Strict string comparison, as values like '0' are valid
//            $value = ( '' !== $csv_row[$header_key] ) ? $csv_row[$header_key] : '';
//            $subscription_data[$header_key] = $value;
//        }
        
        $subscription_data = array();
        foreach ($csv_columns as $key => $value) {
            if (!$csv_row || array_key_exists($key, $csv_row)) {
                $subscription_data[$key] = $csv_row[$key];
            } 
        }
//        return apply_filters('hf_alter_subscription_data', $data, $export_columns, $csv_columns);
//        return apply_filters('hf_alter_subscription_data', $data, $csv_columns, $csv_columns);  //support for old customer
               

        if($this->exclude_line_items){
            return apply_filters('hf_alter_subscription_data', $subscription_data, $csv_columns, $csv_columns,  array('max_line_items' => 0));
        }
      
        $li = 1;
        foreach ($line_items as $line_item) {
            foreach ($line_item as $name => $value) {
                $line_item[$name] = $name . ':' . $value;
            }
            $line_item = implode(apply_filters('wt_change_item_separator', '|'), $line_item);
            $subscription_data["line_item_{$li}"] = $line_item;
            $li++;
        }
                
        $max_line_items = $this->line_items_max_count;
        for ($i = 1; $i <= $max_line_items; $i++) {
            $subscription_data["line_item_{$i}"] = !empty($subscription_data["line_item_{$i}"]) ? self::format_data($subscription_data["line_item_{$i}"]) : '';
        }                       
        if ($this->export_to_separate_columns) {
            $line_item_values = self::get_all_metakeys_and_values($subscription);
            $this->line_item_meta = self::get_all_line_item_metakeys();
            $max_line_items = $this->line_items_max_count; 
            
            for ($i = 1; $i <= $max_line_items; $i++) {
                $line_item_array = explode(apply_filters('wt_change_item_separator', '|'), $subscription_data["line_item_{$i}"]);                 
                foreach ($this->line_item_meta as $meta_val) {
                    $subscription_data["line_item_{$i}_product_id"] = !empty($line_item_array[0]) ? substr($line_item_array[0], strpos($line_item_array[0], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_name"] = !empty($line_item_array[1]) ? substr($line_item_array[1], strpos($line_item_array[1], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_sku"] = !empty($line_item_array[2]) ? substr($line_item_array[2], strpos($line_item_array[2], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_quantity"] = !empty($line_item_array[3]) ? substr($line_item_array[3], strpos($line_item_array[3], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_total"] = !empty($line_item_array[4]) ? substr($line_item_array[4], strpos($line_item_array[4], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_subtotal"] = !empty($line_item_array[5]) ? substr($line_item_array[5], strpos($line_item_array[5], ':') + 1) : '';
                    if (in_array($meta_val, array("_product_id", "_qty", "_variation_id", "_line_total", "_line_subtotal", "_tax_class", "_line_tax", "_line_tax_data", "_line_subtotal_tax"))) {
                        continue;
                    } else {
                        $subscription_data["line_item_{$i}_$meta_val"] = !empty($line_item_values[$i][$meta_val]) ? $line_item_values[$i][$meta_val] : '';
                    }
                }
            }
        }                         

        $order_data_filter_args = array('max_line_items' => $max_line_items);

        if ($this->export_to_separate_rows) {
            $subscription_data = $this->wt_line_item_separate_row_csv_data($subscription, $subscription_data, $order_data_filter_args);
        }  
        
        return apply_filters('hf_alter_subscription_data', $subscription_data, $csv_columns, $csv_columns, $order_data_filter_args);                                
    }
    
    
    public function get_wt_subscriptions_csv_row($subscription) {
        
//        $csv_columns = $this->parent_module->get_selected_column_names();
        $csv_columns = $this->prepare_header();
        
        

        $fee_total = $fee_tax_total = 0;
        $fee_items = $shipping_items = array();

        if (0 != sizeof(array_intersect(array_keys($csv_columns), array('fee_total', 'fee_tax_total', 'fee_items')))) {
            foreach ($subscription->get_fees() as $fee_id => $fee) {
                $fee_items[] = implode('|', array(
                    'name:' . html_entity_decode($fee['name'], ENT_NOQUOTES, 'UTF-8'),
                    'total:' . wc_format_decimal($fee['line_total'], 2),
                    'tax:' . wc_format_decimal($fee['line_tax'], 2),
                    'tax_class:' . $fee['tax_class'],
                ));
				
                $fee_total += (float)$fee['line_total'];
                $fee_tax_total += (float)$fee['line_tax'];
            }
        }
        
        $line_items_shipping = $subscription->get_items('shipping');
        foreach ($line_items_shipping as $item_id => $item) {
            if (is_object($item)) {
                if ($meta_data = $item->get_formatted_meta_data('')) :
                    foreach ($meta_data as $meta_id => $meta) :
                        if (in_array($meta->key, $line_items_shipping)) {
                            continue;
                        }
                        // html entity decode is not working preoperly
                        $shipping_items[] = implode('|', array('item:' . wp_kses_post($meta->display_key), 'value:' . str_replace('&times;', 'X', strip_tags($meta->display_value))));
                    endforeach;
                endif;
            }
        }
        if (!function_exists('get_user_by')) {
            require ABSPATH . 'wp-includes/pluggable.php';
        }
        $version = WC()->version;
        $user_values = get_user_by('ID', ( version_compare( $version, '2.7', '<' ) ) ? $subscription->customer_user : $subscription->get_customer_id());
        
        // Preparing data for export
        foreach ($csv_columns as $header_key => $_) {
            switch ($header_key) {
                case 'subscription_id':
                    $value = $subscription->get_id();
                    break;
                case 'subscription_status':
                    $value = $subscription->get_status();
                    break;
                case 'customer_id':
                    $value = is_object($user_values) ? $user_values->ID : '';
                    break;
                case 'customer_username':
                    $value = is_object($user_values) ? $user_values->user_login : '';
                    break;
                case 'customer_email':
                    $value = is_object($user_values) ? $user_values->user_email : '';
                    break;
                case 'fee_total':
                case 'fee_tax_total':
                    $value = ${$header_key};
                    break;
                case 'order_shipping':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? (empty($subscription->{$header_key}) ? 0 : $subscription->{$header_key}) : $subscription->get_total_shipping();
                    break;
                case 'order_shipping_tax':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? (empty($subscription->{$header_key}) ? 0 : $subscription->{$header_key}) : $subscription->get_shipping_tax();
                    break;
                case 'order_tax':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? (empty($subscription->{$header_key}) ? 0 : $subscription->{$header_key}) : $subscription->get_total_tax();
                    break;
                case 'cart_discount':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? (empty($subscription->{$header_key}) ? 0 : $subscription->{$header_key}) : $subscription->get_total_discount();
                    break;
                case 'cart_discount_tax':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? (empty($subscription->{$header_key}) ? 0 : $subscription->{$header_key}) : $subscription->get_discount_tax();
                    break;
                case 'order_total':
                    $value = empty($subscription->get_total()) ? 0 : $subscription->get_total();
                    break;
                case 'date_created':
                    $value = $subscription->get_date('date_created');
                    break;
                case 'trial_end_date':
                    $value = $subscription->get_date('trial_end_date');
                    break;
                case 'next_payment_date':
                    $value = $subscription->get_date('next_payment_date');
                    break;
                case 'last_order_date_created':
                    $value = $subscription->get_date('last_order_date_created');
                    break;
                case 'end_date':
                    $value = $subscription->get_date('end_date');
                    break;
                case 'order_currency':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? $subscription->{$header_key} :$subscription->get_currency();
                    break;
                case 'billing_period':
                case 'billing_interval':
                case 'payment_method':
                case 'payment_method_title':
                case 'billing_first_name':
                case 'billing_last_name':
                case 'billing_email':
                case 'billing_phone':
                case 'billing_address_1':
                case 'billing_address_2':
                case 'billing_postcode':
                case 'billing_city':
                case 'billing_state':
                case 'billing_country':
                case 'billing_company':
                case 'shipping_first_name':
                case 'shipping_last_name':
                case 'shipping_address_1':
                case 'shipping_address_2':
                case 'shipping_postcode':
                case 'shipping_city':
                case 'shipping_state':
                case 'shipping_country':
                case 'shipping_company':
                case 'shipping_phone':                    
                case 'customer_note':
                    
                    $m_key = "get_$header_key";
                    
                    if(method_exists($subscription, $m_key)){
                        $value = $subscription->{$m_key}();
                    }else{
                        $value = $subscription->{$header_key};
                    }
                    break;
                case 'post_parent':
                        $post = get_post( $subscription->get_id() );
                        $value = $post->post_parent;
                    break;
                case 'order_notes':
					$order_notes = implode('||', (defined('WC_VERSION') && (WC_VERSION >= 3.2)) ? self::get_order_notes_new($subscription) : self::get_order_notes($subscription));
                    /*
					remove_filter('comments_clauses', array('WC_Comments', 'exclude_order_comments'));
                    $notes = get_comments(array('post_id' => $subscription->get_id(), 'approve' => 'approve', 'type' => 'order_note'));
                    add_filter('comments_clauses', array('WC_Comments', 'exclude_order_comments'));
                    $order_notes = array();
                    foreach ($notes as $note) {
                        $order_notes[] = str_replace(array("\r", "\n"), ' ', $note->comment_content);
                    }
                    if (!empty($order_notes)) {
                        $value = implode(';', $order_notes);
                    } else {
                        $value = '';
                    }
					 * 
					 */
					if (!empty($order_notes)) {
                        $value = $order_notes;
                    } else {
                        $value = '';
                    }
                    break;
                case 'renewal_orders':
                    $renewal_orders = $subscription->get_related_orders('ids', 'renewal');
                    if (!empty($renewal_orders)) {
                        $value = implode('|', $renewal_orders);
                    } else {
                        $value = '';
                    }
                    break; 
                    
                case 'order_items':    
                    $line_items = array();
                    $order_items = array();
                    foreach ($subscription->get_items() as $item_id => $item) {
                        $product = $item->get_product();
                        if (!is_object($product)) {
                            $product = new WC_Product(0);
                        }

            //                        $product_id = self::hf_get_canonical_product_id($item);
                        $item_meta = self::get_order_line_item_meta($item_id);
                        $prod_type = ( version_compare( $version, '3.0.0', '<' ) ) ? $product->product_type : $product->get_type();
                        $line_item = array(
                            'product_id' => ( version_compare( $version, '2.7.0', '<' ) ) ? $product->id : (($prod_type == 'variable' || $prod_type == 'variation' || $prod_type == 'subscription_variation') ? $product->get_parent_id() : $product->get_id()),
                            'name' => html_entity_decode($item['name'], ENT_NOQUOTES, 'UTF-8'),
                            'sku' => $product->get_sku(),
                            'quantity' => $item['qty'],
                            'total' => wc_format_decimal($subscription->get_line_total($item), 2),
                            'sub_total' => wc_format_decimal($subscription->get_line_subtotal($item), 2),
                        );

                        // add line item tax
                        $line_tax_data = isset($item['line_tax_data']) ? $item['line_tax_data'] : array();
                        $tax_data = maybe_unserialize($line_tax_data);
                        $tax_detail = isset($tax_data['total']) ? wc_format_decimal(wc_round_tax_total(array_sum((array) $tax_data['total'])), 2) : '';
                        if ($tax_detail != '0.00' && !empty($tax_detail)) {
                            $line_item['tax'] = $tax_detail;
                        }
                        $line_tax_ser = maybe_serialize($line_tax_data);
                        $line_item['tax_data'] = $line_tax_ser;

                        foreach ($item_meta as $key => $value) {
                            switch ($key) {
                                case '_qty':
                                case '_variation_id':
                                case '_product_id':
                                case '_line_total':
                                case '_line_subtotal':
                                case '_tax_class':
                                case '_line_tax':
                                case '_line_tax_data':
                                case '_line_subtotal_tax':
                                    break;

                                default:
                                    if(is_object($value))
                                    $value = $value->meta_value;
                                    if (is_array($value))
                                        $value = implode(',', $value);
                                    $line_item[$key] = $value;
                                    break;
                            }
                        }

                        if ($prod_type === 'variable' || $prod_type === 'variation' || $prod_type === 'subscription_variation') {
                            $line_item['_variation_id'] = ( version_compare( $version, '2.7', '>' ) ) ? $product->get_id() : $product->variation_id;
                        }
                        $order_item_data = array();
                        foreach ($line_item as $name => $value) {
                            $order_item_data[$name] = $name . ':' . $value;
                        }
                        
                        $order_item_data = implode('|', $order_item_data);

                        if ($line_item) {
                            $order_items[] = $order_item_data;
                            $line_items[] = $line_item;
                        }
                    }
                    if (!empty($order_items)) {
                        $value = implode('||', $order_items);
                    }                                        
                    break; 
                case 'coupon_items':
                    $coupon_items = array();
					
                    foreach ($subscription->get_items('coupon') as $_ => $coupon_item) {
						
						$coupon_name = ( version_compare( $version, '4.3.9', '<' ) ) ? $coupon_item['name'] : $coupon_item->get_name();
						$coupon_amount = ( version_compare( $version, '4.3.9', '<' ) ) ? $coupon_item['discount_amount'] : $coupon_item->get_discount();
						
                        $coupon = new WC_Coupon($coupon_name);
                        $coupon_post = get_post((( version_compare( $version, '2.7', '<' ) ) ? $coupon->id : $coupon->get_id() ));
                        $coupon_items[] = implode('|', array(
                            'code:' . $coupon_name,
                            'description:' . ( is_object($coupon_post) ? $coupon_post->post_excerpt : '' ),
                            'amount:' . wc_format_decimal($coupon_amount, 2),
                                )
                        );
                    }

                    if (!empty($coupon_items)) {
                        $value = implode(';', $coupon_items);
                    } else {
                        $value = '';
                    }
                    break;
                case 'download_permissions':
                    $value = ( version_compare( $version, '2.7', '<' ) ) ? ( $subscription->download_permissions_granted ? $subscription->download_permissions_granted : 0 ) :( $subscription->is_download_permitted() );
                    break;
                case 'shipping_method':
                    $shipping_lines = array();
                    foreach ($subscription->get_shipping_methods() as $shipping_item_id => $shipping_item) {
                        $shipping_lines[] = implode('|', array(
                            'method_id:' . $shipping_item['method_id'],
                            'method_title:' . $shipping_item['name'],
                            'total:' . wc_format_decimal($shipping_item['cost'], 2),
                            'total_tax:' . wc_format_decimal( $shipping_item['total_tax'], 2 ),
							'taxes:' . maybe_serialize($shipping_item['taxes']),
                            )
                        );
                    }
                    if (!empty($shipping_lines)) {
                        $value = implode(';', $shipping_lines);
                    } else {
                        $value = '';
                    }
                    break;
                case 'fee_items':
                    $value = implode(';', $fee_items);
                    break;
                 case 'shipping_items':
                    $value = implode(';', $shipping_items);
                    break;
                case 'tax_items':
                    $tax_items = array();
                    foreach ( $subscription->get_taxes() as $tax_code => $tax ) {
						$tax_items[] = implode('|', array(
							'rate_id:' . $tax->get_rate_id(),
							'code:' . $tax->get_rate_code(),
							'total:' . wc_format_decimal($tax->get_tax_total(), 2),
							'label:' . $tax->get_label(),
							'tax_rate_compound:' . $tax->get_compound(),
							'shipping_tax_amount:' . $tax->get_shipping_tax_total(),
							'rate_percent:' . $tax->get_rate_percent(),
						));
                    }
                    if (!empty($tax_items)) {
                        $value = implode(';', $tax_items);
                    } else {
                        $value = '';
                    }
                    break;
                default :
                    if(strstr($header_key, 'meta:')){
                        $meta_key = str_replace('meta:', '', $header_key);
                        //1.3.1 - use proper getters for internal meta keys
                        $value = $this->get_subscription_meta_value($subscription, $meta_key);
                    } else {
                        $value = '';
                    }
            }
            $csv_row[$header_key] = $value;
        }
        
//        $subscription_data = array();
//        foreach ($csv_columns as $header_key => $_) {
//
//            // Strict string comparison, as values like '0' are valid
//            $value = ( '' !== $csv_row[$header_key] ) ? $csv_row[$header_key] : '';
//            $subscription_data[$header_key] = $value;
//        }

        
        $subscription_data = array();
        foreach ($csv_columns as $key => $value) {
            if (!$csv_row || array_key_exists($key, $csv_row)) {
                $subscription_data[$key] = $csv_row[$key];
            } 
        }
        
        if($this->exclude_line_items){
            return apply_filters('hf_alter_subscription_data', $subscription_data, $csv_columns, $csv_columns,  array('max_line_items' => 0));
        }
      
        $li = 1;
        foreach ($line_items as $line_item) {
            foreach ($line_item as $name => $value) {
                $line_item[$name] = $name . ':' . $value;
            }
            $line_item = implode(apply_filters('wt_change_item_separator', '|'), $line_item);
            $subscription_data["line_item_{$li}"] = $line_item;
            $li++;
        }
                
        $max_line_items = $this->line_items_max_count;
        for ($i = 1; $i <= $max_line_items; $i++) {
            $subscription_data["line_item_{$i}"] = !empty($subscription_data["line_item_{$i}"]) ? self::format_data($subscription_data["line_item_{$i}"]) : '';
        }                       
        if ($this->export_to_separate_columns) {
            $line_item_values = self::get_all_metakeys_and_values($subscription);
            $this->line_item_meta = self::get_all_line_item_metakeys();
            $max_line_items = $this->line_items_max_count; 
            
            for ($i = 1; $i <= $max_line_items; $i++) {
                $line_item_array = explode(apply_filters('wt_change_item_separator', '|'), $subscription_data["line_item_{$i}"]);                 
                foreach ($this->line_item_meta as $meta_val) {
                    $subscription_data["line_item_{$i}_product_id"] = !empty($line_item_array[0]) ? substr($line_item_array[0], strpos($line_item_array[0], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_name"] = !empty($line_item_array[1]) ? substr($line_item_array[1], strpos($line_item_array[1], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_sku"] = !empty($line_item_array[2]) ? substr($line_item_array[2], strpos($line_item_array[2], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_quantity"] = !empty($line_item_array[3]) ? substr($line_item_array[3], strpos($line_item_array[3], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_total"] = !empty($line_item_array[4]) ? substr($line_item_array[4], strpos($line_item_array[4], ':') + 1) : '';
                    $subscription_data["line_item_{$i}_subtotal"] = !empty($line_item_array[5]) ? substr($line_item_array[5], strpos($line_item_array[5], ':') + 1) : '';
                    if (in_array($meta_val, array("_product_id", "_qty", "_variation_id", "_line_total", "_line_subtotal", "_tax_class", "_line_tax", "_line_tax_data", "_line_subtotal_tax"))) {
                        continue;
                    } else {
                        $subscription_data["line_item_{$i}_$meta_val"] = !empty($line_item_values[$i][$meta_val]) ? $line_item_values[$i][$meta_val] : '';
                    }
                }
            }
        }                         

        $order_data_filter_args = array('max_line_items' => $max_line_items);

        if ($this->export_to_separate_rows) {
            $subscription_data = $this->wt_line_item_separate_row_csv_data($subscription, $subscription_data, $order_data_filter_args);
        } 
        
//        return apply_filters('hf_alter_subscription_data', $data, $export_columns, $csv_columns);
        return apply_filters('hf_alter_subscription_data', $subscription_data, $csv_columns, $csv_columns);
    }            
    
    public static function hf_sanitize_subscription_status_keys($status_key) {
        if (!is_string($status_key) || empty($status_key)) {
            return '';
        }
        $status_key = ( 'wc-' === substr($status_key, 0, 3) ) ? $status_key : sprintf('wc-%s', $status_key);
        return $status_key;
    }
    
    public static function get_all_line_item_metakeys() {
        $cache_key = 'wt_all_line_item_metakeys';
        $meta_keys = get_transient($cache_key);

        if ($meta_keys === false) {
            global $wpdb;
            $filter_meta = apply_filters('wt_subscription_export_select_line_item_meta', array());

            $query = "SELECT DISTINCT om.meta_key
                      FROM {$wpdb->prefix}woocommerce_order_itemmeta AS om
                      INNER JOIN {$wpdb->prefix}woocommerce_order_items AS oi 
                        ON om.order_item_id = oi.order_item_id
                        AND oi.order_item_type = 'line_item'";

            if (!empty($filter_meta)) {
                $placeholders = implode(',', array_fill(0, count($filter_meta), '%s'));
                $query .= $wpdb->prepare(" AND om.meta_key IN ($placeholders)", $filter_meta);
            }

            $meta_keys = $wpdb->get_col($query);
            set_transient($cache_key, $meta_keys, HOUR_IN_SECONDS);
        }else{
            set_transient($cache_key, $meta_keys, HOUR_IN_SECONDS); // renew.
        }

        return $meta_keys;
    }
    
    public static function get_order_line_item_meta($item_id) {
        global $wpdb;

        $cache_key = 'wt_order_line_item_meta'.$item_id;
        $line_item_meta = wp_cache_get($cache_key);

        if( false !== $line_item_meta ){
            wp_cache_set($cache_key, $line_item_meta, '', HOUR_IN_SECONDS);
            return $line_item_meta;   
        }

        // Ensure item_id is an integer
        $item_id = (int) $item_id;

        // Get filtered meta keys
        $filtered_meta = apply_filters('wt_subscription_export_select_line_item_meta', array());

        $query = "SELECT meta_key, meta_value
                  FROM {$wpdb->prefix}woocommerce_order_itemmeta
                  WHERE order_item_id = %d";

        $params = array($item_id);

        if (!empty($filtered_meta)) {
            $placeholders = implode(',', array_fill(0, count($filtered_meta), '%s'));
            $query .= " AND meta_key IN ($placeholders)";
            $params = array_merge($params, $filtered_meta);
        }

        // Safe prepared query
        $prepared = $wpdb->prepare($query, $params);
        $line_item_meta = $wpdb->get_results($prepared, OBJECT_K);
        wp_cache_set($cache_key, $line_item_meta, '', HOUR_IN_SECONDS);
        return $line_item_meta;
    }
    
    public static function wt_get_subscription_of_coupons($coupons, $subscription_type = 'shop_subscription') {
        global $wpdb;
    
        // Normalize to array
        if ( ! is_array( $coupons ) ) {
            $coupons = [ $coupons ];
        }
    
        if ( empty( $coupons ) ) {
            return [];
        }
    
        // Base table references
        $is_hpos = self::is_hpos_enabled();
        $order_table = $is_hpos ? "{$wpdb->prefix}wc_orders" : $wpdb->posts;
        $order_items_table = "{$wpdb->prefix}woocommerce_order_items";
        $order_id_column = $is_hpos ? 'id' : 'ID';
        $order_type_column = $is_hpos ? 'type' : 'post_type';
        $order_type_value = 'shop_order';
        $parent_column = $is_hpos ? 'parent_order_id' : 'post_parent';
        $subscription_type_column = $is_hpos ? 'type' : 'post_type';
        $subscription_id_column = $is_hpos ? 'id' : 'ID';
    
        // Construct coupon query with placeholders
        $coupon_placeholders = implode( ',', array_fill( 0, count( $coupons ), '%s' ) );
        $order_query = "
            SELECT DISTINCT po.{$order_id_column}
            FROM {$order_table} AS po
            INNER JOIN {$order_items_table} AS oi ON oi.order_id = po.{$order_id_column}
            WHERE po.{$order_type_column} = %s
            AND oi.order_item_type = 'coupon'
            AND oi.order_item_name IN ($coupon_placeholders)
        ";
        $order_query_params = array_merge( [$order_type_value], $coupons );
        $order_ids = $wpdb->get_col( $wpdb->prepare( $order_query, ...$order_query_params ) );
    
        if ( empty( $order_ids ) ) {
            return [];
        }
    
        // Build subscription query using parent order IDs
        $order_ids[] = $subscription_type; // Append post_type/type
        $subs_placeholders = implode( ',', array_fill( 0, count( $order_ids ) - 1, '%s' ) );
    
        $subscription_query = "
            SELECT DISTINCT {$subscription_id_column}
            FROM {$order_table}
            WHERE {$parent_column} IN ($subs_placeholders)
            AND {$subscription_type_column} = %s
        ";
        $subscription_ids = $wpdb->get_col( $wpdb->prepare( $subscription_query, ...$order_ids ) );
    
        return $subscription_ids;
    }
    
    public static function get_order_notes($order) {
        $callback = array('WC_Comments', 'exclude_order_comments');
        $args = array(
            'post_id' => ( version_compare( WC()->version, '2.7.0', '<' ) ) ? $order->id : $order->get_id(),
            'approve' => 'approve',
            'type' => 'order_note'
        );
        remove_filter('comments_clauses', $callback);
        $notes = get_comments($args);
        add_filter('comments_clauses', $callback);
        $notes = array_reverse($notes);
        $order_notes = array();
        foreach ($notes as $note) {
            $date = $note->comment_date;
            $customer_note = 0;
            if (get_comment_meta($note->comment_ID, 'is_customer_note', '1')) {
                $customer_note = 1;
            }
            $order_notes[] = implode('|', array(
                'content:' . str_replace(array("\r", "\n"), ' ', $note->comment_content),
                'date:' . (!empty($date) ? $date : current_time('mysql')),
                'customer:' . $customer_note,
                'added_by:' . $note->added_by
            ));
        }
        return $order_notes;
    }

    public static function get_order_notes_new($order) {
        $notes = wc_get_order_notes(array('order_id' => $order->get_id(), 'order_by' => 'date_created', 'order' => 'ASC'));
        $order_notes = array();
        foreach ($notes as $note) {
            $order_notes[] = implode('|', array(
                'content:' . str_replace(array("\r", "\n"), ' ', $note->content),
                'date:' . $note->date_created->date('Y-m-d H:i:s'),
                'customer:' . $note->customer_note,
                'added_by:' . $note->added_by
            ));
        }
        return $order_notes;
    }
    
    public static function get_all_metakeys_and_values($order = null) {
        $in = 1;
        foreach ($order->get_items() as $item_id => $item) {
            //$item_meta = function_exists('wc_get_order_item_meta') ? wc_get_order_item_meta($item_id, '', false) : $order->get_item_meta($item_id);
            $item_meta = self::get_order_line_item_meta($item_id);
            foreach ($item_meta as $key => $value) {
                switch ($key) {
                    case '_qty':
                    case '_product_id':
                    case '_line_total':
                    case '_line_subtotal':
                    case '_tax_class':
                    case '_line_tax':
                    case '_line_tax_data':
                    case '_line_subtotal_tax':
                        break;

                    default:
                        if (is_object($value))
                            $value = $value->meta_value;
                        if (is_array($value))
                            $value = implode(',', $value);
                        $line_item_value[$key] = $value;
                        break;
                }
            }
            $line_item_values[$in] = !empty($line_item_value) ? $line_item_value : '';
            $in++;
        }
        return $line_item_values;
    }
    
    public static function format_data($data) {
        if (!is_array($data))
            ;
        $data = (string) urldecode($data);
//        $enc = mb_detect_encoding($data, 'UTF-8, ISO-8859-1', true);        
        $use_mb = function_exists('mb_detect_encoding');
        $enc = '';
        if ($use_mb) {
            $enc = mb_detect_encoding($data, 'UTF-8, ISO-8859-1', true);
        }
        $data = ( $enc == 'UTF-8' ) ? $data : utf8_encode($data);

        return $data;
    }
    
    public static function highest_line_item_count($line_item_keys) {
   
        $all_items  = array_count_values(array_column($line_item_keys, 'order_id'));
        return empty($all_items) ? 0 : max($all_items);        
    }
    
    public static function get_max_line_items($order_ids = array()) {
        global $wpdb;

        $cache_key = 'wt_max_line_items_count';
        $line_items_count = get_transient($cache_key);

        if ($line_items_count !== false) {
            set_transient($cache_key, $line_items_count, HOUR_IN_SECONDS); // renew.
            return $line_items_count;
        }

        // Base query: count line items grouped by order_id, then find max
        $query = "SELECT MAX(item_count)
                  FROM (
                      SELECT COUNT(*) as item_count
                      FROM {$wpdb->prefix}woocommerce_order_items
                      WHERE order_item_type = 'line_item'";

        // If specific order IDs are passed, filter them
        if (!empty($order_ids)) {
            $placeholders = implode(',', array_fill(0, count($order_ids), '%d'));
            $query .= $wpdb->prepare(" AND order_id IN ($placeholders)", $order_ids);
        }

        $query .= " GROUP BY order_id
                  ) as counts";

        $line_items_count = (int) $wpdb->get_var($query);

        set_transient($cache_key, $line_items_count, HOUR_IN_SECONDS);

        return $line_items_count;
    }

    public static function is_hpos_enabled() {
		$is_hpos_enabled = false;
        $hpos_data = Wt_Import_Export_For_Woo_Common_Helper::is_hpos_enabled();
	
        if( strpos($hpos_data['table_name'],'wc_orders') !== false ||  $hpos_data['sync']){
            $is_hpos_enabled = true;
        }
        return $is_hpos_enabled;
    }

   



}
