<?php

/**
 * Order section of the plugin
 *
 * @link           
 *
 * @package  Wt_Import_Export_For_Woo 
 */
if (!defined('ABSPATH')) {
    exit;
}

use Automattic\WooCommerce\Utilities\OrderUtil;
class Wt_Import_Export_For_Woo_Subscription {

    public $module_id = '';
    public static $module_id_static = '';
    public $module_base = 'subscription';
    public $module_name = 'Subscription Import Export for WooCommerce';
    public $min_base_version= '1.0.0'; /* Minimum `Import export plugin` required to run this add on plugin */

    private $importer = null;
    private $exporter = null;
    private $all_meta_keys = array();
    private $found_meta = array();
    private $found_hidden_meta = array();
    private $selected_column_names = null;


    public function __construct()
    {      
        /**
        *   Checking the minimum required version of `Import export plugin` plugin available
        */
        if(!Wt_Import_Export_For_Woo_Common_Helper::check_base_version($this->module_base, $this->module_name, $this->min_base_version))
        {
            return;
        }
        if(!function_exists('is_plugin_active'))
        {
            include_once(ABSPATH.'wp-admin/includes/plugin.php');
        }
        if(!is_plugin_active('woocommerce/woocommerce.php'))
        {
            return;
        }

        $this->module_id = Wt_Import_Export_For_Woo::get_module_id($this->module_base);
        self::$module_id_static = $this->module_id;
        
        add_filter('wt_iew_exporter_post_types', array($this, 'wt_iew_exporter_post_types'), 10, 1);
        add_filter('wt_iew_importer_post_types', array($this, 'wt_iew_exporter_post_types'), 10, 1);

        add_filter('wt_iew_exporter_alter_filter_fields', array($this, 'exporter_alter_filter_fields'), 10, 3);
        
        add_filter('wt_iew_exporter_alter_mapping_fields', array($this, 'exporter_alter_mapping_fields'), 10, 3);        
        add_filter('wt_iew_importer_alter_mapping_fields', array($this, 'get_importer_post_columns'), 10, 3);  
        
        add_filter('wt_iew_exporter_alter_advanced_fields', array($this, 'exporter_alter_advanced_fields'), 10, 3);
        add_filter('wt_iew_importer_alter_advanced_fields', array($this, 'importer_alter_advanced_fields'), 10, 3);

        add_filter('wt_iew_exporter_alter_meta_mapping_fields', array($this, 'exporter_alter_meta_mapping_fields'), 10, 3);
        add_filter('wt_iew_importer_alter_meta_mapping_fields', array($this, 'importer_alter_meta_mapping_fields'), 10, 3);

        add_filter('wt_iew_exporter_alter_mapping_enabled_fields', array($this, 'exporter_alter_mapping_enabled_fields'), 10, 3);
        add_filter('wt_iew_importer_alter_mapping_enabled_fields', array($this, 'exporter_alter_mapping_enabled_fields'), 10, 3);

        add_filter('wt_iew_exporter_do_export', array($this, 'exporter_do_export'), 10, 7);
        add_filter('wt_iew_importer_do_import', array($this, 'importer_do_import'), 10, 8);

        add_filter('wt_iew_importer_steps', array($this, 'importer_steps'), 10, 2);
    }

    /**
    *   Altering advanced step description
    */
    public function importer_steps($steps, $base)
    {
        if($this->module_base==$base)
        {
            $steps['advanced']['description']=__('Use options from below to decide updates to existing subscriptions, batch import count or schedule an import. You can also save the template file for future imports.', 'wt-import-export-for-woo');
        }
        return $steps;
    }
    
    public function importer_do_import($import_data, $base, $step, $form_data, $selected_template_data, $method_import, $batch_offset, $is_last_batch) {        
        if ($this->module_base != $base) {
            return $import_data;
        }
        
        if(0 == $batch_offset){                        
            $memory = size_format(wt_let_to_num(ini_get('memory_limit')));
            $wp_memory = size_format(wt_let_to_num(WP_MEMORY_LIMIT));                      
            Wt_Import_Export_For_Woo_Logwriter::write_log($this->module_base, 'import', '---[ New import started at '.date('Y-m-d H:i:s').' ] PHP Memory: ' . $memory . ', WP Memory: ' . $wp_memory);
        }
        
        include plugin_dir_path(__FILE__) . 'import/import.php';
        $import = new Wt_Import_Export_For_Woo_Subscription_Import($this);
        
        $response = $import->prepare_data_to_import($import_data,$form_data, $batch_offset, $is_last_batch);
        
        if($is_last_batch){
            Wt_Import_Export_For_Woo_Logwriter::write_log($this->module_base, 'import', '---[ Import ended at '.date('Y-m-d H:i:s').']---');
        }
        
        return $response;
    }
    public function exporter_do_export($export_data, $base, $step, $form_data, $selected_template_data, $method_export, $batch_offset) {
        if ($this->module_base != $base) {
            return $export_data;
        }

        if (session_id()) {
            session_write_close();
        }

        WC()->session->save_data();
        session_write_close();

        switch ($method_export) {
            case 'quick':
                $this->set_export_columns_for_quick_export($form_data);  
                break;

            case 'template':
            case 'new':
                $this->set_selected_column_names($form_data);
                break;
            
            default:
                break;
        }
        
        include plugin_dir_path(__FILE__) . 'export/export.php';
        $export = new Wt_Import_Export_For_Woo_Subscription_Export($this);
        
        $header_row = $export->prepare_header();

        $data_row = $export->prepare_data_to_export($form_data, $batch_offset);
             
        
        $export_data = array(
            'head_data' => $header_row,
            'body_data' => $data_row['data'],
            'total' => $data_row['total'],
        );
        if(isset($data_row['no_post'])){
            $export_data['no_post'] = $data_row['no_post'];
        }
        return $export_data;
    }
    
    /*
     * Setting default export columns for quick export
     */

    public function set_export_columns_for_quick_export($form_data) {

        $post_columns = self::get_subscription_post_columns();

        $this->selected_column_names = array_combine(array_keys($post_columns), array_keys($post_columns));

        if (isset($form_data['method_export_form_data']['mapping_enabled_fields']) && !empty($form_data['method_export_form_data']['mapping_enabled_fields'])) {
            foreach ($form_data['method_export_form_data']['mapping_enabled_fields'] as $value) {
                $additional_quick_export_fields[$value] = array('fields' => array());
            }

            $export_additional_columns = $this->exporter_alter_meta_mapping_fields($additional_quick_export_fields, $this->module_base, array());
            foreach ($export_additional_columns as $value) {
                $this->selected_column_names = array_merge($this->selected_column_names, $value['fields']);
            }
        }
    }

    
    /**
     * Adding current post type to export list
     *
     */
    public function wt_iew_exporter_post_types($arr) {
        if(class_exists('WC_Subscription') || class_exists('HF_Subscription')){
            $arr['subscription'] = __('Subscription');
        }
        return $arr;
    }
    
    public static function get_payment_gateways() {  
        $payment_gateways = array();
        
        foreach ( WC()->payment_gateways->payment_gateways() as $gateway ) { 
           $payment_gateways[$gateway->id] = $gateway->get_title() ;
        } 
        return $payment_gateways;
    }

    public static function get_subscription_statuses() {                
        $subscription_statuses = array(
            'wc-pending' => 'Pending',
            'wc-active' => 'Active', 
            'wc-on-hold' => 'On hold',
            'wc-cancelled' => 'Cancelled',
            'wc-switched' => 'Switched',
            'wc-expired' => 'Expired',
            'wc-pending-cancel' => 'Pending Cancellation',
        );
        $subscription_statuses =  apply_filters('hf_subscription_statuses', $subscription_statuses);  
        return apply_filters('wt_iew_export_subscription_statuses', $subscription_statuses);
    }
    
    public static function get_subscription_sort_columns() {        
        $sort_columns = array('ID', 'post_parent', 'post_title', 'post_date', 'post_modified', 'post_author', 'menu_order', 'comment_count');
        return apply_filters('wt_iew_export_subscription_sort_columns', array_combine($sort_columns, $sort_columns));
    }

    public static function get_subscription_post_columns() {
        return include plugin_dir_path(__FILE__) . 'data/data-subscription-post-columns.php';
    }
    
    public function get_importer_post_columns($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }
        $colunm = include plugin_dir_path(__FILE__) . 'data/data/data-wf-reserved-fields-pair.php';
//        $colunm = array_map(function($vl){ return array('title'=>$vl, 'description'=>$vl); }, $arr); 
        return $colunm;
    }    
    
    public function exporter_alter_mapping_enabled_fields($mapping_enabled_fields, $base, $form_data_mapping_enabled_fields) {
        if ($base == $this->module_base) {
            $mapping_enabled_fields = array();
            $mapping_enabled_fields['meta'] = array(__('Additional meta', 'wt-import-export-for-woo'), 1);
            $mapping_enabled_fields['hidden_meta'] = array(__('Hidden meta', 'wt-import-export-for-woo'), 0);
             
        }
        return $mapping_enabled_fields;
    }

    
    public function exporter_alter_meta_mapping_fields($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }

        foreach ($fields as $key => $value) {
            switch ($key) {
                
                case 'meta':
                    $meta_attributes = array();
                    $found_meta = $this->wt_get_found_meta();
                    foreach ($found_meta as $meta) {
                        $fields[$key]['fields']['meta:' . $meta] = 'meta:' . $meta;
                    }
                    break;               

                case 'hidden_meta':
                    $found_hidden_meta = $this->wt_get_found_hidden_meta();
                    foreach ($found_hidden_meta as $hidden_meta) {
                        $fields[$key]['fields']['meta:' . $hidden_meta] = 'meta:' . $hidden_meta;
                    }
                    break;
                default:
                    break;
            }
        }

        return $fields;
    }
    
    public function importer_alter_meta_mapping_fields($fields, $base, $step_page_form_data) {
        if ($base != $this->module_base) {
            return $fields;
        }
        $fields=$this->exporter_alter_meta_mapping_fields($fields, $base, $step_page_form_data);
        $out=array();
        foreach ($fields as $key => $value) 
        {
            $value['fields']=array_map(function($vl){ return array('title'=>$vl, 'description'=>$vl); }, $value['fields']);
            $out[$key]=$value;
        }
        return $out;
    }
    
    
    public function wt_get_found_meta() {

        if (!empty($this->found_meta)) {
            return $this->found_meta;
        }

        // Loop products and load meta data
        $found_meta = array();
        // Some of the values may not be usable (e.g. arrays of arrays) but the worse
        // that can happen is we get an empty column.

        $all_meta_keys = $this->wt_get_all_meta_keys();
        $csv_columns = self::get_subscription_post_columns();
        
        foreach ($all_meta_keys as $meta) {

            if (!$meta || (substr((string) $meta, 0, 1) == '_') || in_array($meta, array_keys($csv_columns)) || in_array('meta:' . $meta, array_keys($csv_columns)))
                continue;

            $found_meta[] = $meta;
        }
        
        $found_meta = array_diff($found_meta, array_keys($csv_columns));
        $this->found_meta = $found_meta;
        return $this->found_meta;
    }

    public function wt_get_found_hidden_meta() {

        if (!empty($this->found_hidden_meta)) {
            return $this->found_hidden_meta;
        }

        // Loop products and load meta data
        $found_hidden_meta = array();
        // Some of the values may not be usable (e.g. arrays of arrays) but the worse
        // that can happen is we get an empty column.

        $all_meta_keys = $this->wt_get_all_meta_keys();
        $csv_columns = self::get_subscription_post_columns();

        foreach ($all_meta_keys as $meta) {

            $temp_meta = $meta;
            if (!$meta || (substr((string) $meta, 0, 1) != '_') || in_array($meta, array_keys($csv_columns)) || in_array('meta:' . $meta, array_keys($csv_columns)) || (substr((string) $temp_meta, 0, 1) == '_' &&  in_array( substr((string) $temp_meta, 1), array_keys($csv_columns)) ))
                continue;

            $found_hidden_meta[] = $meta;
        }

        $found_hidden_meta = array_diff($found_hidden_meta, array_keys($csv_columns));

        $this->found_hidden_meta = $found_hidden_meta;
        return $this->found_hidden_meta;
    }


    public function wt_get_all_meta_keys() {

        if (!empty($this->all_meta_keys)) {
            return $this->all_meta_keys;
        }

        $all_meta_keys = self::get_all_metakeys();

        $this->all_meta_keys = $all_meta_keys;
        return $this->all_meta_keys;
    }

        
    public static function get_all_metakeys() {
        global $wpdb;
        
        
        
        if(class_exists('HF_Subscription')){
            $post_type = 'hf_shop_subscription';
        } else {
            $post_type = 'shop_subscription';
        }

        if(self::wt_get_order_table_name()){
            $meta_keys = $wpdb->get_col($wpdb->prepare("SELECT DISTINCT pm.meta_key
                                        FROM {$wpdb->prefix}wc_orders_meta AS pm
                                        LEFT JOIN {$wpdb->prefix}wc_orders AS p ON p.id = pm.order_id
                                        WHERE p.type = %s
                                        AND pm.meta_key NOT IN ('_schedule_next_payment','_schedule_start','_schedule_end','_schedule_trial_end','_download_permissions_granted','_subscription_renewal_order_ids_cache','_subscription_resubscribe_order_ids_cache','_subscription_switch_order_ids_cache','_created_via','_customer_user')
                                        ORDER BY pm.meta_key",$post_type)
                                        );
        }else{
            $meta_keys = $wpdb->get_col($wpdb->prepare("SELECT DISTINCT pm.meta_key
                                        FROM {$wpdb->postmeta} AS pm
                                        LEFT JOIN {$wpdb->posts} AS p ON p.ID = pm.post_id
                                        WHERE p.post_type = %s
                                        AND pm.meta_key NOT IN ('_schedule_next_payment','_schedule_start','_schedule_end','_schedule_trial_end','_download_permissions_granted','_subscription_renewal_order_ids_cache','_subscription_resubscribe_order_ids_cache','_subscription_switch_order_ids_cache','_created_via','_customer_user')
                                        ORDER BY pm.meta_key",$post_type)
                                        );
        }
        
        return $meta_keys;       
    }
    
    
    public function set_selected_column_names($full_form_data) {      
        if (is_null($this->selected_column_names)) {
            if (isset($full_form_data['mapping_form_data']['mapping_selected_fields']) && !empty($full_form_data['mapping_form_data']['mapping_selected_fields'])) {
                $this->selected_column_names = $full_form_data['mapping_form_data']['mapping_selected_fields'];
            }
            if (isset($full_form_data['meta_step_form_data']['mapping_selected_fields']) && !empty($full_form_data['meta_step_form_data']['mapping_selected_fields'])) {
                $export_additional_columns = $full_form_data['meta_step_form_data']['mapping_selected_fields'];
                foreach ($export_additional_columns as $value) {
                    $this->selected_column_names = array_merge($this->selected_column_names, $value);
                }
            }
        }

        return $full_form_data;
    }
    
    public function get_selected_column_names() {
            
        return $this->selected_column_names;
    }    

    public function exporter_alter_mapping_fields($fields, $base, $mapping_form_data) {
        if ($base == $this->module_base) {
            $fields = self::get_subscription_post_columns();
        }
        return $fields;
    }
    
    public function exporter_alter_advanced_fields($fields, $base, $advanced_form_data) {
        if ($this->module_base != $base) {
            return $fields;
        }
        unset($fields['export_shortcode_tohtml']);
        
        $out = array();
        $out['exclude_already_exported'] = array(
            'label' => __("Exclude already exported", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                'Yes' => __('Yes', 'wt-import-export-for-woo'),
                'No' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => 'No',
            'field_name' => 'exclude_already_exported',
            'help_text' => __("Option 'Yes' excludes the previously exported orders.", 'wt-import-export-for-woo'),
        );
        $out['exclude_line_items'] = array(
            'label' => __("Exclude line items", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                'Yes' => __('Yes', 'wt-import-export-for-woo'),
                'No' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => 'No',
            'field_name' => 'exclude_line_items',
            'help_text' => __("Option 'Yes' excludes the line items", 'wt-import-export-for-woo'),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'wt_iew_exclude_line_items',
            )
        );
        $out['export_to_separate'] = array(
            'label' => __("Export line items in", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                'default' => __('Default mode', 'wt-import-export-for-woo'),
                'column' => __('Separate columns', 'wt-import-export-for-woo'),
                'row' => __('Separate rows', 'wt-import-export-for-woo')                
            ),
            'value' => 'default',
            'field_name' => 'export_to_separate',
            //'help_text' => __("Option 'Yes' exports the line items within the order into separate columns in the exported file.", 'wt-import-export-for-woo'),
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('The default option will export each line item details into a single column. This option is mainly used for the order migration purpose.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_export_to_separate', 'value'=>'default')
                    )
                ),
                array(
                    'help_text'=> __('This option will export each line item details into a separate column.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_export_to_separate', 'value'=>'column')
                    )
                ),
                array(
                    'help_text'=> __('This option will export each line item details into a separate row.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_export_to_separate', 'value'=>'row')
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'wt_iew_exclude_line_items',
                'val'=>'No',
                'depth'=>1, /* indicates the left margin of fields */                
            )
        );
        
        foreach ($fields as $fieldk => $fieldv) {
            $out[$fieldk] = $fieldv;
        }
        return $out;
    }

    public function importer_alter_advanced_fields($fields, $base, $advanced_form_data) {
        if ($this->module_base != $base) {
            return $fields;
        }
        unset($fields['export_shortcode_tohtml']);
        $out = array();
        
        $out['skip_new'] = array(
            'label' => __("Update Only", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'skip_new',
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('The store is updated with the data from the input file only for matching/existing records from the file. If the post ID of the subscription being imported exists already(for any of the other post types like coupon, order, user, pages, media etc) skip the subscription from being inserted into the store.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_skip_new', 'value'=>1)
                    )
                ),
                array(
                    'help_text'=> __('The entire data from the input file is processed for an update or insert as the case maybe.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_skip_new', 'value'=>0)
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'wt_iew_skip_new',
            )
        ); 
        
        $out['merge'] = array(
            'label' => __("If the subscription exists in the store", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(                
                '0' => __('Skip', 'wt-import-export-for-woo'),
                '1' => __('Update', 'wt-import-export-for-woo'),
            ),
            'value' => '0',
            'field_name' => 'merge',
            'help_text' => __('Subscriptions are matched by their IDs.', 'wt-import-export-for-woo'),
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('Retains the subscription in the store as is and skips the matching subscription from the input file.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_merge', 'value'=>0)
                    )
                ),
                array(
                    'help_text'=> __('Update subscription as per data from the input file.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_merge', 'value'=>1)
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'parent',
                'target'=>'wt_iew_found_action'
            )
        );         
        
        $out['conflict_with_existing_post'] = array(
            'label' => __("If conflict with an existing Post ID", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(                
                'skip' => __('Skip item', 'wt-import-export-for-woo'),
                'import' => __('Import as new item', 'wt-import-export-for-woo'),
                
            ),
            'value' => 'skip',
            'field_name' => 'id_conflict',
            'help_text' => __('All the items within WooCommerce/WordPress are treated as posts and assigned a unique ID as and when they are created in the store. The post ID uniquely identifies an item irrespective of the post type be it subscription/coupon/product/pages/attachments/revisions etc.', 'wt-import-export-for-woo'),
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('If the post ID of the subscription being imported exists already(for any of the posts like coupon, order, user, pages, media etc) skip the subscription from being inserted into the store.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_id_conflict', 'value'=>'skip')
                    )
                ),
                array(
                    'help_text'=> __('Insert the subscription into the store with a new subscription ID(next available post ID) different from the value in the input file.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_id_conflict', 'value'=>'import')
                    )
                )
            ),
            'form_toggler'=>array(
                'type'=>'child',
                'id'=>'wt_iew_skip_new',
                'val'=>'0',
                'depth'=>0,
            )
        );
                       
        $out['link_wt_import_key'] = array(
            'label' => __("Link related orders using _wt_import_key"),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'link_wt_import_key',
            'help_text' => __('Link underlying orders related to the imported subscriptions by the key _wt_import_key.', 'wt-import-export-for-woo').'<a href="https://www.webtoffee.com/steps-to-import-subscription-order-with-parent-order/" target="_blank">'.__('Read more.').'</a>',
        );
        
        $out['link_product_using_sku'] = array(
            'label' => __("Link products using SKU instead of Product ID", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'link_product_using_sku',
            'help_text_conditional'=>array(
                array(
                    'help_text'=> __('Link the products associated with the imported subscriptions by their SKU.', 'wt-import-export-for-woo'),
                    'condition'=>array(
                        array('field'=>'wt_iew_link_product_using_sku', 'value'=>1)
                    )
                ),
                array(
                    'help_text'=> sprintf(__('Link the products associated with the imported subscriptions by their Product ID. In case of a conflict with %sIDs of other existing post types%s the link cannot be established.', 'wt-import-export-for-woo'),'<b>','</b>'),
                    'condition'=>array(
                        array('field'=>'wt_iew_link_product_using_sku', 'value'=>0)
                    )
                )
            ),
        );
        
        $out['delete_existing'] = array(
            'label' => __("Delete non-matching subscriptions from store", 'wt-import-export-for-woo'),
            'type' => 'radio',
            'radio_fields' => array(
                '1' => __('Yes', 'wt-import-export-for-woo'),
                '0' => __('No', 'wt-import-export-for-woo')
            ),
            'value' => '0',
            'field_name' => 'delete_existing',
            'help_text' => __('Select ‘Yes’ if you need to remove the subscriptions from your store which are not present in the input file. For e.g, if you have a subscription #123 in your store and your import file has subscriptions #234, #345; the subscription #123 is deleted from the store prior to importing #234 and #345.', 'wt-import-export-for-woo'),
        );
        
        
        
        foreach ($fields as $fieldk => $fieldv) {
            $out[$fieldk] = $fieldv;
        }
        return $out;
    }

    /**
     *  Customize the items in filter export page
     */
    public function exporter_alter_filter_fields($fields, $base, $filter_form_data) {
        if ( $this->module_base !== $base ) {
            return $fields;
        }

         // Initialize filter structure
         $fields = $this->initialize_filter_structure($fields);

        // Get saved template data and prepare data
        $saved_template_data = $this->get_saved_template_data($filter_form_data);
        $all_filter_options = $this->get_all_filter_options();
        $selected_elements = $this->extract_selected_elements($saved_template_data);     
        
        if ( isset( $filter_form_data['old_template_data'] ) && is_array($filter_form_data['old_template_data']) ) {
            $fields = $this->create_old_template_rows($fields, $base, $all_filter_options, $selected_elements, $filter_form_data);
        } else {
            // Create filter rows based on data type
            if ( ! empty( $saved_template_data ) ) {
                $fields = $this->create_saved_template_rows($fields, $saved_template_data, $all_filter_options, $selected_elements);
                // Apply saved template values
                $fields = $this->alter_filter_fields_with_saved_template($fields, $saved_template_data);
            } else {
                $fields = $this->create_default_rows($fields, $all_filter_options);
            }
        }

        $fields = $this->add_extra_filter_fields($fields, $base, $filter_form_data);

         // Add available filter options for dropdown
         $fields['multi_field_row']['available_filter_options'] = $all_filter_options;

        return $fields;
    }

    /**
     * Initialize the basic filter structure
     * 
     * @since 1.3.1
     */
    private function initialize_filter_structure($fields) {
        if ( isset( $fields['limit'] ) ) {
            unset( $fields['limit'] );
        }
        if ( isset( $fields['offset'] ) ) {
            unset( $fields['offset'] );
        }
        
        $fields['multi_field_row']['type'] = 'multi_field_row';
        $fields['multi_field_row']['fields'] = array();
        $fields['multi_field_row']['fields'][] = $this->get_filter_column_headers();
        return $fields;
    }
    
    /**
     * Get saved template data from filter form data
     * 
     * @since 1.3.1
     */
    private function get_saved_template_data($filter_form_data) {
        return (isset($filter_form_data['saved_template_data']) && is_array($filter_form_data['saved_template_data'])) ? $filter_form_data['saved_template_data'] : array();
    }
    
    /**
     * Get all available filter options
     * 
     * @since 1.3.1
     */
    private function get_all_filter_options() {
        $default_filter_options = $this->get_default_filter_options();
        $extra_filter_options = $this->get_extra_filter_options();
        
        // Ensure both are arrays before merging
        $default_filter_options = is_array($default_filter_options) ? $default_filter_options : array();
        $extra_filter_options = is_array($extra_filter_options) ? $extra_filter_options : array();
        
        return array_merge($default_filter_options, $extra_filter_options);
    }
    
    /**
     * Extract selected elements from saved template data
     * 
     * @since 1.3.1
     */
    private function extract_selected_elements($saved_template_data) {
        $selected_elements = array();
        if (!empty($saved_template_data)) {
            foreach ($saved_template_data as $template_data) {
                if (!empty($template_data['element'])) {
                    $selected_elements[] = $template_data['element'];
                }
            }
        }
        return $selected_elements;
    }
    
    /**
     * Create saved template rows
     * 
     * @since 1.3.1
     */
    private function create_saved_template_rows($fields, $saved_template_data, $all_filter_options, $selected_elements) {
        $valid_template_count = $this->count_valid_template_entries($saved_template_data);
        $index = 0;
        $current_row = 0;
        
        foreach ($saved_template_data as $template_key => $template_data) {
            $element = isset($template_data['element']) ? $template_data['element'] : '';
            
            if (empty($element)) {
                continue;
            }
            
            $row_fields = $this->create_filter_row_structure($index, $element, $all_filter_options, $current_row === $valid_template_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $element, $all_filter_options, $selected_elements);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
            $current_row++;
        }

        return $fields;
    }

    /**
     * Create old template rows
     * 
     * @since 1.3.1
     */
    private function create_old_template_rows($fields, $base, $all_filter_options, $selected_elements, $filter_form_data) {
        if ($this->module_base != $base) {
            return $fields;
        }

        // Get the old template data from filter_form_data
        $old_template_data = isset($filter_form_data['old_template_data']) ? $filter_form_data['old_template_data'] : array();
        
        if (empty($old_template_data) || !is_array($old_template_data)) {
            return $fields;
        }

        // Convert old template data to new structure
        $converted_template_data = $this->convert_old_template_to_new_structure($old_template_data);

        if ( empty($converted_template_data) ) {
            $fields = $this->create_default_rows($fields, $all_filter_options);
            return $fields;
        }
        
        $valid_template_count = count($converted_template_data);
        $index = 0;
        $current_row = 0;
        
        foreach ($converted_template_data as $template_data) {
            $element = isset($template_data['element']) ? $template_data['element'] : '';
            
            if (empty($element)) {
                continue;
            }
            
            $row_fields = $this->create_filter_row_structure($index, $element, $all_filter_options, $current_row === $valid_template_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $element, $all_filter_options, $selected_elements);
            
            // Set the values from old template data
            $row_fields = $this->set_old_template_values($row_fields, $template_data, $index);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
            $current_row++;
        }

        return $fields;
    }

    /**
     * Convert old template data to new structure
     * 
     * @since 1.3.1
     */
    private function convert_old_template_to_new_structure($old_template_data) {
        
        // Static field mapping for better performance
        static $field_mapping = null;
        
        if (null === $field_mapping) {
            $field_mapping = array(
                'wt_iew_statuses' => array('new_field' => 'subscription_status', 'condition' => 'include'),
                'wt_iew_next_pay_date' => array('new_field' => 'next_payment_date', 'condition' => 'equals'),
                'wt_iew_payment_methods' => array('new_field' => 'payment_method', 'condition' => 'include'),
                'wt_iew_email' => array('new_field' => 'customer', 'condition' => 'include'),
                'wt_iew_products' => array('new_field' => 'item_ordered', 'condition' => 'include'),
            );
        }
        
        $converted_data = array();
        
        // Handle simple field mappings
        foreach ($field_mapping as $old_field => $field_config) {
            if (isset($old_template_data[$old_field]) && !empty($old_template_data[$old_field])) {
                $value = $old_template_data[$old_field];
                $new_field = $field_config['new_field'];
                $condition = $field_config['condition'];

                $converted_data[] = array(
                    'element' => $new_field,
                    'value' => $value,
                    'condition' => $condition,
                    'value_for' => $new_field
                );
            }
        }
        
        // Handle date range specially (start_date / end_date)
        if ( ! empty($old_template_data['wt_iew_start_date'] ) ||
            ! empty($old_template_data['wt_iew_end_date'] ) ) {
            $converted_data[] = array(
                'element' => 'order_date',
                'condition' => 'in_between',
                'value' => array(
                    'from' => $old_template_data['wt_iew_start_date'] ?? '',
                    'to' => $old_template_data['wt_iew_end_date'] ?? ''
                ),
                'value_for' => 'order_date'
            );
        }       
        
        return $converted_data;
    }

    /**
     * Set old template values in the row fields
     * 
     * @since 1.3.1
     */
    private function set_old_template_values($row_fields, $template_data, $index) {
        if (!isset($row_fields['rows']) || !is_array($row_fields['rows'])) {
            return $row_fields;
        }
        
        $element = $template_data['element'];
        $value = $template_data['value'];
        $condition = $template_data['condition'];
        $element_field_name = 'wt_iew_filter_element[' . $index . ']';
        
        foreach ($row_fields['rows'] as $row_index => $row) {
            $field_name = $row['field_name'] ?? '';
            
            // Set element selector value
            if ($field_name === $element_field_name) {
                $row_fields['rows'][$row_index]['value'] = $element;
            }
            
            // Set condition field value
            if (isset($row['condition_field']) && $row['condition_field'] === true) {
                $condition_type = $row['data_value_for'] ?? '';
                
                if ( 'greater-than-less-than' === $condition_type ) {
                    $row_fields['rows'][$row_index]['value'] = $condition;
                }
            }
            
            // Set value field
            if (isset($row['data_value_for']) && $row['data_value_for'] === $element && !empty($value)) {
                $row_fields['rows'][$row_index]['value'] = $value;
            }
        }
        
        return $row_fields;
    }

    /**
     * Create default filter rows
     * 
     * @since 1.3.1
     */
    private function create_default_rows($fields, $all_filter_options) {
        $default_filter_options = $this->get_default_filter_options();
        $default_filter_count = count($default_filter_options);
        $index = 0;
        
        foreach ($default_filter_options as $filter_key => $filter_value) {
            $row_fields = $this->create_filter_row_structure($index, $filter_key, $all_filter_options, $index === $default_filter_count - 1);
            $row_fields['rows'] = $this->build_filter_row_fields($index, $filter_key, $all_filter_options, array(), $default_filter_options);
            
            $fields['multi_field_row']['fields'][] = $row_fields;
            $index++;
        }
        
        return $fields;
    }
    
    /**
     * Count valid template entries
     * 
     * @since 1.3.1
     */
    private function count_valid_template_entries($saved_template_data) {
        $count = 0;
        foreach ($saved_template_data as $template_data) {
            $element = isset($template_data['element']) ? $template_data['element'] : '';
            if (!empty($element)) {
                $count++;
            }
        }
        return $count;
    }
    
    /**
     * Create the basic structure for a filter row
     * 
     * @since 1.3.1
     */
    private function create_filter_row_structure($index, $element, $all_filter_options, $is_last_row) {
        return array(
            'identifier' => 'multi_field_row_' . ($index + 1),
            'add_new_btn' => $is_last_row,
            'row_delete_btn' => true,
            'default_value' => $element,
            'default_condition' => isset($all_filter_options[$element]) ? $all_filter_options[$element]['condition'] : 'equals-not-equals',
            'row_container_attr' => array('style' => 'display:flex;flex-direction:row;gap:40px;'),
            'row_main_attr' => array('style' => 'display:flex;flex-direction:row;gap:4px;align-items:center;padding: 10px 0px;margin-left: 24px;'),
            'rows' => array()
        );
    }
    
    /**
     * Build the fields for a filter row
     * 
     * @since 1.3.1
     */
    private function build_filter_row_fields($index, $element, $all_filter_options, $selected_elements = array(), $default_filter_options = array()) {
        $rows = array();
        
        // Element selector field
        $available_options = $this->get_available_options($element, $all_filter_options, $selected_elements, $index, $default_filter_options);
        $rows[] = array(
            'field_name' => 'wt_iew_filter_element[' . $index . ']',
            'type' => 'select',
            'value' => $element,
            'sele_vals' => $available_options,
            'css_class' => '',
            'attr' => $this->get_common_style_for_element_selector(),
        );
        
        // Add condition fields
        $rows = array_merge($rows, $this->get_condition_fields($index));
        
        // Add value fields
        $value_fields = $this->get_value_fields($index);
        $rows = array_merge($rows, array_values($value_fields));
        
        return $rows;
    }
    
    /**
     * Get available options for element selector
     * 
     * @since 1.3.1
     */
    private function get_available_options($current_element, $all_filter_options, $selected_elements, $index = 0, $default_filter_options = array()) {
        $available_options = array();
        
        if (!empty($selected_elements)) {
            // For saved template data - exclude already selected elements
            foreach ($all_filter_options as $option_key => $option_value) {
                if (!in_array($option_key, $selected_elements) || $option_key === $current_element) {
                    $available_options[$option_key] = $option_value;
                }
            }
        } else {
            // For default rows - all rows show the same available options
            // Exclude only the default elements that are used in other rows
            $default_elements = array_keys($default_filter_options);
            
            foreach ($all_filter_options as $option_key => $option_value) {
                // Include all options except the ones used as defaults in other rows
                if (!in_array($option_key, $default_elements) || $option_key === $current_element) {
                    $available_options[$option_key] = $option_value;
                }
            }
        }
        
        return $available_options;
    }

    /**
     * Apply saved template values to fields
     * 
     * @since 1.3.1
     */
    private function alter_filter_fields_with_saved_template($fields, $saved_template_data) {
        // Early validation
        if (empty($fields) || !is_array($fields) || 
            empty($saved_template_data) || !is_array($saved_template_data)) {
            return $fields;
        }
        
        // Cache filter options to avoid repeated array_merge calls
        $all_filter_options = array_merge($this->get_default_filter_options(), $this->get_extra_filter_options());

        foreach ($saved_template_data as $template_key => $template_data) {
            // Extract data with null coalescing for cleaner code
            $element = $template_data['element'] ?? '';
            $condition = $template_data['condition'] ?? '';
            $value = $template_data['value'] ?? '';

            // Skip empty elements early
            if (empty($element)) {
                continue;
            }

            // Get condition type and skip if not found
            $element_condition_type = $all_filter_options[$element]['condition'] ?? '';
            if (empty($element_condition_type)) {
                continue;
            }

            // Extract row index with validation
            if (!preg_match('/multi_field_row_(\d+)/', $template_key, $matches)) {
                continue;
            }
            $row_index = max(0, intval($matches[1]) - 1);

            // Validate field exists and has rows
            if (!isset($fields['multi_field_row']['fields'][$row_index]['rows']) || 
                !is_array($fields['multi_field_row']['fields'][$row_index]['rows'])) {
                continue;
            }

            $rows = &$fields['multi_field_row']['fields'][$row_index]['rows'];

            // Process each row field
            foreach ($rows as $row_idx => &$row) {
                $this->set_field_value_if_matches($row, $row_idx, $element, $condition, $value, $element_condition_type, $row_index);
            }
        }

        return $fields;
    }

    /**
     * Helper method to set field values if they match the criteria
     * 
     * @since 1.3.1
     */
    private function set_field_value_if_matches(&$row, $row_idx, $element, $condition, $value, $element_condition_type, $row_index) {
        // Set element selector value
        if (isset($row['field_name']) && 
            preg_match('/wt_iew_filter_element\[(\d+)\]/', $row['field_name'], $matches) &&
            intval($matches[1]) === $row_index && 
            isset($row['sele_vals'][$element])) {
            $row['value'] = $element;
        }

        // Set condition field value
        if (isset($row['condition_field']) && true === $row['condition_field'] &&
            isset($row['data_value_for']) && $row['data_value_for'] === $element_condition_type && 
            !empty($condition)) {
            $row['value'] = $condition;
        }

        // Set value field
        if (isset($row['data_value_for']) && $row['data_value_for'] === $element && !empty($value)) {
            $row['value'] = $value;
        }
    }

    /**
     * Get filter column headers
     * 
     * @since 1.3.1
     */
    private function get_filter_column_headers() {
        return array(
            'row_container_attr' => array('style' => 'display:flex;flex-direction:row;gap:40px;'),
            'row_main_attr' => array('style' => 'display:flex;flex-direction:row;gap:4px;align-items:center;padding: 10px 0px;margin-left: 24px;'),
            'type' => 'header_row',
            'rows' => array(
                array(
                    'type' => 'label',
                    'label' => __('Element', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 250px;'),
                ),
                array(
                    'type' => 'label',
                    'label' => __('Condition', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 132px;'),
                ),
                array(
                    'type' => 'label',
                    'label' => __('Value', 'wt-import-export-for-woo'),
                    'attr' => array('style' => 'font-weight: 600; width: 350px;'),
                ),
            ),
        );
    }

    /**
     * Get default filter options
     * 
     * @since 1.3.1
     */
    private function get_default_filter_options() {
        return array(
            'subscriptions' => array('label' => __('Subscription', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Subscription', 'wt-import-export-for-woo')),
            'subscription_status' => array('label' => __('Status', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Status', 'wt-import-export-for-woo')),
            'order_date' => array('label' => __('Order Date', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Order Date', 'wt-import-export-for-woo')),
            'next_payment_date' => array('label' => __('Next Payment Date', 'wt-import-export-for-woo'), 'condition' => 'greater-than-less-than', 'value' => __('Next Payment Date', 'wt-import-export-for-woo')),
            'customer' => array('label' => __('Customer', 'wt-import-export-for-woo'), 'condition' => 'equals-not-equals', 'value' => __('Customer', 'wt-import-export-for-woo')),
        );
    }
    
    /**
     * Get extra filter options
     * 
     * @since 1.3.1
     */
    private function get_extra_filter_options() {
        return array(
            'payment_method' => array('label' => __('Payment Method', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Payment Method', 'wt-import-export-for-woo')),
            'item_ordered' => array('label' => __('Item ordered', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Item ordered', 'wt-import-export-for-woo')),
            'coupons' => array('label' => __('Coupons', 'wt-import-export-for-woo'), 'condition' => 'equals-not-equals', 'value' => __('Coupons', 'wt-import-export-for-woo')),
            'billing_address' => array('label' => __('Billing Address', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Billing Address', 'wt-import-export-for-woo')),
            'shipping_address' => array('label' => __('Shipping Address', 'wt-import-export-for-woo'), 'condition' => 'exc-inc', 'value' => __('Shipping Address', 'wt-import-export-for-woo')),
        );
    }

    /**
     * Get common style for element selector
     * 
     * @since 1.3.1
     */
    private function get_common_style_for_element_selector() {
        return array('style' => 'width: 250px; height: 40px; border-radius: 6px;');
    }

    /**
     * Get common style for value fields
     * 
     * @since 1.3.1
     */
    private function get_common_style_for_value_fields() {
        return array('style' => 'width: 350px; height: 40px; border-radius: 6px;');
    }

    /**
     * Get condition fields
     * 
     * @since 1.3.1
     */
    private function get_condition_fields($index) {
        $style = array('style' => 'width: 132px; height: 40px; border-radius: 6px;');
        return array(
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'include' => __('Include', 'wt-import-export-for-woo'),
                    'exclude' => __('Exclude', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'exc-inc',
                'visible_if' => 'exc-inc',
                'condition_field' => true,
            ),
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'equals' => __('Equals', 'wt-import-export-for-woo'),
                    'not_equals' => __('Not equals', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'equals-not-equals',
                'visible_if' => 'equals-not-equals',
                'condition_field' => true,
            ),
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'contains' => __('Contains', 'wt-import-export-for-woo'),
                    'not_contains' => __('Not contains', 'wt-import-export-for-woo'),
                    'starts_with' => __('Starts with', 'wt-import-export-for-woo'),
                    'ends_with' => __('Ends with', 'wt-import-export-for-woo'),
                    'equals' => __('Equals', 'wt-import-export-for-woo'),
                    'not_equals' => __('Not equals', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'contains-not-contains',
                'visible_if' => 'contains-not-contains',
                'condition_field' => true,
            ),
            array(
                'field_name' => "wt_iew_filter_condition[$index]",
                'type' => 'select',
                'value' => '',
                'sele_vals' => array(
                    'greater_than' => __('Greater than', 'wt-import-export-for-woo'),
                    'less_than' => __('Less than', 'wt-import-export-for-woo'),
                    'equals' => __('Equals', 'wt-import-export-for-woo'),
                    'not_equals' => __('Not equals', 'wt-import-export-for-woo'),
                    'in_between' => __('In between', 'wt-import-export-for-woo'),
                ),
                'attr' => $style,
                'data_value_for' => 'greater-than-less-than',
                'visible_if' => 'greater-than-less-than',
                'condition_field' => true,
            ),
        );
    }

    /**
     * Get value fields
     * 
     * @since 1.3.1
     */
    private function get_value_fields($index) {
        $style = $this->get_common_style_for_value_fields();
    
        return array(
            'subscriptions' => array(
                'field_name' => "subscriptions[$index]",
                'data_value_for' => 'subscriptions',
                'type' => 'text',
                'placeholder' => __('Enter subscription IDs/numbers', 'wt-import-export-for-woo'),
                'attr' => $style,
                'visible_if' => 'subscriptions',
            ),
            'subscription_status' => array(
                'field_name' => "subscription_status[$index]",
                'data_value_for' => 'subscription_status',
                'type' => 'multi_select',
                'placeholder' => __('All statuses', 'wt-import-export-for-woo'),
                'sele_vals' => self::get_subscription_statuses(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'subscription_status',
            ),
            'order_date' => array(
                'field_name' => "order_date[$index]",
                'data_value_for' => 'order_date',
                'type' => 'custom_date_range',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'order_date',
                'index' => $index,
            ),
            'next_payment_date' => array(
                'field_name' => "next_payment_date[$index]",
                'data_value_for' => 'next_payment_date',
                'type' => 'custom_date_range',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'next_payment_date',
                'index' => $index,
            ),
            'customer' => array(
                'field_name' => "customer[$index]",
                'data_value_for' => 'customer',
                'type' => 'multi_select',
                'placeholder' => __('Search for customers', 'wt-import-export-for-woo'),
                'sele_vals' => array(),
                'css_class' => 'wc-customer-search',
                'attr' => $style,
                'visible_if' => 'customer',
            ),
            'payment_method' => array(
                'field_name' => "payment_method[$index]",
                'data_value_for' => 'payment_method',
                'type' => 'multi_select',
                'placeholder' => __('All payment methods', 'wt-import-export-for-woo'),
                'sele_vals' => self::get_payment_gateways(),
                'css_class' => 'wc-enhanced-select',
                'attr' => $style,
                'visible_if' => 'payment_method',
            ),
            'item_ordered' => array(
                'field_name' => "item_ordered[$index]",
                'data_value_for' => 'item_ordered',
                'type' => 'multi_select',
                'placeholder' => __('Search for products', 'wt-import-export-for-woo'),
                'sele_vals' => array(),
                'css_class' => 'wc-product-search',
                'attr' => $style,
                'visible_if' => 'item_ordered',
            ),
            'coupons' => array(
                'field_name' => "coupons[$index]",
                'data_value_for' => 'coupons',
                'type' => 'multi_select',
                'placeholder' => __('Search for coupons', 'wt-import-export-for-woo'),
                'sele_vals' => array(),
                'css_class' => 'wt-coupon-search',
                'attr' => $style,
                'visible_if' => 'coupons',
            ),
            'billing_address' => array(
                'field_name' => "billing_address[$index]",
                'data_value_for' => 'billing_address',
                'type' => 'custom_address_filter',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'billing_address',
                'index' => $index,
                'address_type' => 'billing',
            ),
            'shipping_address' => array(
                'field_name' => "shipping_address[$index]",
                'data_value_for' => 'shipping_address',
                'type' => 'custom_address_filter',
                'placeholder' => '',
                'attr' => array('style' => 'width: auto; display: flex; gap: 10px; align-items: center;'),
                'visible_if' => 'shipping_address',
                'index' => $index,
                'address_type' => 'shipping',
            ),
        );
    }

    /**
     * Add extra filter fields (limit, offset, sort, etc.)
     * 
     * @since 1.3.1
     */
    public function add_extra_filter_fields($fields, $base, $filter_form_data) {
        if (!apply_filters('wt_iew_exporter_add_extra_filter_fields', false, $base, $filter_form_data)) {
            return $fields;
        }
        
        $fields['limit'] = array(
            'label' => __('Total number of subscriptions to export', 'wt-import-export-for-woo'),
            'value' => '',
            'type' => 'number',
            'attr' => array('step' => 1, 'min' => 0),
            'field_name' => 'limit',
            'placeholder' => __('Unlimited', 'wt-import-export-for-woo'),
            'help_text' => __('Exports specified number of subscriptions. e.g. Entering 500 with a skip count of 10 will export subscriptions from 11th to 510th position.', 'wt-import-export-for-woo'),
        );
        
        $fields['offset'] = array(
            'label' => __('Skip first <i>n</i> subscriptions', 'wt-import-export-for-woo'),
            'value' => '',
            'field_name' => 'offset',
            'placeholder' => '0',
            'help_text' => __('Skips specified number of subscriptions from the beginning. e.g. Enter 10 to skip first 10 subscriptions from export.', 'wt-import-export-for-woo'),
            'type' => 'number',
            'attr' => array('step' => 1, 'min' => 0),
            'validation_rule' => array('type' => 'absint')
        );
        
        $sort_columns = self::get_subscription_sort_columns();
        $fields['sort_columns'] = array(
            'label' => __('Sort Columns', 'wt-import-export-for-woo'),
            'placeholder' => __('ID'),
            'field_name' => 'sort_columns',
            'sele_vals' => $sort_columns,
            'help_text' => __('Sort the exported data based on the selected columns in order specified. Defaulted to ID.', 'wt-import-export-for-woo'),
            'type' => 'select',
            'validation_rule' => array('type' => 'text_arr')
        );
        
        $fields['order_by'] = array(
            'label' => __('Sort By', 'wt-import-export-for-woo'),
            'placeholder' => __('ASC'),
            'field_name' => 'order_by',
            'sele_vals' => array('ASC' => __('Ascending', 'wt-import-export-for-woo'), 'DESC' => __('Descending', 'wt-import-export-for-woo')),
            'help_text' => __('Defaulted to Ascending. Applicable to above selected columns in the order specified.', 'wt-import-export-for-woo'),
            'type' => 'select',
        );
        
        return $fields;
    }
   
    public function get_item_by_id($id) {
        $post['edit_url']=get_edit_post_link($id);
        $post['title'] = $id;
        return $post; 
    }
    public static function get_item_link_by_id($id) {
        if(self::wt_get_order_table_name()){
            if ( class_exists( 'HF_Subscription' ) ) {
                $post['edit_url'] = admin_url( "admin.php?page=wc-orders--hf_shop_subscription&action=edit&id={$id}" );
            } else {
                $post['edit_url'] = admin_url( "admin.php?page=wc-orders--shop_subscription&action=edit&id={$id}" );
            }
        } else {
            $post['edit_url'] = get_edit_post_link( $id );
        }
        $post['title']    = $id;
        return $post;
    }
     /**
	  * Is HPOS table enabled.
	  *
	  * @return bool
	  */
      public static function wt_get_order_table_name() {
		if(get_option('woocommerce_custom_orders_table_enabled') === 'yes'){
            return true;
		}else{
            return false;
        }
	}	

}

new Wt_Import_Export_For_Woo_Subscription();
